# AUTOGENERATED! DO NOT EDIT! File to edit: 00_core.ipynb (unless otherwise specified).

__all__ = ['sparse_mask', 'sparse_mask_like', 'sparse_params', 'apply_masks', 'is_sparseable_module',
           'sparseable_modules', 'uniform_sparsity', 'first_layer_dense_uniform', 'erdos_renyi_sparsity',
           'sparsify_model', 'weight_magnitude', 'gradient_magnitude', 'gradient_momentum', 'top_k_mask',
           'DynamicSparseTrainingCallback', 'SET_kwargs', 'STFS_kwargs', 'RigL_kwargs']

# Cell
import numpy as np
import torch
import torch.nn as nn

# Cell
from fastcore.all import * # L, etc...
from fastai.basics import * # flatten_model, etc...
from fastai.callback.all import * # combine_scheds

# Cell
@torch.no_grad()
def sparse_mask(sizes, sparsity):
    n_total = np.prod(sizes)
    n_ones = round((1-sparsity) * n_total)
    shuffled_ones = torch.randperm(n_total)[:n_ones]
    mask = torch.zeros(n_total, dtype=torch.bool)
    mask[shuffled_ones] = True
    return mask.reshape(*sizes)

def sparse_mask_like(param, sparsity): return sparse_mask(param.shape, sparsity).to(param.device)

# Cell
def sparse_params(module):
    '''Returns list of all (param, mask, sparsity) tuples in a module.'''
    buffer_d = {name:b for name, b in module.named_buffers()}
    param_mask_sparsities = [(p, buffer_d[f'{name}_mask'], buffer_d.get(f'{name}_sparsity'))
                             for name, p in module.named_parameters()
                             if f'{name}_mask' in buffer_d]
    return list(set(param_mask_sparsities))

# Cell
@torch.no_grad()
def apply_masks(module, *args, inplace=True):
    for param, mask, sparsity in sparse_params(module):
        if inplace: param.data.mul_(mask)
        else:       param.data = param.data.mul(mask)

# Cell
_sparseable_module_types = (nn.Linear,
                            nn.Conv1d, nn.Conv2d, nn.Conv3d,
                            nn.ConvTranspose1d, nn.ConvTranspose2d, nn.ConvTranspose3d,
                            nn.MultiheadAttention,
                            nn.RNN, nn.RNNCell, nn.GRU, nn.GRUCell, nn.LSTM, nn.LSTMCell)

def is_sparseable_module(m, additional_types=[]):
    types = set(_sparseable_module_types) | set(additional_types)
    return isinstance(m, tuple(types))

# Cell

# TODO: flatten_model gets rid of nn.MultiheadAttention which has it's own parameter 'in_proj_weight'
#       which means sparsity_model doesn't sparsify this parameter
def sparseable_modules(model, additional_types=[]):
    filt = partial(is_sparseable_module, additional_types=additional_types)
    return L(flatten_model(model)).filter(filt)

# Cell
def uniform_sparsity(params, model_sparsity):
    return [model_sparsity] * len(params)

# Cell
def first_layer_dense_uniform(params, model_sparsity):
    sparsities = [0.] + [model_sparsity] * (len(params) - 1)
    return sparsities

# Cell
# modified from https://github.com/google-research/rigl/blob/master/rigl/sparse_utils.py.
def erdos_renyi_sparsity(params, model_sparsity, include_kernel=True, erk_power_scale=1.0):
    """
    Returns a list of sparsities in the same order as params. Sparsities satisfy
    the Erdos-Renyi(Kernel) distribution, where the model has a total parameter count
    as one with uniform sparsities, that is, satisfying the following equation:
    # eps * (p_1 * N_1 + p_2 * N_2) = (1 - model_sparsity) * (N_1 + N_2), for some float `eps`.

    Args:
    params: list of all sparseable parameters
    model_sparsity: target overall sparsity between 0 and 1
    include_kernel: if True, kernel dimensions are included in the scaling (e.g. for ConvNd layers)
    erk_power_scale: scale < 1 softens the erdos_renyi distribution (i.e. closer to uniform)

    Returns a list of sparsities where values correspond to individual param sparsities.
    """
    # Enforce custom sparsities, then find correct scaling factor, `eps` for remaining params
    dense_layers = set()
    is_eps_valid = False
    while not is_eps_valid:
        # Start with all layers and try to find right eps. If any sparsity exceeds 1,
        # make that layer dense and repeat with the non-dense layers.
        #
        # E.g. where N_3, and N_4 are found to be dense:
        # eps * (p_1 * N_1 + p_2 * N_2) + (N_3 + N_4) =
        #    (1 - model_sparsity) * (N_1 + N_2 + N_3 + N_4)
        # eps * (p_1 * N_1 + p_2 * N_2) =
        #    (1 - model_sparsity) * (N_1 + N_2) - model_sparsity * (N_3 + N_4) <--- == rhs
        # eps = rhs / (\sum_i p_i * N_i) <--- == divisor
        # eps = rhs / divisor

        divisor = 0
        rhs = 0
        raw_sparsity = {}
        for p in params:
            n_zeros = int(np.floor(model_sparsity * p.numel()))
            if p in dense_layers:
                rhs -= n_zeros
            else:
                n_ones = p.numel() - n_zeros
                rhs += n_ones
                if include_kernel:
                    raw_sparsity[p] = (np.sum(p.shape) / np.prod(p.shape))**erk_power_scale
                else:
                    raw_sparsity[p] = (np.sum(p.shape[:2]) / np.prod(p.shape[:2]))
                divisor += raw_sparsity[p] * p.numel()

        eps = rhs / divisor

        # If eps * raw_sparsity[p] > 1, we add the param to the set of dense_layers
        max_sparsity = np.max(list(raw_sparsity.values()))
        if eps * max_sparsity > 1:
            for p, p_raw_sparsity in raw_sparsity.items():
                if p_raw_sparsity == max_sparsity:
                    dense_layers.add(p)
        else:
            is_eps_valid = True

    # With the valid eps, we can set sparsities of the remaining layers
    sparsities = [0. if p in dense_layers else (1. - eps * raw_sparsity[p]) for p in params]
    return sparsities

# Cell
@torch.no_grad()
def sparsify_model(model, model_sparsity, sparse_init_f=uniform_sparsity, enforce_mask=True):
    '''
    Adds a sparse mask for each sparseable-module weight in model and applies mask to weights.

    If `enforce_mask` is True, a forward_pre_hook will be registered to each module
    to apply the weight mask before every forward pass of the module.

    `sparsify_method`: per RigL paper, `uniform_sparsity` has fewer FLOPs, `erdos_renyi_sparsity`
    results in better model.

    Returns hooks if `enforce_mask` == True, otherwise None.
    '''
    modules = sparseable_modules(model)
    module_name_param = L([(m, p_name, p) for m in modules for p_name, p in m.named_parameters()
                         if 'weight' in p_name])
    params = module_name_param.itemgot(2)
    sparsities = sparse_init_f(params, model_sparsity)

    hooks = []
    for (m, p_name, p), s in zip(module_name_param, sparsities):
        if s > 0:
            mask = sparse_mask_like(m.weight, s)
            m.register_buffer('weight_mask', mask)
            m.register_buffer('weight_sparsity', tensor(s))
            apply_masks(m)
            if enforce_mask:
                h = m.register_forward_pre_hook(apply_masks)
                hooks.append(h)
    return hooks or None

# Cell
def weight_magnitude(p, **kwargs): return p.data.abs()

# Cell
def gradient_magnitude(p, **kwargs): return p.grad.abs()

# Cell
def gradient_momentum(p, opt, **kwargs):
    '''Calculates the momentum of the gradient for a parameter `p` from the `opt` state.'''
    state = opt.state[p]
    grad_avg = state['grad_avg'] if 'grad_avg' in state else None
    sqr_avg = state['sqr_avg'] if 'sqr_avg' in state else None
    if grad_avg is None:
        raise Exception(f"Error: 'grad_avg' key not found in optimizer state. Tip: set the `mom` hyperparamter in the learner.")
    if sqr_avg is None:
        grad_mom = grad_avg
    else:
        try: eps = opt.state_dict()['hypers'][0]['eps']
        except: eps = 1e-6
        grad_mom =  grad_avg / (torch.sqrt(sqr_avg + eps))
    return grad_mom

# Cell
def top_k_mask(t, n_keep):
    '''Returns a mask with `n_keep` ones cooresponding to the largest values in `t`'''
    n_drop = t.numel() - n_keep
    _, sorted_ixs = torch.topk(t.flatten(), k=t.numel())
    mask = torch.cat([torch.ones(n_keep, dtype=torch.bool, device=t.device),
                      torch.zeros(n_drop, dtype=torch.bool, device=t.device)])
    mask = mask.scatter(0, sorted_ixs, mask)
    return mask.view(*t.shape)

# Cell
class DynamicSparseTrainingCallback(Callback):
    toward_end = True # run after GradientAccumulation and other cbs that modify the gradients

    def __init__(self, sparse_modules=None,
                 batches_per_update=None, initial_drop_grow_pct=0.3, stop_pct=0.75,
                 keep_score_f=weight_magnitude, grow_score_f=gradient_magnitude):
        '''
        Args:
        module_sparsity_map: dictionary mapping modules to sparsity values
        batches_per_update: # of batches per update, None (default) updates at end of each training epoch
        initial_drop_grow_pct: percentage of weights to change during each dynamic weight update
        stop_pct: stop dynamic weight updates after `stop_pct` of training
        keep_score_f: function scoring each weight, top n are kept and the rest are zeroed
        grow_score_f: function scoring each weight, top n excl. kept weights are unmasked and initialized to zero
        '''
        store_attr('initial_drop_grow_pct,stop_pct,keep_score_f,grow_score_f,batches_per_update')
        self.modules = sparse_modules

    def before_fit(self):
        self.modules = ifnone(self.modules, sparseable_modules(self.learn.model))
        self.batches_per_update = ifnone(self.batches_per_update, len(self.dls.train))
        self.drop_grow_pct_sched = combine_scheds(
            [self.stop_pct, 1-self.stop_pct],
            [SchedCos(self.initial_drop_grow_pct, 0.), SchedNo(0.,0.)]
        )

    def after_backward(self):
        self.step()
        if self.is_update_step:
            for m in self.modules:
                self.rewire_module(m)
            raise CancelBatchException()

    def step(self):
        if not self.training:
            self.is_update_step = False
        else:
            step = self.epoch * self.n_iter + self.iter
            n_steps = self.n_epoch * self.n_iter
            pct_train = step / n_steps
            self.drop_grow_pct = self.drop_grow_pct_sched(pct_train)
            self.is_update_step = step > 0 and step % self.batches_per_update == 0 and self.drop_grow_pct > 0

    @torch.no_grad()
    def rewire_module(self, m):
        for param, param_name, sparsity in sparse_params(m):
            if sparsity <= 0: continue

            param, mask = m.weight, m.weight_mask

            n_grow = int((1 - sparsity) * param.numel() * self.drop_grow_pct)
            n_keep = int((1 - sparsity) * param.numel() * (1 - self.drop_grow_pct))

            gt0, n = m.weight.abs().gt(0).sum(), m.weight.numel()
            # determine which weights to keep
            keep_score = self.keep_score_f(param, opt=self.learn.opt)
            keep_mask = top_k_mask(keep_score, n_keep)

            # determine which weights to grow
            grow_score = self.grow_score_f(param, opt=self.learn.opt)
            # make all keep weights to negative so we don't choose to grow them
            grow_score = grow_score * keep_mask.logical_not() - keep_mask.float()
            grow_mask = top_k_mask(grow_score, n_grow)

            # update network connectivity
            mask.data = keep_mask | grow_mask

            # zero momentum for new connections
            self.reset_momentum(param, grow_mask & keep_mask.logical_not())

    @torch.no_grad()
    def reset_momentum(self, p, mask):
        state = self.opt.state[p]
        if 'grad_avg' in state: state['grad_avg'].mul_(mask)
        if 'sqr_avg' in state: state['sqr_avg'].mul_(mask)

    _docs = dict(before_fit="Schedule the number of connections to drop & grow per update",
                 before_batch="Add dynamic update hooks",
                 after_backward="Remove dynamic update hooks and cancel batch",
                 step="Update self.is_update_step and self.drop_grow_pct",
                 rewire_module="Update step for one module",
                 reset_momentum="Initialize momentum to zero for newly-added connections")

# Cell
SET_kwargs = {}

# Cell
STFS_kwargs = {'keep_method': weight_magnitude, 'grow_method': gradient_momentum,
               'batches_per_update': None, 'initial_drop_grow_pct': 0.3, 'stop_pct': 1.0,}

# Cell
RigL_kwargs = {'keep_method': weight_magnitude, 'grow_method': gradient_magnitude,
               'batches_per_update':None, 'initial_drop_grow_pct':0.3, 'stop_pct':0.75,}