"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cluster = exports.ClusterType = exports.NodeType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("aws-cdk-lib/aws-ec2");
const secretsmanager = require("aws-cdk-lib/aws-secretsmanager");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const aws_redshift_1 = require("aws-cdk-lib/aws-redshift");
const subnet_group_1 = require("./subnet-group");
/**
 * Possible Node Types to use in the cluster
 * used for defining {@link ClusterProps.nodeType}.
 */
var NodeType;
(function (NodeType) {
    /**
     * ds2.xlarge
     */
    NodeType["DS2_XLARGE"] = "ds2.xlarge";
    /**
     * ds2.8xlarge
     */
    NodeType["DS2_8XLARGE"] = "ds2.8xlarge";
    /**
     * dc1.large
     */
    NodeType["DC1_LARGE"] = "dc1.large";
    /**
     * dc1.8xlarge
     */
    NodeType["DC1_8XLARGE"] = "dc1.8xlarge";
    /**
     * dc2.large
     */
    NodeType["DC2_LARGE"] = "dc2.large";
    /**
     * dc2.8xlarge
     */
    NodeType["DC2_8XLARGE"] = "dc2.8xlarge";
    /**
     * ra3.xlplus
     */
    NodeType["RA3_XLPLUS"] = "ra3.xlplus";
    /**
     * ra3.4xlarge
     */
    NodeType["RA3_4XLARGE"] = "ra3.4xlarge";
    /**
     * ra3.16xlarge
     */
    NodeType["RA3_16XLARGE"] = "ra3.16xlarge";
})(NodeType = exports.NodeType || (exports.NodeType = {}));
/**
 * What cluster type to use.
 * Used by {@link ClusterProps.clusterType}
 */
var ClusterType;
(function (ClusterType) {
    /**
     * single-node cluster, the {@link ClusterProps.numberOfNodes} parameter is not required
     */
    ClusterType["SINGLE_NODE"] = "single-node";
    /**
     * multi-node cluster, set the amount of nodes using {@link ClusterProps.numberOfNodes} parameter
     */
    ClusterType["MULTI_NODE"] = "multi-node";
})(ClusterType = exports.ClusterType || (exports.ClusterType = {}));
/**
 * A new or imported clustered database.
 */
class ClusterBase extends aws_cdk_lib_1.Resource {
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterName,
            targetType: secretsmanager.AttachmentTargetType.REDSHIFT_CLUSTER,
        };
    }
}
/**
 * Create a Redshift cluster a given number of nodes.
 *
 * @resource AWS::Redshift::Cluster
 */
class Cluster extends ClusterBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_ClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Cluster);
            }
            throw error;
        }
        this.vpc = props.vpc;
        this.vpcSubnets = props.vpcSubnets ?? {
            subnetType: ec2.SubnetType.PRIVATE_WITH_NAT,
        };
        const removalPolicy = props.removalPolicy ?? aws_cdk_lib_1.RemovalPolicy.RETAIN;
        const subnetGroup = props.subnetGroup ?? new subnet_group_1.ClusterSubnetGroup(this, 'Subnets', {
            description: `Subnets for ${id} Redshift cluster`,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            removalPolicy: removalPolicy,
        });
        const securityGroups = props.securityGroups ?? [new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'Redshift security group',
                vpc: this.vpc,
            })];
        const securityGroupIds = securityGroups.map(sg => sg.securityGroupId);
        let secret;
        if (!props.masterUser.masterPassword) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUser.masterUsername,
                encryptionKey: props.masterUser.encryptionKey,
            });
        }
        const clusterType = props.clusterType || ClusterType.MULTI_NODE;
        const nodeCount = this.validateNodeCount(clusterType, props.numberOfNodes);
        if (props.encrypted === false && props.encryptionKey !== undefined) {
            throw new Error('Cannot set property encryptionKey without enabling encryption!');
        }
        this.singleUserRotationApplication = secretsmanager.SecretRotationApplication.REDSHIFT_ROTATION_SINGLE_USER;
        this.multiUserRotationApplication = secretsmanager.SecretRotationApplication.REDSHIFT_ROTATION_MULTI_USER;
        let loggingProperties;
        if (props.loggingBucket) {
            loggingProperties = {
                bucketName: props.loggingBucket.bucketName,
                s3KeyPrefix: props.loggingKeyPrefix,
            };
        }
        const cluster = new aws_redshift_1.CfnCluster(this, 'Resource', {
            // Basic
            allowVersionUpgrade: true,
            automatedSnapshotRetentionPeriod: 1,
            clusterType,
            clusterIdentifier: props.clusterName,
            clusterSubnetGroupName: subnetGroup.clusterSubnetGroupName,
            vpcSecurityGroupIds: securityGroupIds,
            port: props.port,
            clusterParameterGroupName: props.parameterGroup && props.parameterGroup.clusterParameterGroupName,
            // Admin (unsafeUnwrap here is safe)
            masterUsername: secret?.secretValueFromJson('username').unsafeUnwrap() ?? props.masterUser.masterUsername,
            masterUserPassword: secret?.secretValueFromJson('password').unsafeUnwrap()
                ?? props.masterUser.masterPassword?.unsafeUnwrap()
                ?? 'default',
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            nodeType: props.nodeType || NodeType.DC2_LARGE,
            numberOfNodes: nodeCount,
            loggingProperties,
            iamRoles: props?.roles?.map(role => role.roleArn),
            dbName: props.defaultDatabaseName || 'default_db',
            publiclyAccessible: props.publiclyAccessible || false,
            // Encryption
            kmsKeyId: props.encryptionKey?.keyId,
            encrypted: props.encrypted ?? true,
            classic: props.classicResizing,
            elasticIp: props.elasticIp,
        });
        cluster.applyRemovalPolicy(removalPolicy, {
            applyToUpdateReplacePolicy: true,
        });
        this.clusterName = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = aws_cdk_lib_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        if (secret) {
            this.secret = secret.attach(this);
        }
        const defaultPort = ec2.Port.tcp(this.clusterEndpoint.port);
        this.connections = new ec2.Connections({ securityGroups, defaultPort });
    }
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromClusterAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_ClusterAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromClusterAttributes);
            }
            throw error;
        }
        class Import extends ClusterBase {
            constructor() {
                super(...arguments);
                this.connections = new ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: ec2.Port.tcp(attrs.clusterEndpointPort),
                });
                this.clusterName = attrs.clusterName;
                this.instanceIdentifiers = [];
                this.clusterEndpoint = new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.clusterEndpointPort);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_RotationMultiUserOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addRotationMultiUser);
            }
            throw error;
        }
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    validateNodeCount(clusterType, numberOfNodes) {
        if (clusterType === ClusterType.SINGLE_NODE) {
            // This property must not be set for single-node clusters; be generous and treat a value of 1 node as undefined.
            if (numberOfNodes !== undefined && numberOfNodes !== 1) {
                throw new Error('Number of nodes must be not be supplied or be 1 for cluster type single-node');
            }
            return undefined;
        }
        else {
            if (aws_cdk_lib_1.Token.isUnresolved(numberOfNodes)) {
                return numberOfNodes;
            }
            const nodeCount = numberOfNodes ?? 2;
            if (nodeCount < 2 || nodeCount > 100) {
                throw new Error('Number of nodes for cluster type multi-node must be at least 2 and no more than 100');
            }
            return nodeCount;
        }
    }
}
exports.Cluster = Cluster;
_a = JSII_RTTI_SYMBOL_1;
Cluster[_a] = { fqn: "@aws-cdk/aws-redshift-alpha.Cluster", version: "2.33.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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