"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Port = exports.Protocol = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
/**
 * Protocol for use in Connection Rules
 *
 * https://www.iana.org/assignments/protocol-numbers/protocol-numbers.xhtml
 */
var Protocol;
(function (Protocol) {
    Protocol["ALL"] = "-1";
    Protocol["HOPOPT"] = "0";
    Protocol["ICMP"] = "icmp";
    Protocol["IGMP"] = "2";
    Protocol["GGP"] = "3";
    Protocol["IPV4"] = "4";
    Protocol["ST"] = "5";
    Protocol["TCP"] = "tcp";
    Protocol["CBT"] = "7";
    Protocol["EGP"] = "8";
    Protocol["IGP"] = "9";
    Protocol["BBN_RCC_MON"] = "10";
    Protocol["NVP_II"] = "11";
    Protocol["PUP"] = "12";
    Protocol["EMCON"] = "14";
    Protocol["XNET"] = "15";
    Protocol["CHAOS"] = "16";
    Protocol["UDP"] = "udp";
    Protocol["MUX"] = "18";
    Protocol["DCN_MEAS"] = "19";
    Protocol["HMP"] = "20";
    Protocol["PRM"] = "21";
    Protocol["XNS_IDP"] = "22";
    Protocol["TRUNK_1"] = "23";
    Protocol["TRUNK_2"] = "24";
    Protocol["LEAF_1"] = "25";
    Protocol["LEAF_2"] = "26";
    Protocol["RDP"] = "27";
    Protocol["IRTP"] = "28";
    Protocol["ISO_TP4"] = "29";
    Protocol["NETBLT"] = "30";
    Protocol["MFE_NSP"] = "31";
    Protocol["MERIT_INP"] = "32";
    Protocol["DCCP"] = "33";
    Protocol["THREEPC"] = "34";
    Protocol["IDPR"] = "35";
    Protocol["XTP"] = "36";
    Protocol["DDP"] = "37";
    Protocol["IDPR_CMTP"] = "38";
    Protocol["TPPLUSPLUS"] = "39";
    Protocol["IL"] = "40";
    Protocol["IPV6"] = "41";
    Protocol["SDRP"] = "42";
    Protocol["IPV6_ROUTE"] = "43";
    Protocol["IPV6_FRAG"] = "44";
    Protocol["IDRP"] = "45";
    Protocol["RSVP"] = "46";
    Protocol["GRE"] = "47";
    Protocol["DSR"] = "48";
    Protocol["BNA"] = "49";
    Protocol["ESP"] = "50";
    Protocol["AH"] = "51";
    Protocol["I_NLSP"] = "52";
    Protocol["SWIPE"] = "53";
    Protocol["NARP"] = "54";
    Protocol["MOBILE"] = "55";
    Protocol["TLSP"] = "56";
    Protocol["SKIP"] = "57";
    Protocol["ICMPV6"] = "icmpv6";
    Protocol["IPV6_NONXT"] = "59";
    Protocol["IPV6_OPTS"] = "60";
    Protocol["CFTP"] = "62";
    Protocol["ANY_LOCAL"] = "63";
    Protocol["SAT_EXPAK"] = "64";
    Protocol["KRYPTOLAN"] = "65";
    Protocol["RVD"] = "66";
    Protocol["IPPC"] = "67";
    Protocol["ANY_DFS"] = "68";
    Protocol["SAT_MON"] = "69";
    Protocol["VISA"] = "70";
    Protocol["IPCV"] = "71";
    Protocol["CPNX"] = "72";
    Protocol["CPHB"] = "73";
    Protocol["WSN"] = "74";
    Protocol["PVP"] = "75";
    Protocol["BR_SAT_MON"] = "76";
    Protocol["SUN_ND"] = "77";
    Protocol["WB_MON"] = "78";
    Protocol["WB_EXPAK"] = "79";
    Protocol["ISO_IP"] = "80";
    Protocol["VMTP"] = "81";
    Protocol["SECURE_VMTP"] = "82";
    Protocol["VINES"] = "83";
    Protocol["TTP"] = "84";
    Protocol["IPTM"] = "84";
    Protocol["NSFNET_IGP"] = "85";
    Protocol["DGP"] = "86";
    Protocol["TCF"] = "87";
    Protocol["EIGRP"] = "88";
    Protocol["OSPFIGP"] = "89";
    Protocol["SPRITE_RPC"] = "90";
    Protocol["LARP"] = "91";
    Protocol["MTP"] = "92";
    Protocol["AX_25"] = "93";
    Protocol["IPIP"] = "94";
    Protocol["MICP"] = "95";
    Protocol["SCC_SP"] = "96";
    Protocol["ETHERIP"] = "97";
    Protocol["ENCAP"] = "98";
    Protocol["ANY_ENC"] = "99";
    Protocol["GMTP"] = "100";
    Protocol["IFMP"] = "101";
    Protocol["PNNI"] = "102";
    Protocol["PIM"] = "103";
    Protocol["ARIS"] = "104";
    Protocol["SCPS"] = "105";
    Protocol["QNX"] = "106";
    Protocol["A_N"] = "107";
    Protocol["IPCOMP"] = "108";
    Protocol["SNP"] = "109";
    Protocol["COMPAQ_PEER"] = "110";
    Protocol["IPX_IN_IP"] = "111";
    Protocol["VRRP"] = "112";
    Protocol["PGM"] = "113";
    Protocol["ANY_0_HOP"] = "114";
    Protocol["L2_T_P"] = "115";
    Protocol["DDX"] = "116";
    Protocol["IATP"] = "117";
    Protocol["STP"] = "118";
    Protocol["SRP"] = "119";
    Protocol["UTI"] = "120";
    Protocol["SMP"] = "121";
    Protocol["SM"] = "122";
    Protocol["PTP"] = "123";
    Protocol["ISIS_IPV4"] = "124";
    Protocol["FIRE"] = "125";
    Protocol["CRTP"] = "126";
    Protocol["CRUDP"] = "127";
    Protocol["SSCOPMCE"] = "128";
    Protocol["IPLT"] = "129";
    Protocol["SPS"] = "130";
    Protocol["PIPE"] = "131";
    Protocol["SCTP"] = "132";
    Protocol["FC"] = "133";
    Protocol["RSVP_E2E_IGNORE"] = "134";
    Protocol["MOBILITY_HEADER"] = "135";
    Protocol["UDPLITE"] = "136";
    Protocol["MPLS_IN_IP"] = "137";
    Protocol["MANET"] = "138";
    Protocol["HIP"] = "139";
    Protocol["SHIM6"] = "140";
    Protocol["WESP"] = "141";
    Protocol["ROHC"] = "142";
    Protocol["ETHERNET"] = "143";
    Protocol["EXPERIMENT_1"] = "253";
    Protocol["EXPERIMENT_2"] = "254";
    Protocol["RESERVED"] = "255";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
/**
 * Interface for classes that provide the connection-specification parts of a security group rule
 */
class Port {
    constructor(props) {
        this.props = props;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_PortProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.canInlineRule = !core_1.Token.isUnresolved(props.fromPort) && !core_1.Token.isUnresolved(props.toPort);
    }
    /**
     * A single TCP port
     */
    static tcp(port) {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: port,
            toPort: port,
            stringRepresentation: renderPort(port),
        });
    }
    /**
     * A TCP port range
     */
    static tcpRange(startPort, endPort) {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: startPort,
            toPort: endPort,
            stringRepresentation: `${renderPort(startPort)}-${renderPort(endPort)}`,
        });
    }
    /**
     * Any TCP traffic
     */
    static allTcp() {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: 0,
            toPort: 65535,
            stringRepresentation: 'ALL PORTS',
        });
    }
    /**
     * A single UDP port
     */
    static udp(port) {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: port,
            toPort: port,
            stringRepresentation: `UDP ${renderPort(port)}`,
        });
    }
    /**
     * A UDP port range
     */
    static udpRange(startPort, endPort) {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: startPort,
            toPort: endPort,
            stringRepresentation: `UDP ${renderPort(startPort)}-${renderPort(endPort)}`,
        });
    }
    /**
     * Any UDP traffic
     */
    static allUdp() {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: 0,
            toPort: 65535,
            stringRepresentation: 'UDP ALL PORTS',
        });
    }
    /**
     * A specific combination of ICMP type and code
     *
     * @see https://www.iana.org/assignments/icmp-parameters/icmp-parameters.xhtml
     */
    static icmpTypeAndCode(type, code) {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: type,
            toPort: code,
            stringRepresentation: `ICMP Type ${type} Code ${code}`,
        });
    }
    /**
     * All codes for a single ICMP type
     */
    static icmpType(type) {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: type,
            toPort: -1,
            stringRepresentation: `ICMP Type ${type}`,
        });
    }
    /**
     * ICMP ping (echo) traffic
     */
    static icmpPing() {
        return Port.icmpType(8);
    }
    /**
     * All ICMP traffic
     */
    static allIcmp() {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: -1,
            toPort: -1,
            stringRepresentation: 'ALL ICMP',
        });
    }
    /**
     * All traffic
     */
    static allTraffic() {
        return new Port({
            protocol: Protocol.ALL,
            stringRepresentation: 'ALL TRAFFIC',
        });
    }
    /**
     * A single ESP port
     */
    static esp() {
        return new Port({
            protocol: Protocol.ESP,
            fromPort: 50,
            toPort: 50,
            stringRepresentation: 'ESP 50',
        });
    }
    /**
     * A single AH port
     */
    static ah() {
        return new Port({
            protocol: Protocol.AH,
            fromPort: 51,
            toPort: 51,
            stringRepresentation: 'AH 51',
        });
    }
    /**
     * Produce the ingress/egress rule JSON for the given connection
     */
    toRuleJson() {
        return {
            ipProtocol: this.props.protocol,
            fromPort: this.props.fromPort,
            toPort: this.props.toPort,
        };
    }
    toString() {
        return this.props.stringRepresentation;
    }
}
exports.Port = Port;
_a = JSII_RTTI_SYMBOL_1;
Port[_a] = { fqn: "@aws-cdk/aws-ec2.Port", version: "1.158.0" };
function renderPort(port) {
    return core_1.Token.isUnresolved(port) ? '{IndirectPort}' : port.toString();
}
//# sourceMappingURL=data:application/json;base64,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