# -*- coding: utf-8 -*-
# Copyright (c) 2015-2022, Exa Analytics Development Team
# Distributed under the terms of the Apache License 2.0
r"""
Cartesian to spherical transforms, obtained symbolically, numbafied.
Generated using the following code.

.. code-block:: Python

    from exatomic.algorithms.basis import enum_cartesian, car2sph

    def _car2sph_raw(L):
        fmt = '{: 20.16f},'
        c2s = car2sph(sh, enum_cartesian)
        for L in range(11):
            if not L:
                f += '    if not L: return np.array([1.])\n'
                continue
            tmt = (fmt * 4 + '\n') * (c2s[L].size // 4) + fmt * (c2s[L].size % 4)
            cdim, sdim = c2s[L].shape
            scaled = c2s[L] / (2 * np.pi ** 0.5)
            vals = tmt.format(*scaled.flatten())
            f += '    if L == {}:\n        return np.array([\n{}])\n'.format(L, vals)
        return f

"""
import numpy as np
from numba import jit

@jit(nopython=True, cache=True)
def car2sph_scaled(L):
    """Coefficients of symbolic solid harmonics / (2 * pi ** 0.5)."""
    return _car2sph_2d(L)

@jit(nopython=True, cache=True)
def _car2sph_2d(L):
    """Reshape the arrays to avoid nested memory-managed objects."""
    cdim = (L + 1) * (L + 2) // 2
    sdim = 2 * L + 1
    flat = _car2sph_raw(L)
    sq = np.empty((cdim, sdim))
    for c in range(cdim):
        for s in range(sdim):
            sq[c, s] = flat[c * sdim + s]
    return sq

@jit(nopython=True, cache=True)
def _car2sph_raw(L):
    """Dump of numerical coefficients from symbolic solid harmonics
    scaled by (4pi)**-0.5 to match the real spherical harmonics."""
    if not L:
        return np.array([1.])
    if L == 1:
        return np.array([
  0.2820947917738781,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.2820947917738781,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.2820947917738781,])
    if L == 2:
        return np.array([
  0.0000000000000000,  0.0000000000000000, -0.1410473958869391,  0.0000000000000000,
  0.2443012559514601,  0.4886025119029208,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.4886025119029208,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -0.1410473958869391,  0.0000000000000000, -0.2443012559514601,
  0.0000000000000000,  0.4886025119029208,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.2820947917738781,
  0.0000000000000000,  0.0000000000000000,])
    if L == 3:
        return np.array([
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 -0.1727470747356676,  0.0000000000000000,  0.2230155145190965,  0.6690465435572880,
  0.0000000000000000, -0.1727470747356676,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -0.4231421876608172,  0.0000000000000000,  0.5462742152960400,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -0.1727470747356676,  0.0000000000000000, -0.6690465435572880,
  0.0000000000000000,  1.0925484305920801,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.6909882989426716,
  0.0000000000000000,  0.0000000000000000, -0.2230155145190965,  0.0000000000000000,
 -0.1727470747356676,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 -0.4231421876608172,  0.0000000000000000, -0.5462742152960400,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.6909882989426716,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.2820947917738781,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,])
    if L == 4:
        return np.array([
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.1057855469152043,  0.0000000000000000, -0.1576957826262602,  0.0000000000000000,
  0.2086119118163921,  0.8344476472655687,  0.0000000000000000, -0.3153915652525215,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, -0.6690465435572880,
  0.0000000000000000,  0.5900435899266439,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.2115710938304086,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, -1.2516714708983516,
  0.0000000000000000,  1.7701307697799316,  0.0000000000000000, -0.6690465435572880,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -0.8462843753216345,  0.0000000000000000,  0.9461746957575587,
  0.0000000000000000,  0.0000000000000000, -0.8344476472655687,  0.0000000000000000,
 -0.3153915652525215,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 -0.6690465435572880,  0.0000000000000000, -1.7701307697799316,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  1.8923493915151206,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.8920620580763858,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.1057855469152043,  0.0000000000000000,
  0.1576957826262602,  0.0000000000000000,  0.2086119118163921,  0.0000000000000000,
 -0.5900435899266439,  0.0000000000000000, -0.6690465435572880,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 -0.8462843753216345,  0.0000000000000000, -0.9461746957575587,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.8920620580763858,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.2820947917738781,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,])
    if L == 5:
        return np.array([
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.1365685538240099,  0.0000000000000000,
 -0.1475108974816608,  0.0000000000000000,  0.1979066365145978,  0.9895331825729888,
  0.0000000000000000, -0.4425326924449822,  0.0000000000000000,  0.1365685538240099,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.5289277345760216,
  0.0000000000000000, -0.7226528606601387,  0.0000000000000000,  0.6258357354491774,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.2731371076480197,
  0.0000000000000000,  0.2950217949633205,  0.0000000000000000, -1.9790663651459777,
  0.0000000000000000,  2.5033429417967032,  0.0000000000000000, -1.4453057213202773,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -1.6388226458881201,  0.0000000000000000,  1.1800871798532877,
  0.0000000000000000,  0.0000000000000000, -1.9790663651459777,  0.0000000000000000,
 -0.2950217949633205,  0.0000000000000000,  0.2731371076480197,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  1.0578554691520432,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, -3.7550144126950467,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  3.5402615395598520,  0.0000000000000000,
 -1.6388226458881201,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 -1.4104739588693909,  0.0000000000000000,  1.4453057213202773,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.1365685538240099,  0.0000000000000000,  0.4425326924449822,  0.0000000000000000,
  0.9895331825729888,  0.0000000000000000, -2.5033429417967032,  0.0000000000000000,
 -1.4453057213202773,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, -1.6388226458881201,  0.0000000000000000,
 -3.5402615395598520,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  2.8906114426405547,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  1.0925484305920801,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.1979066365145978,  0.0000000000000000,  0.1475108974816608,
  0.0000000000000000,  0.1365685538240099,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.5289277345760216,  0.0000000000000000,  0.7226528606601387,
  0.0000000000000000,  0.6258357354491774,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -1.1800871798532877,  0.0000000000000000, -1.6388226458881201,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, -1.4104739588693909,
  0.0000000000000000, -1.4453057213202773,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  1.0925484305920801,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.2820947917738781,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,])
    if L == 6:
        return np.array([
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, -0.0881546224293369,  0.0000000000000000,
  0.1277481845541603,  0.0000000000000000, -0.1399411247212934,  0.0000000000000000,
  0.1894811786039562,  1.1368870716237383,  0.0000000000000000, -0.5597644988851723,
  0.0000000000000000,  0.2554963691083206,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.8079504602853766,  0.0000000000000000, -0.7664891073249619,
  0.0000000000000000,  0.6563820568401715,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -0.2644638672880108,  0.0000000000000000,  0.1277481845541603,
  0.0000000000000000,  0.6997056236064654,  0.0000000000000000, -2.8422176790593316,
  0.0000000000000000,  3.2819102842008427,  0.0000000000000000, -2.2994673219748853,
  0.0000000000000000,  0.8079504602853766,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  1.5867832037280647,
  0.0000000000000000, -2.0439709528665650,  0.0000000000000000,  1.3994112472129336,
  0.0000000000000000,  0.0000000000000000, -3.7896235720791180,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.5109927382166413,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  1.6159009205707533,  0.0000000000000000,
  1.5329782146499238,  0.0000000000000000, -6.5638205684017139,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  5.5976449888517230,  0.0000000000000000,
 -4.0879419057331194,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 -3.2318018411415066,  0.0000000000000000,  2.0439709528665650,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 -0.2644638672880108,  0.0000000000000000, -0.1277481845541603,  0.0000000000000000,
  0.6997056236064654,  0.0000000000000000,  2.8422176790593316,  0.0000000000000000,
 -6.5638205684017139,  0.0000000000000000, -1.5329782146499238,  0.0000000000000000,
  1.6159009205707533,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  3.1735664074561294,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, -8.3964674832775845,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  6.1319128585997067,  0.0000000000000000, -3.2318018411415066,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 -2.1157109383040864,  0.0000000000000000,  2.0439709528665650,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  1.1368870716237383,
  0.0000000000000000,  0.5597644988851723,  0.0000000000000000,  0.2554963691083206,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.8079504602853766,
  0.0000000000000000,  2.2994673219748853,  0.0000000000000000,  3.2819102842008427,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, -5.5976449888517230,
  0.0000000000000000, -4.0879419057331194,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -3.2318018411415066,  0.0000000000000000, -6.1319128585997067,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  4.0879419057331194,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  1.2927207364566027,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, -0.0881546224293369,
  0.0000000000000000, -0.1277481845541603,  0.0000000000000000, -0.1399411247212934,
  0.0000000000000000, -0.1894811786039562,  0.0000000000000000,  0.6563820568401715,
  0.0000000000000000,  0.7664891073249619,  0.0000000000000000,  0.8079504602853766,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  1.5867832037280647,  0.0000000000000000,  2.0439709528665650,
  0.0000000000000000,  1.3994112472129336,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, -2.0439709528665650,
  0.0000000000000000, -3.2318018411415066,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, -2.1157109383040864,
  0.0000000000000000, -2.0439709528665650,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  1.2927207364566027,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.2820947917738781,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
])
    if L == 7:
        return np.array([
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 -0.1166176039344110,  0.0000000000000000,  0.1211925690428065,  0.0000000000000000,
 -0.1339834262980767,  0.0000000000000000,  0.1825886324084193,  1.2781204268589372,
  0.0000000000000000, -0.6699171314903851,  0.0000000000000000,  0.3635777071284195,
  0.0000000000000000, -0.1166176039344110,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -0.6170823570053585,  0.0000000000000000,  0.8569608739958748,
  0.0000000000000000, -0.8039005577884610,  0.0000000000000000,  0.6831841051919156,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -0.3498528118032341,  0.0000000000000000, -0.1211925690428065,
  0.0000000000000000,  1.2058508366826917,  0.0000000000000000, -3.8343612805767977,
  0.0000000000000000,  4.0991046311514943,  0.0000000000000000, -3.2156022311538441,
  0.0000000000000000,  1.7139217479917470,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  2.7988224944258673,
  0.0000000000000000, -2.4238513808561302,  0.0000000000000000,  1.6078011155769221,
  0.0000000000000000,  0.0000000000000000, -6.3906021342946717,  0.0000000000000000,
  0.6699171314903851,  0.0000000000000000,  0.6059628452140325,  0.0000000000000000,
 -0.3498528118032341,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 -1.8512470710160756,  0.0000000000000000,  0.8569608739958748,  0.0000000000000000,
  4.0195027889423054,  0.0000000000000000,-10.2477615778787072,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  8.0390055778846108,  0.0000000000000000,
 -7.2715541425683901,  0.0000000000000000,  2.7988224944258673,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  3.7024941420321511,  0.0000000000000000,
 -4.5704579946446717,  0.0000000000000000,  2.6796685259615378,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, -0.3498528118032341,  0.0000000000000000,
 -0.6059628452140325,  0.0000000000000000,  0.6699171314903851,  0.0000000000000000,
  6.3906021342946717,  0.0000000000000000,-13.6636821038382852,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  3.4278434959834825,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  5.5976449888517230,  0.0000000000000000,  4.8477027617122603,  0.0000000000000000,
-16.0780111557692216,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, 10.7186741038461566,  0.0000000000000000,
 -9.1409159892893150,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, -5.5976449888517230,  0.0000000000000000,
  3.2318018411415066,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  3.8343612805767977,  0.0000000000000000,  1.2058508366826917,
  0.0000000000000000,  0.1211925690428065,  0.0000000000000000, -0.3498528118032341,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, -1.8512470710160756,
  0.0000000000000000, -0.8569608739958748,  0.0000000000000000,  4.0195027889423054,
  0.0000000000000000, 10.2477615778787072,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,-16.0780111557692216,  0.0000000000000000, -4.8477027617122603,
  0.0000000000000000,  5.5976449888517230,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  7.4049882840643022,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,-16.0780111557692216,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  9.6954055234245207,  0.0000000000000000, -5.5976449888517230,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, -2.9619953136257209,
  0.0000000000000000,  2.7422747967867949,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, -0.1166176039344110,
  0.0000000000000000, -0.3635777071284195,  0.0000000000000000, -0.6699171314903851,
  0.0000000000000000, -1.2781204268589372,  0.0000000000000000,  4.0991046311514943,
  0.0000000000000000,  3.2156022311538441,  0.0000000000000000,  1.7139217479917470,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  2.7988224944258673,  0.0000000000000000,  7.2715541425683901,
  0.0000000000000000,  8.0390055778846108,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,-10.7186741038461566,
  0.0000000000000000, -9.1409159892893150,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, -5.5976449888517230,
  0.0000000000000000, -9.6954055234245207,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  5.4845495935735942,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  1.4927053303604612,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 -0.1825886324084193,  0.0000000000000000, -0.1339834262980767,  0.0000000000000000,
 -0.1211925690428065,  0.0000000000000000, -0.1166176039344110,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, -0.6170823570053585,  0.0000000000000000,
 -0.8569608739958748,  0.0000000000000000, -0.8039005577884610,  0.0000000000000000,
 -0.6831841051919156,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  1.6078011155769221,  0.0000000000000000,  2.4238513808561302,  0.0000000000000000,
  2.7988224944258673,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  3.7024941420321511,  0.0000000000000000,  4.5704579946446717,  0.0000000000000000,
  2.6796685259615378,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 -3.2318018411415066,  0.0000000000000000, -5.5976449888517230,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, -2.9619953136257209,  0.0000000000000000,
 -2.7422747967867949,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  1.4927053303604612,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.2820947917738779,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
])
    if L == 8:
        return np.array([
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0771352946256698,  0.0000000000000000, -0.1106331731112457,  0.0000000000000000,
  0.1160330508602146,  0.0000000000000000, -0.1291096601435711,  0.0000000000000000,
  0.1767906831311491,  1.4143254650491921,  0.0000000000000000, -0.7746579608614266,
  0.0000000000000000,  0.4641322034408574,  0.0000000000000000, -0.2212663462224914,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, -0.9256235355080378,
  0.0000000000000000,  0.8987881471824997,  0.0000000000000000, -0.8367262290500490,
  0.0000000000000000,  0.7071627325245974,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.3085411785026793,
  0.0000000000000000, -0.2212663462224914,  0.0000000000000000, -0.4641322034408574,
  0.0000000000000000,  1.8075352420099953,  0.0000000000000000, -4.9501391276721654,
  0.0000000000000000,  4.9501391276721654,  0.0000000000000000, -4.1836311452502448,
  0.0000000000000000,  2.6963644415474963,  0.0000000000000000, -0.9256235355080378,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -2.4683294280214341,  0.0000000000000000,  3.3189951933373596,
  0.0000000000000000, -2.7847932206451502,  0.0000000000000000,  1.8075352420099982,
  0.0000000000000000,  0.0000000000000000, -9.9002782553443591,  0.0000000000000000,
  1.8075352420099953,  0.0000000000000000,  0.4641322034408574,  0.0000000000000000,
 -0.6637990386674747,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 -2.7768706065241133,  0.0000000000000000, -0.8987881471824997,  0.0000000000000000,
  7.5305360614504462,  0.0000000000000000,-14.8504173830165236,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, 10.8452114520599725,  0.0000000000000000,
-11.1391728825806116,  0.0000000000000000,  6.6379903866747476,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  7.4049882840643022,  0.0000000000000000,
 -5.9919209812166452,  0.0000000000000000,  3.3469049162002014,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.4628117677540189,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, -1.1603305086021463,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, 12.3753478191804405,  0.0000000000000000,
-24.7506956383608525,  0.0000000000000000,  4.1836311452502448,  0.0000000000000000,
  4.4939407359124841,  0.0000000000000000, -2.7768706065241133,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 -7.4049882840643022,  0.0000000000000000,  3.3189951933373596,  0.0000000000000000,
 13.9239661032257498,  0.0000000000000000,-27.1130286301499588,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 16.7345245810009793,  0.0000000000000000,-17.9757629436499649,  0.0000000000000000,
  7.4049882840643022,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  7.4049882840643022,  0.0000000000000000,
 -8.8506538488996434,  0.0000000000000000,  4.6413220344085735,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  9.9002782553443591,
  0.0000000000000000,  1.8075352420099953,  0.0000000000000000, -0.4641322034408574,
  0.0000000000000000, -0.6637990386674747,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -2.7768706065241133,  0.0000000000000000, -4.4939407359124841,
  0.0000000000000000,  4.1836311452502448,  0.0000000000000000, 24.7506956383608525,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,-36.1507048401999143,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, 13.2759807733494952,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, 14.8099765681286044,
  0.0000000000000000, 11.9838419624333188,  0.0000000000000000,-33.4690491620020154,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, 18.5652881376343224,
  0.0000000000000000,-17.7013076977993151,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -8.8859859408771626,  0.0000000000000000,  4.7935367849733161,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.3085411785026793,  0.0000000000000000,  0.2212663462224914,
  0.0000000000000000, -0.4641322034408574,  0.0000000000000000, -1.8075352420099953,
  0.0000000000000000, -4.9501391276721654,  0.0000000000000000, 14.8504173830165236,
  0.0000000000000000,  7.5305360614504462,  0.0000000000000000,  0.8987881471824997,
  0.0000000000000000, -2.7768706065241133,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, -7.4049882840643022,
  0.0000000000000000, -3.3189951933373596,  0.0000000000000000, 13.9239661032257498,
  0.0000000000000000, 27.1130286301499588,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,-33.4690491620020154,
  0.0000000000000000,-11.9838419624333188,  0.0000000000000000, 14.8099765681286044,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, 14.8099765681286044,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,-27.8479322064514996,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, 14.3806103549199769,
  0.0000000000000000, -8.8859859408771626,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, -3.9493270848342945,
  0.0000000000000000,  3.5402615395598520,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 -1.4143254650491921,  0.0000000000000000, -0.7746579608614266,  0.0000000000000000,
 -0.4641322034408574,  0.0000000000000000, -0.2212663462224914,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, -0.9256235355080378,  0.0000000000000000,
 -2.6963644415474963,  0.0000000000000000, -4.1836311452502448,  0.0000000000000000,
 -4.9501391276721654,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 10.8452114520599725,  0.0000000000000000, 11.1391728825806116,  0.0000000000000000,
  6.6379903866747476,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  7.4049882840643022,  0.0000000000000000, 17.9757629436499649,  0.0000000000000000,
 16.7345245810009793,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
-18.5652881376343224,  0.0000000000000000,-17.7013076977993151,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, -8.8859859408771626,  0.0000000000000000,
-14.3806103549199769,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  7.0805230791197324,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  1.6925687506432689,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0771352946256698,  0.0000000000000000,  0.1106331731112457,  0.0000000000000000,
  0.1160330508602146,  0.0000000000000000,  0.1291096601435711,  0.0000000000000000,
  0.1767906831311491,  0.0000000000000000, -0.7071627325245974,  0.0000000000000000,
 -0.8367262290500490,  0.0000000000000000, -0.8987881471824997,  0.0000000000000000,
 -0.9256235355080378,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, -2.4683294280214341,  0.0000000000000000,
 -3.3189951933373596,  0.0000000000000000, -2.7847932206451502,  0.0000000000000000,
 -1.8075352420099982,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  3.3469049162002014,  0.0000000000000000,
  5.9919209812166452,  0.0000000000000000,  7.4049882840643022,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  7.4049882840643022,  0.0000000000000000,  8.8506538488996434,  0.0000000000000000,
  4.6413220344085735,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, -4.7935367849733161,  0.0000000000000000,
 -8.8859859408771626,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, -3.9493270848342945,  0.0000000000000000,
 -3.5402615395598520,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  1.6925687506432689,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.2820947917738779,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,])
    if L == 9:
        return np.array([
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.1034878573484831,  0.0000000000000000,
 -0.1059231989538063,  0.0000000000000000,  0.1118122455583079,  0.0000000000000000,
 -0.1250098908926377,  0.0000000000000000,  0.1718096614657615,  1.5462869531918517,
  0.0000000000000000, -0.8750692362484647,  0.0000000000000000,  0.5590612277915396,
  0.0000000000000000, -0.3177695968614187,  0.0000000000000000,  0.1034878573484831,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.6942176516310283,
  0.0000000000000000, -0.9708021541153685,  0.0000000000000000,  0.9354883633564849,
  0.0000000000000000, -0.8660939298987606,  0.0000000000000000,  0.7289266601748308,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.4139514293939334,
  0.0000000000000000,  0.0000000000000001,  0.0000000000000000, -0.8944979644664643,
  0.0000000000000000,  2.5001978178527531,  0.0000000000000000, -6.1851478127674238,
  0.0000000000000000,  5.8314132813986461,  0.0000000000000000, -5.1965635793925626,
  0.0000000000000000,  3.7419534534259395,  0.0000000000000000, -1.9416043082307370,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -4.1395142939393335,  0.0000000000000000,  3.8132351623370364,
  0.0000000000000000, -3.1307428756326225,  0.0000000000000000,  2.0001582542822032,
  0.0000000000000000,  0.0000000000000000,-14.4320115631239627,  0.0000000000000000,
  3.5002769449938533,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 -0.8473855916304500,  0.0000000000000000,  0.4139514293939334,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  2.7768706065241133,  0.0000000000000000,
 -1.9416043082307370,  0.0000000000000000, -3.7419534534259395,  0.0000000000000000,
 12.1253150185826488,  0.0000000000000000,-20.4099464848952330,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, 14.0011077799754133,  0.0000000000000000,
-15.6537143781631123,  0.0000000000000000, 11.4397054870110804,  0.0000000000000000,
 -4.1395142939393335,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 -7.4049882840643022,  0.0000000000000000,  9.7080215411536841,  0.0000000000000000,
 -7.4839069068518791,  0.0000000000000000,  4.0417716728608823,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.6209271440909000,  0.0000000000000000,  0.6355391937228374,  0.0000000000000000,
 -1.5653714378163113,  0.0000000000000000, -1.7501384724969267,  0.0000000000000000,
 21.6480173446859432,  0.0000000000000000,-40.8198929697904092,  0.0000000000000000,
 12.1253150185826488,  0.0000000000000000,  3.7419534534259395,  0.0000000000000000,
 -5.8248129246922220,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,-12.4185428818179737,  0.0000000000000000,
 -3.8132351623370364,  0.0000000000000000, 28.1766858806935723,  0.0000000000000000,
-42.0033233399263537,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, 24.2506300371653261,  0.0000000000000000,
-29.9356276274075164,  0.0000000000000000, 19.4160430823073682,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 16.5580571757573090,  0.0000000000000000,-12.7107838744567498,  0.0000000000000000,
  6.2614857512652451,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, 21.6480173446859432,  0.0000000000000000,  1.7501384724969267,
  0.0000000000000000, -1.5653714378163113,  0.0000000000000000, -0.6355391937228374,
  0.0000000000000000,  0.6209271440909000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  4.1653059097861700,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -9.3548836335648495,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, 51.0248662122380026,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,-70.0055388998770667,  0.0000000000000000, 15.6537143781631123,
  0.0000000000000000, 19.0661758116851274,  0.0000000000000000,-12.4185428818179737,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,-22.2149648521929066,
  0.0000000000000000,  9.7080215411536841,  0.0000000000000000, 37.4195345342593981,
  0.0000000000000000,-60.6265750929132423,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, 31.3074287563262246,  0.0000000000000000,-38.1323516233703614,
  0.0000000000000000, 16.5580571757573090,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, 13.3289789113157440,  0.0000000000000000,-15.5328344658459070,
  0.0000000000000000,  7.4839069068518791,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.4139514293939334,  0.0000000000000000,  0.8473855916304500,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, -3.5002769449938533,
  0.0000000000000000,-14.4320115631239627,  0.0000000000000000, 40.8198929697904092,
  0.0000000000000000, 12.1253150185826488,  0.0000000000000000, -3.7419534534259395,
  0.0000000000000000, -5.8248129246922220,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,-12.4185428818179737,
  0.0000000000000000,-19.0661758116851274,  0.0000000000000000, 15.6537143781631123,
  0.0000000000000000, 70.0055388998770667,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,-80.8354334572176469,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, 38.8320861646146227,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, 33.1161143515146179,  0.0000000000000000, 25.4215677489134997,
  0.0000000000000000,-62.6148575126524491,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, 29.9356276274075164,
  0.0000000000000000,-31.0656689316918140,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,-13.2464457406058447,
  0.0000000000000000,  6.7790847330435886,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 -6.1851478127674238,  0.0000000000000000, -2.5001978178527531,  0.0000000000000000,
 -0.8944979644664643,  0.0000000000000000, -0.0000000000000001,  0.0000000000000000,
  0.4139514293939334,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  2.7768706065241133,  0.0000000000000000,  1.9416043082307370,  0.0000000000000000,
 -3.7419534534259395,  0.0000000000000000,-12.1253150185826488,  0.0000000000000000,
-20.4099464848952330,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 42.0033233399263537,  0.0000000000000000, 28.1766858806935723,  0.0000000000000000,
  3.8132351623370364,  0.0000000000000000,-12.4185428818179737,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,-22.2149648521929066,  0.0000000000000000,
 -9.7080215411536841,  0.0000000000000000, 37.4195345342593981,  0.0000000000000000,
 60.6265750929132423,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
-62.6148575126524491,  0.0000000000000000,-25.4215677489134997,  0.0000000000000000,
 33.1161143515146179,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 26.6579578226314879,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
-44.9034414411112763,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 20.3372541991307934,  0.0000000000000000,-13.2464457406058447,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, -5.0777062519298068,  0.0000000000000000,
  4.4379527045274063,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.1034878573484831,  0.0000000000000000,
  0.3177695968614187,  0.0000000000000000,  0.5590612277915396,  0.0000000000000000,
  0.8750692362484647,  0.0000000000000000,  1.5462869531918517,  0.0000000000000000,
 -5.8314132813986461,  0.0000000000000000, -5.1965635793925626,  0.0000000000000000,
 -3.7419534534259395,  0.0000000000000000, -1.9416043082307370,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 -4.1395142939393335,  0.0000000000000000,-11.4397054870110804,  0.0000000000000000,
-15.6537143781631123,  0.0000000000000000,-14.0011077799754133,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 24.2506300371653261,  0.0000000000000000, 29.9356276274075164,  0.0000000000000000,
 19.4160430823073682,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, 16.5580571757573090,  0.0000000000000000,
 38.1323516233703614,  0.0000000000000000, 31.3074287563262246,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
-29.9356276274075164,  0.0000000000000000,-31.0656689316918140,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
-13.2464457406058447,  0.0000000000000000,-20.3372541991307934,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  8.8759054090547842,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  1.8923493915151175,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.1718096614657615,
  0.0000000000000000,  0.1250098908926377,  0.0000000000000000,  0.1118122455583079,
  0.0000000000000000,  0.1059231989538063,  0.0000000000000000,  0.1034878573484831,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.6942176516310283,
  0.0000000000000000,  0.9708021541153685,  0.0000000000000000,  0.9354883633564849,
  0.0000000000000000,  0.8660939298987606,  0.0000000000000000,  0.7289266601748308,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, -2.0001582542822032,
  0.0000000000000000, -3.1307428756326225,  0.0000000000000000, -3.8132351623370364,
  0.0000000000000000, -4.1395142939393335,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -7.4049882840643022,  0.0000000000000000, -9.7080215411536841,
  0.0000000000000000, -7.4839069068518791,  0.0000000000000000, -4.0417716728608823,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  6.2614857512652451,
  0.0000000000000000, 12.7107838744567498,  0.0000000000000000, 16.5580571757573090,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, 13.3289789113157440,
  0.0000000000000000, 15.5328344658459070,  0.0000000000000000,  7.4839069068518791,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, -6.7790847330435886,
  0.0000000000000000,-13.2464457406058447,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -5.0777062519298068,  0.0000000000000000, -4.4379527045274063,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  1.8923493915151175,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.2820947917738776,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,])
    if L == 10:
        return np.array([
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, -0.0694217651631028,  0.0000000000000000,
  0.0990820799490588,  0.0000000000000000, -0.1020701484959867,  0.0000000000000000,
  0.1082617412373452,  0.0000000000000000, -0.1214877766958050,  0.0000000000000000,
  0.1674593436758026,  1.6745934367580257,  0.0000000000000000, -0.9719022135664390,
  0.0000000000000000,  0.6495704474240718,  0.0000000000000000, -0.4082805939839479,
  0.0000000000000000,  0.1981641598981174,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  1.0296911795482282,  0.0000000000000000, -1.0104429182152206,
  0.0000000000000000,  0.9683224510767847,  0.0000000000000000, -0.8927491886697227,
  0.0000000000000000,  0.7489009518531884,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -0.3471088258155142,  0.0000000000000000,  0.2972462398471751,
  0.0000000000000000,  0.3062104454879610,  0.0000000000000000, -1.4074026360854865,
  0.0000000000000000,  3.2801699707867455,  0.0000000000000000, -7.5356704654111191,
  0.0000000000000000,  6.7401085666786891,  0.0000000000000000, -6.2492443206880450,
  0.0000000000000000,  4.8416122553839092,  0.0000000000000000, -3.0313287546456760,
  0.0000000000000000,  1.0296911795482282,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  3.4710882581551417,
  0.0000000000000000, -4.7559398375548234,  0.0000000000000000,  4.2869462368314428,
  0.0000000000000000, -3.4643757195950422,  0.0000000000000000,  2.1867799805244892,
  0.0000000000000000,  0.0000000000000000,-20.0951212410963187,  0.0000000000000000,
  5.8314132813986461,  0.0000000000000000, -0.8660939298987606,  0.0000000000000000,
 -0.8165611879678932,  0.0000000000000000,  0.7926566395924706,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  4.1187647181929128,  0.0000000000000000,
  0.0000000000000003,  0.0000000000000000, -7.7465796086142662,  0.0000000000000000,
 17.8549837733944550,  0.0000000000000000,-26.9604342667147847,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, 17.4942398441959064,  0.0000000000000000,
-20.7862543175702790,  0.0000000000000000, 17.1477849473257713,  0.0000000000000000,
 -9.5118796751096468,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
-13.7292157273097004,  0.0000000000000000, 12.1253150185826488,  0.0000000000000000,
 -9.0376762100499786,  0.0000000000000000,  4.7613290062385216,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 -0.6942176516310283,  0.0000000000000000,  0.1981641598981174,  0.0000000000000000,
  1.4289820789438152,  0.0000000000000000, -1.5156643773228322,  0.0000000000000000,
 -5.1024866212238011,  0.0000000000000000, 35.1664621719184467,  0.0000000000000000,
-62.9076799556677315,  0.0000000000000000, 24.9969772827522370,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, -8.0835433457217647,  0.0000000000000000,
  4.1187647181929128,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, 13.8843530326205666,  0.0000000000000000,
 -9.5118796751096468,  0.0000000000000000,-17.1477849473257713,  0.0000000000000000,
 48.5012600743307587,  0.0000000000000000,-61.2298394546856031,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 33.3293030436695332,  0.0000000000000000,-45.1883810502500296,  0.0000000000000000,
 36.3759450557479980,  0.0000000000000000,-13.7292157273097004,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
-18.5124707101607555,  0.0000000000000000, 23.7796991877740886,  0.0000000000000000,
-17.1477849473257713,  0.0000000000000000,  8.0835433457217647,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, 42.1997546063021360,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, -3.0313287546456760,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  1.1889849593887059,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  6.1781470772893687,
  0.0000000000000000,  6.0626575092913244,  0.0000000000000000,-13.5565143150749794,
  0.0000000000000000,-12.4984886413761185,  0.0000000000000000, 94.3615199335015973,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,-122.4596789093715046,
  0.0000000000000000, 48.5012600743307587,  0.0000000000000000, 17.1477849473257713,
  0.0000000000000000,-28.5356390253289369,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,-41.1876471819291226,  0.0000000000000000,-12.1253150185826488,
  0.0000000000000000, 81.3390858904499368,  0.0000000000000000,-99.9879091310088768,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, 48.5012600743304745,
  0.0000000000000000,-68.5911397893031989,  0.0000000000000000, 47.5593983755482270,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, 32.9501177455433591,
  0.0000000000000000,-24.2506300371653261,  0.0000000000000000, 10.8452114520599725,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, -0.6942176516310283,
  0.0000000000000000, -0.1981641598981174,  0.0000000000000000,  1.4289820789438152,
  0.0000000000000000,  1.5156643773228322,  0.0000000000000000, -5.1024866212238011,
  0.0000000000000000,-35.1664621719184467,  0.0000000000000000, 94.3615199335015973,
  0.0000000000000000, 12.4984886413761185,  0.0000000000000000,-13.5565143150749794,
  0.0000000000000000, -6.0626575092913244,  0.0000000000000000,  6.1781470772893687,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, 20.8265295489308500,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,-42.8694623683144229,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, 153.0745986367143132,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,-166.6465152183482417,
  0.0000000000000000, 45.1883810502500296,  0.0000000000000000, 60.6265750929132423,
  0.0000000000000000,-41.1876471819291226,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,-55.5374121304822665,
  0.0000000000000000, 23.7796991877740886,  0.0000000000000000, 85.7389247366288458,
  0.0000000000000000,-121.2531501858264846,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, 54.2260572602998607,
  0.0000000000000000,-72.7518901114959959,  0.0000000000000000, 32.9501177455433591,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, 22.2149648521929066,  0.0000000000000000,-25.3650124669590298,
  0.0000000000000000, 11.4318566315505308,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
-20.0951212410963187,  0.0000000000000000, -5.8314132813986461,  0.0000000000000000,
 -0.8660939298987606,  0.0000000000000000,  0.8165611879678932,  0.0000000000000000,
  0.7926566395924706,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  4.1187647181929128,  0.0000000000000000,  8.0835433457217647,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,-24.9969772827522370,  0.0000000000000000,
-62.9076799556677315,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 122.4596789093715046,  0.0000000000000000, 48.5012600743307587,  0.0000000000000000,
-17.1477849473257713,  0.0000000000000000,-28.5356390253289369,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,-41.1876471819291226,  0.0000000000000000,
-60.6265750929132423,  0.0000000000000000, 45.1883810502500296,  0.0000000000000000,
 166.6465152183482417,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
-161.6708669144352939,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 95.1187967510964540,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 65.9002354910864341,  0.0000000000000000, 48.5012600743307587,  0.0000000000000000,
-108.4521145205997215,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 45.7274265262021231,  0.0000000000000000,-50.7300249339181164,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,-18.8286387117390213,  0.0000000000000000,
  9.2383352522534725,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, -0.3471088258155142,  0.0000000000000000,
 -0.2972462398471751,  0.0000000000000000,  0.3062104454879610,  0.0000000000000000,
  1.4074026360854865,  0.0000000000000000,  3.2801699707867455,  0.0000000000000000,
  7.5356704654111191,  0.0000000000000000,-26.9604342667147847,  0.0000000000000000,
-17.8549837733944550,  0.0000000000000000, -7.7465796086142662,  0.0000000000000000,
 -0.0000000000000003,  0.0000000000000000,  4.1187647181929128,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 13.8843530326205666,  0.0000000000000000,  9.5118796751096468,  0.0000000000000000,
-17.1477849473257713,  0.0000000000000000,-48.5012600743307587,  0.0000000000000000,
-61.2298394546856031,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, 99.9879091310088768,  0.0000000000000000,
 81.3390858904499368,  0.0000000000000000, 12.1253150185826488,  0.0000000000000000,
-41.1876471819291226,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,-55.5374121304822665,  0.0000000000000000,
-23.7796991877740886,  0.0000000000000000, 85.7389247366288458,  0.0000000000000000,
 121.2531501858264846,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,-108.4521145205997215,  0.0000000000000000,
-48.5012600743307587,  0.0000000000000000, 65.9002354910864341,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
 44.4299297043858132,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
-68.5911397893031989,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, 27.7150057567603660,  0.0000000000000000,
-18.8286387117390213,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000, -6.3471328149122588,  0.0000000000000000,
  5.4353598143483648,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  1.6745934367580257,  0.0000000000000000,  0.9719022135664390,
  0.0000000000000000,  0.6495704474240718,  0.0000000000000000,  0.4082805939839479,
  0.0000000000000000,  0.1981641598981174,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  1.0296911795482282,  0.0000000000000000,  3.0313287546456760,
  0.0000000000000000,  4.8416122553839092,  0.0000000000000000,  6.2492443206880450,
  0.0000000000000000,  6.7401085666786891,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,-17.4942398441959064,  0.0000000000000000,-20.7862543175702790,
  0.0000000000000000,-17.1477849473257713,  0.0000000000000000, -9.5118796751096468,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,-13.7292157273097004,
  0.0000000000000000,-36.3759450557479980,  0.0000000000000000,-45.1883810502500296,
  0.0000000000000000,-33.3293030436695332,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, 48.5012600743304745,  0.0000000000000000, 68.5911397893031989,
  0.0000000000000000, 47.5593983755482270,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, 32.9501177455433591,  0.0000000000000000, 72.7518901114959959,
  0.0000000000000000, 54.2260572602998607,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,-45.7274265262021231,  0.0000000000000000,-50.7300249339181164,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,-18.8286387117390213,
  0.0000000000000000,-27.7150057567603660,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, 10.8707196286967296,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  2.0920709679710008,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -0.0694217651631028,  0.0000000000000000, -0.0990820799490588,
  0.0000000000000000, -0.1020701484959867,  0.0000000000000000, -0.1082617412373452,
  0.0000000000000000, -0.1214877766958050,  0.0000000000000000, -0.1674593436758026,
  0.0000000000000000,  0.7489009518531884,  0.0000000000000000,  0.8927491886697227,
  0.0000000000000000,  0.9683224510767847,  0.0000000000000000,  1.0104429182152206,
  0.0000000000000000,  1.0296911795482282,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  3.4710882581551417,
  0.0000000000000000,  4.7559398375548234,  0.0000000000000000,  4.2869462368314428,
  0.0000000000000000,  3.4643757195950422,  0.0000000000000000,  2.1867799805244892,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -4.7613290062385216,  0.0000000000000000, -9.0376762100499786,
  0.0000000000000000,-12.1253150185826488,  0.0000000000000000,-13.7292157273097004,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,-18.5124707101607555,  0.0000000000000000,-23.7796991877740886,
  0.0000000000000000,-17.1477849473257713,  0.0000000000000000, -8.0835433457217647,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, 10.8452114520599725,  0.0000000000000000, 24.2506300371653261,
  0.0000000000000000, 32.9501177455433591,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000, 22.2149648521929066,
  0.0000000000000000, 25.3650124669590298,  0.0000000000000000, 11.4318566315505308,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -9.2383352522534725,  0.0000000000000000,-18.8286387117390213,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000, -6.3471328149122588,  0.0000000000000000, -5.4353598143483648,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  2.0920709679710008,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.2820947917738776,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,  0.0000000000000000,
  0.0000000000000000,  0.0000000000000000,])
