"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RegionInfo = void 0;
const fact_1 = require("./fact");
/**
 * Information pertaining to an AWS region.
 */
class RegionInfo {
    constructor(name) {
        this.name = name;
    }
    /**
     * @returns the list of names of AWS regions for which there is at least one registered fact. This
     *          may not be an exaustive list of all available AWS regions.
     */
    static get regions() {
        return fact_1.Fact.regions.map(RegionInfo.get);
    }
    /**
     * Obtain region info for a given region name.
     *
     * @param name the name of the region (e.g: us-east-1)
     */
    static get(name) {
        return new RegionInfo(name);
    }
    /**
     * Whether the `AWS::CDK::Metadata` CloudFormation Resource is available in this region or not.
     */
    get cdkMetadataResourceAvailable() {
        return fact_1.Fact.find(this.name, fact_1.FactName.CDK_METADATA_RESOURCE_AVAILABLE) === 'YES';
    }
    /**
     * The domain name suffix (e.g: amazonaws.com) for this region.
     */
    get domainSuffix() {
        return fact_1.Fact.find(this.name, fact_1.FactName.DOMAIN_SUFFIX);
    }
    /**
     * The name of the ARN partition for this region (e.g: aws).
     */
    get partition() {
        return fact_1.Fact.find(this.name, fact_1.FactName.PARTITION);
    }
    /**
     * The endpoint used by S3 static website hosting in this region (e.g: s3-static-website-us-east-1.amazonaws.com)
     */
    get s3StaticWebsiteEndpoint() {
        return fact_1.Fact.find(this.name, fact_1.FactName.S3_STATIC_WEBSITE_ENDPOINT);
    }
    /**
     * The hosted zone ID used by Route 53 to alias a S3 static website in this region (e.g: Z2O1EMRO9K5GLX)
     */
    get s3StaticWebsiteHostedZoneId() {
        return fact_1.Fact.find(this.name, fact_1.FactName.S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID);
    }
    /**
     * The prefix for VPC Endpoint Service names,
     * cn.com.amazonaws.vpce for China regions,
     * com.amazonaws.vpce otherwise.
     */
    get vpcEndpointServiceNamePrefix() {
        return fact_1.Fact.find(this.name, fact_1.FactName.VPC_ENDPOINT_SERVICE_NAME_PREFIX);
    }
    /**
     * The name of the service principal for a given service in this region.
     * @param service the service name (e.g: s3.amazonaws.com)
     */
    servicePrincipal(service) {
        return fact_1.Fact.find(this.name, fact_1.FactName.servicePrincipal(service));
    }
    /**
     * The account ID for ELBv2 in this region
     *
     */
    get elbv2Account() {
        return fact_1.Fact.find(this.name, fact_1.FactName.ELBV2_ACCOUNT);
    }
}
exports.RegionInfo = RegionInfo;
//# sourceMappingURL=data:application/json;base64,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