"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FactName = exports.Fact = void 0;
/**
 * A database of regional information.
 */
class Fact {
    constructor() {
        throw new Error('Use the static methods of Fact instead!');
    }
    /**
     * @returns the list of names of AWS regions for which there is at least one registered fact. This
     *          may not be an exhaustive list of all available AWS regions.
     */
    static get regions() {
        return Object.keys(this.database);
    }
    /**
     * Retrieves a fact from this Fact database.
     *
     * @param region the name of the region (e.g: `us-east-1`)
     * @param name   the name of the fact being looked up (see the `FactName` class for details)
     *
     * @returns the fact value if it is known, and `undefined` otherwise.
     */
    static find(region, name) {
        const regionFacts = this.database[region];
        return regionFacts && regionFacts[name];
    }
    /**
     * Retrieve a fact from the Fact database. (retrieval will fail if the specified region or
     * fact name does not exist.)
     *
     * @param region the name of the region (e.g: `us-east-1`)
     * @param name the name of the fact being looked up (see the `FactName` class for details)
     */
    static requireFact(region, name) {
        const foundFact = this.find(region, name);
        if (!foundFact) {
            throw new Error(`No fact ${name} could be found for region: ${region} and name: ${name}`);
        }
        return foundFact;
    }
    /**
     * Registers a new fact in this Fact database.
     *
     * @param fact           the new fact to be registered.
     * @param allowReplacing whether new facts can replace existing facts or not.
     */
    static register(fact, allowReplacing = false) {
        const regionFacts = this.database[fact.region] || (this.database[fact.region] = {});
        if (fact.name in regionFacts && regionFacts[fact.name] !== fact.value && !allowReplacing) {
            throw new Error(`Region ${fact.region} already has a fact ${fact.name}, with value ${regionFacts[fact.name]}`);
        }
        if (fact.value !== undefined) {
            regionFacts[fact.name] = fact.value;
        }
    }
    /**
     * Removes a fact from the database.
     * @param region the region for which the fact is to be removed.
     * @param name   the name of the fact to remove.
     * @param value  the value that should be removed (removal will fail if the value is specified, but does not match the
     *               current stored value).
     */
    static unregister(region, name, value) {
        const regionFacts = this.database[region] || {};
        if (name in regionFacts && value && regionFacts[name] !== value) {
            throw new Error(`Attempted to remove ${name} from ${region} with value ${value}, but the fact's value is ${regionFacts[name]}`);
        }
        delete regionFacts[name];
    }
}
exports.Fact = Fact;
Fact.database = {};
/**
 * All standardized fact names.
 */
class FactName {
    /**
     * The name of the regional service principal for a given service.
     *
     * @param service the service name, either simple (e.g: `s3`, `codedeploy`) or qualified (e.g: `s3.amazonaws.com`).
     *                The `.amazonaws.com` and `.amazonaws.com.cn` domains are stripped from service names, so they are
     *                canonicalized in that respect.
     */
    static servicePrincipal(service) {
        return `service-principal:${service.replace(/\.amazonaws\.com(\.cn)?$/, '')}`;
    }
}
exports.FactName = FactName;
/**
 * The name of the partition for a region (e.g: 'aws', 'aws-cn', ...)
 */
FactName.PARTITION = 'partition';
/**
 * The domain suffix for a region (e.g: 'amazonaws.com`)
 */
FactName.DOMAIN_SUFFIX = 'domainSuffix';
/**
 * Whether the AWS::CDK::Metadata CloudFormation Resource is available in-region or not. The value is a boolean
 * modelled as `YES` or `NO`.
 */
FactName.CDK_METADATA_RESOURCE_AVAILABLE = 'cdk:metadata-resource:available';
/**
 * The endpoint used for hosting S3 static websites
 */
FactName.S3_STATIC_WEBSITE_ENDPOINT = 's3-static-website:endpoint';
/**
 * The endpoint used for aliasing S3 static websites in Route 53
 */
FactName.S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID = 's3-static-website:route-53-hosted-zone-id';
/**
 * The prefix for VPC Endpoint Service names,
 * cn.com.amazonaws.vpce for China regions,
 * com.amazonaws.vpce otherwise.
 */
FactName.VPC_ENDPOINT_SERVICE_NAME_PREFIX = 'vpcEndpointServiceNamePrefix';
/**
 * The account for ELBv2 in this region
 */
FactName.ELBV2_ACCOUNT = 'elbv2Account';
//# sourceMappingURL=data:application/json;base64,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