"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ShellScriptAction = void 0;
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const codepipeline_actions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
/**
 * Validate a revision using shell commands
 */
class ShellScriptAction {
    constructor(props) {
        var _a, _b;
        this.props = props;
        // A number of actionProperties get read before bind() is even called (so before we
        // have made the Project and can construct the actual CodeBuildAction)
        //
        // - actionName
        // - resource
        // - region
        // - category
        // - role
        // - owner
        this._actionProperties = {
            actionName: props.actionName,
            category: codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 0, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [],
            outputs: [],
        };
        if (Object.keys((_a = props.useOutputs) !== null && _a !== void 0 ? _a : {}).length + ((_b = props.additionalArtifacts) !== null && _b !== void 0 ? _b : []).length === 0) {
            throw new Error('You must supply either \'useOutputs\' or \'additionalArtifacts\', since a CodeBuild Action must always have at least one input artifact.');
        }
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        return this._actionProperties;
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        var _a, _b, _c, _d;
        const inputs = new Array();
        inputs.push(...(_a = this.props.additionalArtifacts) !== null && _a !== void 0 ? _a : []);
        const envVarCommands = new Array();
        const bashOptions = (_b = this.props.bashOptions) !== null && _b !== void 0 ? _b : '-eu';
        if (bashOptions) {
            envVarCommands.push(`set ${bashOptions}`);
        }
        for (const [varName, output] of Object.entries((_c = this.props.useOutputs) !== null && _c !== void 0 ? _c : {})) {
            const outputArtifact = output.artifactFile;
            // Add the artifact to the list of inputs, if it's not in there already. Determine
            // the location where CodeBuild is going to stick it based on whether it's the first (primary)
            // input or an 'extra input', then parse.
            let artifactIndex = inputs.findIndex(a => a.artifactName === outputArtifact.artifact.artifactName);
            if (artifactIndex === -1) {
                artifactIndex = inputs.push(outputArtifact.artifact) - 1;
            }
            const dirEnv = artifactIndex === 0 ? 'CODEBUILD_SRC_DIR' : `CODEBUILD_SRC_DIR_${outputArtifact.artifact.artifactName}`;
            envVarCommands.push(`export ${varName}="$(node -pe 'require(process.env.${dirEnv} + "/${outputArtifact.fileName}")["${output.outputName}"]')"`);
        }
        this._project = new codebuild.PipelineProject(scope, 'Project', {
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    build: {
                        commands: [
                            ...envVarCommands,
                            ...this.props.commands,
                        ],
                    },
                },
            }),
        });
        this._action = new codepipeline_actions.CodeBuildAction({
            actionName: this.props.actionName,
            input: inputs[0],
            extraInputs: inputs.slice(1),
            runOrder: (_d = this.props.runOrder) !== null && _d !== void 0 ? _d : 100,
            project: this._project,
        });
        // Replace the placeholder actionProperties at the last minute
        this._actionProperties = this._action.actionProperties;
        return this._action.bind(scope, stage, options);
    }
    /**
     * Project generated to run the shell script in
     */
    get project() {
        if (!this._project) {
            throw new Error('Project becomes available after ShellScriptAction has been bound to a stage');
        }
        return this._project;
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        if (!this._action) {
            throw new Error('Need bind() first');
        }
        return this._action.onStateChange(name, target, options);
    }
}
exports.ShellScriptAction = ShellScriptAction;
//# sourceMappingURL=data:application/json;base64,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