"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StackOutput = exports.CdkStage = void 0;
const cpactions = require("../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const actions_1 = require("./actions");
const asset_manifest_1 = require("./private/asset-manifest");
const toposort_1 = require("./private/toposort");
/**
 * Stage in a CdkPipeline
 *
 * You don't need to instantiate this class directly. Use
 * `cdkPipeline.addStage()` instead.
 */
class CdkStage extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this._nextSequentialRunOrder = 1; // Must start at 1 eh
        this._manualApprovalCounter = 1;
        this.stacksToDeploy = new Array();
        this._prepared = false;
        this.stageName = props.stageName;
        this.pipelineStage = props.pipelineStage;
        this.cloudAssemblyArtifact = props.cloudAssemblyArtifact;
        this.host = props.host;
    }
    /**
     * Add all stacks in the application Stage to this stage
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     */
    addApplication(appStage, options = {}) {
        const asm = appStage.synth();
        const sortedTranches = toposort_1.topologicalSort(asm.stacks, stack => stack.id, stack => stack.dependencies.map(d => d.id));
        for (const stacks of sortedTranches) {
            const runOrder = this.nextSequentialRunOrder(2); // We need 2 actions
            let executeRunOrder = runOrder + 1;
            // If we need to insert a manual approval action, then what's the executeRunOrder
            // now is where we add a manual approval step, and we allocate 1 more runOrder
            // for the execute.
            if (options.manualApprovals) {
                this.addManualApprovalAction({ runOrder: executeRunOrder });
                executeRunOrder = this.nextSequentialRunOrder();
            }
            // These don't have a dependency on each other, so can all be added in parallel
            for (const stack of stacks) {
                this.addStackArtifactDeployment(stack, { runOrder, executeRunOrder });
            }
        }
    }
    /**
     * Add a deployment action based on a stack artifact
     */
    addStackArtifactDeployment(stackArtifact, options = {}) {
        var _a, _b;
        // Get all assets manifests and add the assets in 'em to the asset publishing stage.
        this.publishAssetDependencies(stackArtifact);
        // Remember for later, see 'prepare()'
        // We know that deploying a stack is going to take up 2 runorder slots later on.
        const runOrder = (_a = options.runOrder) !== null && _a !== void 0 ? _a : this.nextSequentialRunOrder(2);
        const executeRunOrder = (_b = options.executeRunOrder) !== null && _b !== void 0 ? _b : runOrder + 1;
        this.stacksToDeploy.push({
            prepareRunOrder: runOrder,
            executeRunOrder,
            stackArtifact,
        });
        this.advanceRunOrderPast(runOrder);
        this.advanceRunOrderPast(executeRunOrder);
    }
    /**
     * Add a manual approval action
     *
     * If you need more flexibility than what this method offers,
     * use `addAction` with a `ManualApprovalAction`.
     */
    addManualApprovalAction(options = {}) {
        var _a;
        let actionName = options.actionName;
        if (!actionName) {
            actionName = `ManualApproval${this._manualApprovalCounter > 1 ? this._manualApprovalCounter : ''}`;
            this._manualApprovalCounter += 1;
        }
        this.addActions(new cpactions.ManualApprovalAction({
            actionName,
            runOrder: (_a = options.runOrder) !== null && _a !== void 0 ? _a : this.nextSequentialRunOrder(),
        }));
    }
    /**
     * Add one or more CodePipeline Actions
     *
     * You need to make sure it is created with the right runOrder. Call `nextSequentialRunOrder()`
     * for every action to get actions to execute in sequence.
     */
    addActions(...actions) {
        for (const action of actions) {
            this.pipelineStage.addAction(action);
        }
    }
    /**
     * Return the runOrder number necessary to run the next Action in sequence with the rest
     *
     * FIXME: This is here because Actions are immutable and can't be reordered
     * after creation, nor is there a way to specify relative priorities, which
     * is a limitation that we should take away in the base library.
     */
    nextSequentialRunOrder(count = 1) {
        const ret = this._nextSequentialRunOrder;
        this._nextSequentialRunOrder += count;
        return ret;
    }
    /**
     * Whether this Stage contains an action to deploy the given stack, identified by its artifact ID
     */
    deploysStack(artifactId) {
        return this.stacksToDeploy.map(s => s.stackArtifact.id).includes(artifactId);
    }
    /**
     * Actually add all the DeployStack actions to the stage.
     *
     * We do this late because before we can render the actual DeployActions,
     * we need to know whether or not we need to capture the stack outputs.
     *
     * FIXME: This is here because Actions are immutable and can't be reordered
     * after creation, nor is there a way to specify relative priorities, which
     * is a limitation that we should take away in the base library.
     */
    prepare() {
        // FIXME: Make sure this only gets run once. There seems to be an issue in the reconciliation
        // loop that may trigger this more than once if it throws an error somewhere, and the exception
        // that gets thrown here will then override the actual failure.
        if (this._prepared) {
            return;
        }
        this._prepared = true;
        for (const { prepareRunOrder: runOrder, stackArtifact } of this.stacksToDeploy) {
            const artifact = this.host.stackOutputArtifact(stackArtifact.id);
            this.pipelineStage.addAction(actions_1.DeployCdkStackAction.fromStackArtifact(this, stackArtifact, {
                baseActionName: this.simplifyStackName(stackArtifact.stackName),
                cloudAssemblyInput: this.cloudAssemblyArtifact,
                output: artifact,
                outputFileName: artifact ? 'outputs.json' : undefined,
                prepareRunOrder: runOrder,
            }));
        }
    }
    /**
     * Advance the runorder counter so that the next sequential number is higher than the given one
     */
    advanceRunOrderPast(lastUsed) {
        this._nextSequentialRunOrder = Math.max(lastUsed + 1, this._nextSequentialRunOrder);
    }
    /**
     * Simplify the stack name by removing the `Stage-` prefix if it exists.
     */
    simplifyStackName(s) {
        return stripPrefix(s, `${this.stageName}-`);
    }
    /**
     * Make sure all assets depended on by this stack are published in this pipeline
     *
     * Taking care to exclude the stack template itself -- it is being published
     * as an asset because the CLI needs to know the asset publishing role when
     * pushing the template to S3, but in the case of CodePipeline we always
     * reference the template from the artifact bucket.
     *
     * (NOTE: this is only true for top-level stacks, not nested stacks. Nested
     * Stack templates are always published as assets).
     */
    publishAssetDependencies(stackArtifact) {
        const assetManifests = stackArtifact.dependencies.filter(isAssetManifest);
        for (const manifestArtifact of assetManifests) {
            const manifest = asset_manifest_1.AssetManifestReader.fromFile(manifestArtifact.file);
            for (const entry of manifest.entries) {
                let assetType;
                if (entry instanceof asset_manifest_1.DockerImageManifestEntry) {
                    assetType = actions_1.AssetType.DOCKER_IMAGE;
                }
                else if (entry instanceof asset_manifest_1.FileManifestEntry) {
                    // Don't publishg the template for this stack
                    if (entry.source.packaging === 'file' && entry.source.path === stackArtifact.templateFile) {
                        continue;
                    }
                    assetType = actions_1.AssetType.FILE;
                }
                else {
                    throw new Error(`Unrecognized asset type: ${entry.type}`);
                }
                this.host.publishAsset({
                    assetManifestPath: manifestArtifact.file,
                    assetId: entry.id.assetId,
                    assetSelector: entry.id.toString(),
                    assetType,
                });
            }
        }
    }
}
exports.CdkStage = CdkStage;
/**
 * A single output of a Stack
 */
class StackOutput {
    /**
     * Build a StackOutput from a known artifact and an output name
     */
    constructor(artifactFile, outputName) {
        this.artifactFile = artifactFile;
        this.outputName = outputName;
    }
}
exports.StackOutput = StackOutput;
function stripPrefix(s, prefix) {
    return s.startsWith(prefix) ? s.substr(prefix.length) : s;
}
function isAssetManifest(s) {
    // instanceof is too risky, and we're at a too late stage to properly fix.
    // return s instanceof cxapi.AssetManifestArtifact;
    return s.constructor.name === 'AssetManifestArtifact';
}
//# sourceMappingURL=data:application/json;base64,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