"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.topologicalSort = void 0;
/**
 * Return a topological sort of all elements of xs, according to the given dependency functions
 *
 * Dependencies outside the referenced set are ignored.
 *
 * Not a stable sort, but in order to keep the order as stable as possible, we'll sort by key
 * among elements of equal precedence.
 *
 * Returns tranches of elements of equal precedence.
 */
function topologicalSort(xs, keyFn, depFn) {
    const remaining = new Map();
    for (const element of xs) {
        const key = keyFn(element);
        remaining.set(key, { key, element, dependencies: depFn(element) });
    }
    const ret = new Array();
    while (remaining.size > 0) {
        // All elements with no more deps in the set can be ordered
        const selectable = Array.from(remaining.values()).filter(e => e.dependencies.every(d => !remaining.has(d)));
        selectable.sort((a, b) => a.key < b.key ? -1 : b.key < a.key ? 1 : 0);
        // If we didn't make any progress, we got stuck
        if (selectable.length === 0) {
            throw new Error(`Could not determine ordering between: ${Array.from(remaining.keys()).join(', ')}`);
        }
        ret.push(selectable.map(s => s.element));
        for (const selected of selectable) {
            remaining.delete(selected.key);
        }
    }
    return ret;
}
exports.topologicalSort = topologicalSort;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidG9wb3NvcnQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0b3Bvc29ydC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFFQTs7Ozs7Ozs7O0dBU0c7QUFDSCxTQUFnQixlQUFlLENBQUksRUFBZSxFQUFFLEtBQWlCLEVBQUUsS0FBaUI7SUFDcEYsTUFBTSxTQUFTLEdBQUcsSUFBSSxHQUFHLEVBQTBCLENBQUM7SUFDcEQsS0FBSyxNQUFNLE9BQU8sSUFBSSxFQUFFLEVBQUU7UUFDdEIsTUFBTSxHQUFHLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQzNCLFNBQVMsQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFLEVBQUUsR0FBRyxFQUFFLE9BQU8sRUFBRSxZQUFZLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsQ0FBQztLQUN0RTtJQUNELE1BQU0sR0FBRyxHQUFHLElBQUksS0FBSyxFQUFPLENBQUM7SUFDN0IsT0FBTyxTQUFTLENBQUMsSUFBSSxHQUFHLENBQUMsRUFBRTtRQUN2QiwyREFBMkQ7UUFDM0QsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDNUcsVUFBVSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxHQUFHLEdBQUcsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLEdBQUcsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN0RSwrQ0FBK0M7UUFDL0MsSUFBSSxVQUFVLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUN6QixNQUFNLElBQUksS0FBSyxDQUFDLHlDQUF5QyxLQUFLLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7U0FDdkc7UUFDRCxHQUFHLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztRQUN6QyxLQUFLLE1BQU0sUUFBUSxJQUFJLFVBQVUsRUFBRTtZQUMvQixTQUFTLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUNsQztLQUNKO0lBQ0QsT0FBTyxHQUFHLENBQUM7QUFDZixDQUFDO0FBckJELDBDQXFCQyIsInNvdXJjZXNDb250ZW50IjpbImV4cG9ydCB0eXBlIEtleUZ1bmM8VD4gPSAoeDogVCkgPT4gc3RyaW5nO1xuZXhwb3J0IHR5cGUgRGVwRnVuYzxUPiA9ICh4OiBUKSA9PiBzdHJpbmdbXTtcbi8qKlxuICogUmV0dXJuIGEgdG9wb2xvZ2ljYWwgc29ydCBvZiBhbGwgZWxlbWVudHMgb2YgeHMsIGFjY29yZGluZyB0byB0aGUgZ2l2ZW4gZGVwZW5kZW5jeSBmdW5jdGlvbnNcbiAqXG4gKiBEZXBlbmRlbmNpZXMgb3V0c2lkZSB0aGUgcmVmZXJlbmNlZCBzZXQgYXJlIGlnbm9yZWQuXG4gKlxuICogTm90IGEgc3RhYmxlIHNvcnQsIGJ1dCBpbiBvcmRlciB0byBrZWVwIHRoZSBvcmRlciBhcyBzdGFibGUgYXMgcG9zc2libGUsIHdlJ2xsIHNvcnQgYnkga2V5XG4gKiBhbW9uZyBlbGVtZW50cyBvZiBlcXVhbCBwcmVjZWRlbmNlLlxuICpcbiAqIFJldHVybnMgdHJhbmNoZXMgb2YgZWxlbWVudHMgb2YgZXF1YWwgcHJlY2VkZW5jZS5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIHRvcG9sb2dpY2FsU29ydDxUPih4czogSXRlcmFibGU8VD4sIGtleUZuOiBLZXlGdW5jPFQ+LCBkZXBGbjogRGVwRnVuYzxUPik6IFRbXVtdIHtcbiAgICBjb25zdCByZW1haW5pbmcgPSBuZXcgTWFwPHN0cmluZywgVG9wb0VsZW1lbnQ8VD4+KCk7XG4gICAgZm9yIChjb25zdCBlbGVtZW50IG9mIHhzKSB7XG4gICAgICAgIGNvbnN0IGtleSA9IGtleUZuKGVsZW1lbnQpO1xuICAgICAgICByZW1haW5pbmcuc2V0KGtleSwgeyBrZXksIGVsZW1lbnQsIGRlcGVuZGVuY2llczogZGVwRm4oZWxlbWVudCkgfSk7XG4gICAgfVxuICAgIGNvbnN0IHJldCA9IG5ldyBBcnJheTxUW10+KCk7XG4gICAgd2hpbGUgKHJlbWFpbmluZy5zaXplID4gMCkge1xuICAgICAgICAvLyBBbGwgZWxlbWVudHMgd2l0aCBubyBtb3JlIGRlcHMgaW4gdGhlIHNldCBjYW4gYmUgb3JkZXJlZFxuICAgICAgICBjb25zdCBzZWxlY3RhYmxlID0gQXJyYXkuZnJvbShyZW1haW5pbmcudmFsdWVzKCkpLmZpbHRlcihlID0+IGUuZGVwZW5kZW5jaWVzLmV2ZXJ5KGQgPT4gIXJlbWFpbmluZy5oYXMoZCkpKTtcbiAgICAgICAgc2VsZWN0YWJsZS5zb3J0KChhLCBiKSA9PiBhLmtleSA8IGIua2V5ID8gLTEgOiBiLmtleSA8IGEua2V5ID8gMSA6IDApO1xuICAgICAgICAvLyBJZiB3ZSBkaWRuJ3QgbWFrZSBhbnkgcHJvZ3Jlc3MsIHdlIGdvdCBzdHVja1xuICAgICAgICBpZiAoc2VsZWN0YWJsZS5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgQ291bGQgbm90IGRldGVybWluZSBvcmRlcmluZyBiZXR3ZWVuOiAke0FycmF5LmZyb20ocmVtYWluaW5nLmtleXMoKSkuam9pbignLCAnKX1gKTtcbiAgICAgICAgfVxuICAgICAgICByZXQucHVzaChzZWxlY3RhYmxlLm1hcChzID0+IHMuZWxlbWVudCkpO1xuICAgICAgICBmb3IgKGNvbnN0IHNlbGVjdGVkIG9mIHNlbGVjdGFibGUpIHtcbiAgICAgICAgICAgIHJlbWFpbmluZy5kZWxldGUoc2VsZWN0ZWQua2V5KTtcbiAgICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4gcmV0O1xufVxuaW50ZXJmYWNlIFRvcG9FbGVtZW50PFQ+IHtcbiAgICBrZXk6IHN0cmluZztcbiAgICBkZXBlbmRlbmNpZXM6IHN0cmluZ1tdO1xuICAgIGVsZW1lbnQ6IFQ7XG59XG4iXX0=