"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkPipeline = void 0;
const codepipeline = require("../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const path = require("path");
const actions_1 = require("./actions");
const construct_internals_1 = require("./private/construct-internals");
const stage_1 = require("./stage");
/**
 * A Pipeline to deploy CDK apps
 *
 * Defines an AWS CodePipeline-based Pipeline to deploy CDK applications.
 *
 * Automatically manages the following:
 *
 * - Stack dependency order.
 * - Asset publishing.
 * - Keeping the pipeline up-to-date as the CDK apps change.
 * - Using stack outputs later on in the pipeline.
 */
class CdkPipeline extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this._stages = [];
        this._outputArtifacts = {};
        if (!core_1.App.isApp(this.node.root)) {
            throw new Error('CdkPipeline must be created under an App');
        }
        this._cloudAssemblyArtifact = props.cloudAssemblyArtifact;
        const pipelineStack = core_1.Stack.of(this);
        this._pipeline = new codepipeline.Pipeline(this, 'Pipeline', {
            ...props,
            restartExecutionOnUpdate: true,
            stages: [
                {
                    stageName: 'Source',
                    actions: [props.sourceAction],
                },
                {
                    stageName: 'Build',
                    actions: [props.synthAction],
                },
                {
                    stageName: 'UpdatePipeline',
                    actions: [new actions_1.UpdatePipelineAction(this, 'UpdatePipeline', {
                            cloudAssemblyInput: this._cloudAssemblyArtifact,
                            pipelineStackName: pipelineStack.stackName,
                            cdkCliVersion: props.cdkCliVersion,
                            projectName: maybeSuffix(props.pipelineName, '-selfupdate'),
                        })],
                },
            ],
        });
        this._assets = new AssetPublishing(this, 'Assets', {
            cloudAssemblyInput: this._cloudAssemblyArtifact,
            cdkCliVersion: props.cdkCliVersion,
            pipeline: this._pipeline,
            projectName: maybeSuffix(props.pipelineName, '-publish'),
        });
    }
    /**
     * Add pipeline stage that will deploy the given application stage
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     */
    addApplicationStage(appStage, options = {}) {
        const stage = this.addStage(appStage.stageName);
        stage.addApplication(appStage, options);
        return stage;
    }
    /**
     * Add a new, empty stage to the pipeline
     *
     * Prefer to use `addApplicationStage` if you are intended to deploy a CDK
     * application, but you can use this method if you want to add other kinds of
     * Actions to a pipeline.
     */
    addStage(stageName) {
        const pipelineStage = this._pipeline.addStage({
            stageName,
        });
        const stage = new stage_1.CdkStage(this, stageName, {
            cloudAssemblyArtifact: this._cloudAssemblyArtifact,
            pipelineStage,
            stageName,
            host: {
                publishAsset: this._assets.addPublishAssetAction.bind(this._assets),
                stackOutputArtifact: (artifactId) => this._outputArtifacts[artifactId],
            },
        });
        this._stages.push(stage);
        return stage;
    }
    /**
     * Get the StackOutput object that holds this CfnOutput's value in this pipeline
     *
     * `StackOutput` can be used in validation actions later in the pipeline.
     */
    stackOutput(cfnOutput) {
        const stack = core_1.Stack.of(cfnOutput);
        if (!this._outputArtifacts[stack.artifactId]) {
            // We should have stored the ArtifactPath in the map, but its Artifact
            // property isn't publicly readable...
            this._outputArtifacts[stack.artifactId] = new codepipeline.Artifact(`Artifact_${stack.artifactId}_Outputs`);
        }
        return new stage_1.StackOutput(this._outputArtifacts[stack.artifactId].atPath('outputs.json'), cfnOutput.logicalId);
    }
    /**
     * Validate that we don't have any stacks violating dependency order in the pipeline
     *
     * Our own convenience methods will never generate a pipeline that does that (although
     * this is a nice verification), but a user can also add the stacks by hand.
     */
    validate() {
        const ret = new Array();
        ret.push(...this.validateDeployOrder());
        ret.push(...this.validateRequestedOutputs());
        return ret;
    }
    onPrepare() {
        super.onPrepare();
        // TODO: Support this in a proper way in the upstream library. For now, we
        // "un-add" the Assets stage if it turns out to be empty.
        this._assets.removeAssetsStageIfEmpty();
    }
    /**
     * Return all StackDeployActions in an ordered list
     */
    get stackActions() {
        return flatMap(this._pipeline.stages, s => s.actions.filter(isDeployAction));
    }
    *validateDeployOrder() {
        const stackActions = this.stackActions;
        for (const stackAction of stackActions) {
            // For every dependency, it must be executed in an action before this one is prepared.
            for (const depId of stackAction.dependencyStackArtifactIds) {
                const depAction = stackActions.find(s => s.stackArtifactId === depId);
                if (depAction === undefined) {
                    this.node.addWarning(`Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depId}', but that dependency is not deployed through the pipeline!`);
                }
                else if (!(depAction.executeRunOrder < stackAction.prepareRunOrder)) {
                    yield `Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depAction.stackName}', but is deployed before it in the pipeline!`;
                }
            }
        }
    }
    *validateRequestedOutputs() {
        const artifactIds = this.stackActions.map(s => s.stackArtifactId);
        for (const artifactId of Object.keys(this._outputArtifacts)) {
            if (!artifactIds.includes(artifactId)) {
                yield `Trying to use outputs for Stack '${artifactId}', but Stack is not deployed in this pipeline. Add it to the pipeline.`;
            }
        }
    }
}
exports.CdkPipeline = CdkPipeline;
function isDeployAction(a) {
    return a instanceof actions_1.DeployCdkStackAction;
}
function flatMap(xs, f) {
    return Array.prototype.concat([], ...xs.map(f));
}
/**
 * Add appropriate publishing actions to the asset publishing stage
 */
class AssetPublishing extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.publishers = {};
        this._fileAssetCtr = 1;
        this._dockerAssetCtr = 1;
        this.myCxAsmRoot = path.resolve(construct_internals_1.assemblyBuilderOf(construct_internals_1.appOf(this)).outdir);
        // We MUST add the Stage immediately here, otherwise it will be in the wrong place
        // in the pipeline!
        this.stage = this.props.pipeline.addStage({ stageName: 'Assets' });
    }
    /**
     * Make sure there is an action in the stage to publish the given asset
     *
     * Assets are grouped by asset ID (which represent individual assets) so all assets
     * are published in parallel. For each assets, all destinations are published sequentially
     * so that we can reuse expensive operations between them (mostly: building a Docker image).
     */
    addPublishAssetAction(command) {
        // FIXME: this is silly, we need the relative path here but no easy way to get it
        const relativePath = path.relative(this.myCxAsmRoot, command.assetManifestPath);
        let action = this.publishers[command.assetId];
        if (!action) {
            // The asset ID would be a logical candidate for the construct path and project names, but if the asset
            // changes it leads to recreation of a number of Role/Policy/Project resources which is slower than
            // necessary. Number sequentially instead.
            //
            // FIXME: The ultimate best solution is probably to generate a single Project per asset type
            // and reuse that for all assets.
            const id = command.assetType === actions_1.AssetType.FILE ? `FileAsset${this._fileAssetCtr++}` : `DockerAsset${this._dockerAssetCtr++}`;
            // NOTE: It's important that asset changes don't force a pipeline self-mutation.
            // This can cause an infinite loop of updates (see https://github.com/aws/aws-cdk/issues/9080).
            // For that reason, we use the id as the actionName below, rather than the asset hash.
            action = this.publishers[command.assetId] = new actions_1.PublishAssetsAction(this, id, {
                actionName: id,
                cloudAssemblyInput: this.props.cloudAssemblyInput,
                cdkCliVersion: this.props.cdkCliVersion,
                assetType: command.assetType,
            });
            this.stage.addAction(action);
        }
        action.addPublishCommand(relativePath, command.assetSelector);
    }
    /**
     * Remove the Assets stage if it turns out we didn't add any Assets to publish
     */
    removeAssetsStageIfEmpty() {
        if (Object.keys(this.publishers).length === 0) {
            // Hacks to get access to innards of Pipeline
            // Modify 'stages' array in-place to remove Assets stage if empty
            const stages = this.props.pipeline._stages;
            const ix = stages.indexOf(this.stage);
            if (ix > -1) {
                stages.splice(ix, 1);
            }
        }
    }
}
function maybeSuffix(x, suffix) {
    if (x === undefined) {
        return undefined;
    }
    return `${x}${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,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