"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UpdatePipelineAction = void 0;
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const cpactions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const construct_internals_1 = require("../private/construct-internals");
/**
 * Action to self-mutate the pipeline
 *
 * Creates a CodeBuild project which will use the CDK CLI
 * to deploy the pipeline stack.
 *
 * You do not need to instantiate this action -- it will automatically
 * be added by the pipeline.
 */
class UpdatePipelineAction extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const installSuffix = props.cdkCliVersion ? `@${props.cdkCliVersion}` : '';
        const selfMutationProject = new codebuild.PipelineProject(this, 'SelfMutation', {
            projectName: props.projectName,
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        commands: `npm install -g aws-cdk${installSuffix}`,
                    },
                    build: {
                        commands: [
                            // Cloud Assembly is in *current* directory.
                            `cdk -a ${construct_internals_1.embeddedAsmPath(scope)} deploy ${props.pipelineStackName} --require-approval=never --verbose`,
                        ],
                    },
                },
            }),
        });
        // allow the self-mutating project permissions to assume the bootstrap Action role
        selfMutationProject.addToRolePolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: ['arn:*:iam::*:role/*-deploy-role-*', 'arn:*:iam::*:role/*-publishing-role-*'],
        }));
        selfMutationProject.addToRolePolicy(new iam.PolicyStatement({
            actions: ['cloudformation:DescribeStacks'],
            resources: ['*'],
        }));
        // S3 checks for the presence of the ListBucket permission
        selfMutationProject.addToRolePolicy(new iam.PolicyStatement({
            actions: ['s3:ListBucket'],
            resources: ['*'],
        }));
        this.action = new cpactions.CodeBuildAction({
            actionName: 'SelfMutate',
            input: props.cloudAssemblyInput,
            project: selfMutationProject,
        });
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        return this.action.bind(scope, stage, options);
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        return this.action.onStateChange(name, target, options);
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        // FIXME: I have had to make this class a Construct, because:
        //
        // - It needs access to the Construct tree, because it is going to add a `PipelineProject`.
        // - I would have liked to have done that in bind(), however,
        // - `actionProperties` (this method) is called BEFORE bind() is called, and by that point I
        //   don't have the "inner" Action yet to forward the call to.
        //
        // I've therefore had to construct the inner CodeBuildAction in the constructor, which requires making this
        // Action a Construct.
        //
        // Combined with how non-intuitive it is to make the "StackDeployAction", I feel there is something
        // wrong with the Action abstraction here.
        return this.action.actionProperties;
    }
}
exports.UpdatePipelineAction = UpdatePipelineAction;
//# sourceMappingURL=data:application/json;base64,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