"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.topologicalSort = void 0;
/**
 * Return a topological sort of all elements of xs, according to the given dependency functions
 *
 * Dependencies outside the referenced set are ignored.
 *
 * Not a stable sort, but in order to keep the order as stable as possible, we'll sort by key
 * among elements of equal precedence.
 */
function topologicalSort(xs, keyFn, depFn) {
    const remaining = new Map();
    for (const element of xs) {
        const key = keyFn(element);
        remaining.set(key, { key, element, dependencies: depFn(element) });
    }
    const ret = new Array();
    while (remaining.size > 0) {
        // All elements with no more deps in the set can be ordered
        const selectable = Array.from(remaining.values()).filter(e => e.dependencies.every(d => !remaining.has(d)));
        selectable.sort((a, b) => a.key < b.key ? -1 : b.key < a.key ? 1 : 0);
        for (const selected of selectable) {
            ret.push(selected.element);
            remaining.delete(selected.key);
        }
        // If we didn't make any progress, we got stuck
        if (selectable.length === 0) {
            throw new Error(`Could not determine ordering between: ${Array.from(remaining.keys()).join(', ')}`);
        }
    }
    return ret;
}
exports.topologicalSort = topologicalSort;
//# sourceMappingURL=data:application/json;base64,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