"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsCustomResource = exports.AwsCustomResourcePolicy = exports.PhysicalResourceId = void 0;
const fs = require("fs");
const path = require("path");
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
// don't use "require" since the typescript compiler emits errors since this
// file is not listed in tsconfig.json.
const metadata = JSON.parse(fs.readFileSync(path.join(__dirname, 'sdk-api-metadata.json'), 'utf-8'));
const awsSdkMetadata = metadata;
/**
 * Physical ID of the custom resource.
 */
class PhysicalResourceId {
    /**
     * @param responsePath Path to a response data element to be used as the physical id.
     * @param id Literal string to be used as the physical id.
     */
    constructor(responsePath, id) {
        this.responsePath = responsePath;
        this.id = id;
    }
    /**
     * Extract the physical resource id from the path (dot notation) to the data in the API call response.
     */
    static fromResponse(responsePath) {
        return new PhysicalResourceId(responsePath, undefined);
    }
    /**
     * Explicit physical resource id.
     */
    static of(id) {
        return new PhysicalResourceId(undefined, id);
    }
}
exports.PhysicalResourceId = PhysicalResourceId;
/**
 * The IAM Policy that will be applied to the different calls.
 */
class AwsCustomResourcePolicy {
    /**
     * @param statements statements for explicit policy.
     * @param resources resources for auto-generated from SDK calls.
     */
    constructor(statements, resources) {
        this.statements = statements;
        this.resources = resources;
    }
    /**
     * Explicit IAM Policy Statements.
     *
     * @param statements the statements to propagate to the SDK calls.
     */
    static fromStatements(statements) {
        return new AwsCustomResourcePolicy(statements, undefined);
    }
    /**
     * Generate IAM Policy Statements from the configured SDK calls.
     *
     * Each SDK call with be translated to an IAM Policy Statement in the form of: `call.service:call.action` (e.g `s3:PutObject`).
     *
     * @param options options for the policy generation
     */
    static fromSdkCalls(options) {
        return new AwsCustomResourcePolicy([], options.resources);
    }
}
exports.AwsCustomResourcePolicy = AwsCustomResourcePolicy;
/**
 * Use this constant to configure access to any resource.
 */
AwsCustomResourcePolicy.ANY_RESOURCE = ['*'];
/**
 * Defines a custom resource that is materialized using specific AWS API calls.
 *
 * Use this to bridge any gap that might exist in the CloudFormation Coverage.
 * You can specify exactly which calls are invoked for the 'CREATE', 'UPDATE' and 'DELETE' life cycle events.
 *
 */
class AwsCustomResource extends cdk.Construct {
    // 'props' cannot be optional, even though all its properties are optional.
    // this is because at least one sdk call must be provided.
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        if (!props.onCreate && !props.onUpdate && !props.onDelete) {
            throw new Error('At least `onCreate`, `onUpdate` or `onDelete` must be specified.');
        }
        for (const call of [props.onCreate, props.onUpdate]) {
            if (call && !call.physicalResourceId) {
                throw new Error('`physicalResourceId` must be specified for onCreate and onUpdate calls.');
            }
        }
        for (const call of [props.onCreate, props.onUpdate, props.onDelete]) {
            if ((_a = call === null || call === void 0 ? void 0 : call.physicalResourceId) === null || _a === void 0 ? void 0 : _a.responsePath) {
                AwsCustomResource.breakIgnoreErrorsCircuit([call], 'PhysicalResourceId.fromResponse');
            }
        }
        this.props = props;
        const provider = new lambda.SingletonFunction(this, 'Provider', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'runtime')),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            uuid: '679f53fa-c002-430c-b0da-5b7982bd2287',
            lambdaPurpose: 'AWS',
            timeout: props.timeout || cdk.Duration.minutes(2),
            role: props.role,
            logRetention: props.logRetention,
        });
        this.grantPrincipal = provider.grantPrincipal;
        if (props.policy.statements.length !== 0) {
            // Use custom statements provided by the user
            for (const statement of props.policy.statements) {
                provider.addToRolePolicy(statement);
            }
        }
        else {
            // Derive statements from AWS SDK calls
            for (const call of [props.onCreate, props.onUpdate, props.onDelete]) {
                if (call) {
                    provider.addToRolePolicy(new iam.PolicyStatement({
                        actions: [awsSdkToIamAction(call.service, call.action)],
                        resources: props.policy.resources,
                    }));
                }
            }
        }
        const create = props.onCreate || props.onUpdate;
        this.customResource = new cdk.CustomResource(this, 'Resource', {
            resourceType: props.resourceType || 'Custom::AWS',
            serviceToken: provider.functionArn,
            pascalCaseProperties: true,
            properties: {
                create: create && encodeBooleans(create),
                update: props.onUpdate && encodeBooleans(props.onUpdate),
                delete: props.onDelete && encodeBooleans(props.onDelete),
            },
        });
    }
    static breakIgnoreErrorsCircuit(sdkCalls, caller) {
        for (const call of sdkCalls) {
            if (call === null || call === void 0 ? void 0 : call.ignoreErrorCodesMatching) {
                throw new Error(`\`${caller}\`` + ' cannot be called along with `ignoreErrorCodesMatching`.');
            }
        }
    }
    /**
     * Returns response data for the AWS SDK call.
     *
     * Example for S3 / listBucket : 'Buckets.0.Name'
     *
     * Use `Token.asXxx` to encode the returned `Reference` as a specific type or
     * use the convenience `getDataString` for string attributes.
     *
     * Note that you cannot use this method if `ignoreErrorCodesMatching`
     * is configured for any of the SDK calls. This is because in such a case,
     * the response data might not exist, and will cause a CloudFormation deploy time error.
     *
     * @param dataPath the path to the data
     */
    getResponseFieldReference(dataPath) {
        AwsCustomResource.breakIgnoreErrorsCircuit([this.props.onCreate, this.props.onUpdate], 'getData');
        return this.customResource.getAtt(dataPath);
    }
    /**
     * Returns response data for the AWS SDK call as string.
     *
     * Example for S3 / listBucket : 'Buckets.0.Name'
     *
     * Note that you cannot use this method if `ignoreErrorCodesMatching`
     * is configured for any of the SDK calls. This is because in such a case,
     * the response data might not exist, and will cause a CloudFormation deploy time error.
     *
     * @param dataPath the path to the data
     */
    getResponseField(dataPath) {
        AwsCustomResource.breakIgnoreErrorsCircuit([this.props.onCreate, this.props.onUpdate], 'getDataString');
        return this.customResource.getAttString(dataPath);
    }
}
exports.AwsCustomResource = AwsCustomResource;
/**
 * Transform SDK service/action to IAM action using metadata from aws-sdk module.
 * Example: CloudWatchLogs with putRetentionPolicy => logs:PutRetentionPolicy
 *
 * TODO: is this mapping correct for all services?
 */
function awsSdkToIamAction(service, action) {
    const srv = service.toLowerCase();
    const iamService = (awsSdkMetadata[srv] && awsSdkMetadata[srv].prefix) || srv;
    const iamAction = action.charAt(0).toUpperCase() + action.slice(1);
    return `${iamService}:${iamAction}`;
}
/**
 * Encodes booleans as special strings
 */
function encodeBooleans(object) {
    return JSON.parse(JSON.stringify(object), (_k, v) => {
        switch (v) {
            case true:
                return 'TRUE:BOOLEAN';
            case false:
                return 'FALSE:BOOLEAN';
            default:
                return v;
        }
    });
}
//# sourceMappingURL=data:application/json;base64,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