"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findLastCommonElement = exports.pathToTopLevelStack = exports.PostResolveToken = exports.filterUndefined = exports.ignoreEmpty = exports.capitalizePropertyNames = void 0;
const intrinsic_1 = require("./private/intrinsic");
const stack_1 = require("./stack");
/**
 * Given an object, converts all keys to PascalCase given they are currently in camel case.
 * @param obj The object.
 */
function capitalizePropertyNames(construct, obj) {
    const stack = stack_1.Stack.of(construct);
    obj = stack.resolve(obj);
    if (typeof (obj) !== 'object') {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(x => capitalizePropertyNames(construct, x));
    }
    const newObj = {};
    for (const key of Object.keys(obj)) {
        const value = obj[key];
        const first = key.charAt(0).toUpperCase();
        const newKey = first + key.slice(1);
        newObj[newKey] = capitalizePropertyNames(construct, value);
    }
    return newObj;
}
exports.capitalizePropertyNames = capitalizePropertyNames;
/**
 * Turns empty arrays/objects to undefined (after evaluating tokens).
 */
function ignoreEmpty(obj) {
    return new PostResolveToken(obj, o => {
        // undefined/null
        if (o == null) {
            return o;
        }
        if (Array.isArray(o) && o.length === 0) {
            return undefined;
        }
        if (typeof (o) === 'object' && Object.keys(o).length === 0) {
            return undefined;
        }
        return o;
    });
}
exports.ignoreEmpty = ignoreEmpty;
/**
 * Returns a copy of `obj` without `undefined` (or `null`) values in maps or arrays.
 */
function filterUndefined(obj) {
    if (Array.isArray(obj)) {
        return obj.filter(x => x != null).map(x => filterUndefined(x));
    }
    if (typeof (obj) === 'object') {
        const ret = {};
        for (const [key, value] of Object.entries(obj)) {
            if (value == null) {
                continue;
            }
            ret[key] = filterUndefined(value);
        }
        return ret;
    }
    return obj;
}
exports.filterUndefined = filterUndefined;
/**
 * A Token that applies a function AFTER resolve resolution
 */
class PostResolveToken extends intrinsic_1.Intrinsic {
    constructor(value, processor) {
        super(value);
        this.processor = processor;
    }
    resolve(context) {
        context.registerPostProcessor(this);
        return super.resolve(context);
    }
    postProcess(o, _context) {
        return this.processor(o);
    }
}
exports.PostResolveToken = PostResolveToken;
/**
 * @returns the list of stacks that lead from the top-level stack (non-nested) all the way to a nested stack.
 */
function pathToTopLevelStack(s) {
    if (s.nestedStackParent) {
        return [...pathToTopLevelStack(s.nestedStackParent), s];
    }
    else {
        return [s];
    }
}
exports.pathToTopLevelStack = pathToTopLevelStack;
/**
 * Given two arrays, returns the last common element or `undefined` if there
 * isn't (arrays are foriegn).
 */
function findLastCommonElement(path1, path2) {
    let i = 0;
    while (i < path1.length && i < path2.length) {
        if (path1[i] !== path2[i]) {
            break;
        }
        i++;
    }
    return path1[i - 1];
}
exports.findLastCommonElement = findLastCommonElement;
//# sourceMappingURL=data:application/json;base64,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