"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = void 0;
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const construct_compat_1 = require("./construct-compat");
const runtime_info_1 = require("./private/runtime-info");
const synthesis_1 = require("./private/synthesis");
/**
 * An abstract application modeling unit consisting of Stacks that should be
 * deployed together.
 *
 * Derive a subclass of `Stage` and use it to model a single instance of your
 * application.
 *
 * You can then instantiate your subclass multiple times to model multiple
 * copies of your application which should be be deployed to different
 * environments.
 */
class Stage extends construct_compat_1.Construct {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e, _f, _g;
        super(scope, id);
        if (id !== '' && !/^[a-z][a-z0-9\-\_\.]+$/i.test(id)) {
            throw new Error(`invalid stage name "${id}". Stage name must start with a letter and contain only alphanumeric characters, hypens ('-'), underscores ('_') and periods ('.')`);
        }
        this.parentStage = Stage.of(this);
        this.region = (_b = (_a = props.env) === null || _a === void 0 ? void 0 : _a.region) !== null && _b !== void 0 ? _b : (_c = this.parentStage) === null || _c === void 0 ? void 0 : _c.region;
        this.account = (_e = (_d = props.env) === null || _d === void 0 ? void 0 : _d.account) !== null && _e !== void 0 ? _e : (_f = this.parentStage) === null || _f === void 0 ? void 0 : _f.account;
        this._assemblyBuilder = this.createBuilder(props.outdir);
        this.stageName = [(_g = this.parentStage) === null || _g === void 0 ? void 0 : _g.stageName, id].filter(x => x).join('-');
    }
    /**
     * Return the stage this construct is contained with, if available. If called
     * on a nested stage, returns its parent.
     *
     * @experimental
     */
    static of(construct) {
        return construct.node.scopes.reverse().slice(1).find(Stage.isStage);
    }
    /**
     * Test whether the given construct is a stage.
     *
     * @experimental
     */
    static isStage(x) {
        return x !== null && x instanceof Stage;
    }
    /**
     * Artifact ID of the assembly if it is a nested stage. The root stage (app)
     * will return an empty string.
     *
     * Derived from the construct path.
     *
     * @experimental
     */
    get artifactId() {
        if (!this.node.path) {
            return '';
        }
        return `assembly-${this.node.path.replace(/\//g, '-').replace(/^-+|-+$/g, '')}`;
    }
    /**
     * Synthesize this stage into a cloud assembly.
     *
     * Once an assembly has been synthesized, it cannot be modified. Subsequent
     * calls will return the same assembly.
     */
    synth(options = {}) {
        if (!this.assembly) {
            const runtimeInfo = this.node.tryGetContext(cxapi.DISABLE_VERSION_REPORTING) ? undefined : runtime_info_1.collectRuntimeInformation();
            this.assembly = synthesis_1.synthesize(this, {
                skipValidation: options.skipValidation,
                runtimeInfo,
            });
        }
        return this.assembly;
    }
    createBuilder(outdir) {
        // cannot specify "outdir" if we are a nested stage
        if (this.parentStage && outdir) {
            throw new Error('"outdir" cannot be specified for nested stages');
        }
        // Need to determine fixed output directory already, because we must know where
        // to write sub-assemblies (which must happen before we actually get to this app's
        // synthesize() phase).
        return this.parentStage
            ? this.parentStage._assemblyBuilder.createNestedAssembly(this.artifactId, this.node.path)
            : new cxapi.CloudAssemblyBuilder(outdir);
    }
}
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,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