"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretValue = void 0;
const cfn_dynamic_reference_1 = require("./cfn-dynamic-reference");
const intrinsic_1 = require("./private/intrinsic");
/**
 * Work with secret values in the CDK
 *
 * Secret values in the CDK (such as those retrieved from SecretsManager) are
 * represented as regular strings, just like other values that are only
 * available at deployment time.
 *
 * To help you avoid accidental mistakes which would lead to you putting your
 * secret values directly into a CloudFormation template, constructs that take
 * secret values will not allow you to pass in a literal secret value. They do
 * so by calling `Secret.assertSafeSecret()`.
 *
 * You can escape the check by calling `Secret.plainText()`, but doing
 * so is highly discouraged.
 */
class SecretValue extends intrinsic_1.Intrinsic {
    /**
     * Construct a literal secret value for use with secret-aware constructs
     *
     * *Do not use this method for any secrets that you care about.*
     *
     * The only reasonable use case for using this method is when you are testing.
     */
    static plainText(secret) {
        return new SecretValue(secret);
    }
    /**
     * Creates a `SecretValue` with a value which is dynamically loaded from AWS Secrets Manager.
     * @param secretId The ID or ARN of the secret
     * @param options Options
     */
    static secretsManager(secretId, options = {}) {
        if (!secretId) {
            throw new Error('secretId cannot be empty');
        }
        if (!secretId.startsWith('arn:') && secretId.includes(':')) {
            throw new Error(`secret id "${secretId}" is not an ARN but contains ":"`);
        }
        const parts = [
            secretId,
            'SecretString',
            options.jsonField || '',
            options.versionStage || '',
            options.versionId || '',
        ];
        const dyref = new cfn_dynamic_reference_1.CfnDynamicReference(cfn_dynamic_reference_1.CfnDynamicReferenceService.SECRETS_MANAGER, parts.join(':'));
        return this.cfnDynamicReference(dyref);
    }
    /**
     * Use a secret value stored from a Systems Manager (SSM) parameter.
     *
     * @param parameterName The name of the parameter in the Systems Manager
     * Parameter Store. The parameter name is case-sensitive.
     *
     * @param version An integer that specifies the version of the parameter to
     * use. You must specify the exact version. You cannot currently specify that
     * AWS CloudFormation use the latest version of a parameter.
     */
    static ssmSecure(parameterName, version) {
        const parts = [parameterName, version];
        return this.cfnDynamicReference(new cfn_dynamic_reference_1.CfnDynamicReference(cfn_dynamic_reference_1.CfnDynamicReferenceService.SSM_SECURE, parts.join(':')));
    }
    /**
     * Obtain the secret value through a CloudFormation dynamic reference.
     *
     * If possible, use `SecretValue.ssmSecure` or `SecretValue.secretsManager` directly.
     *
     * @param ref The dynamic reference to use.
     */
    static cfnDynamicReference(ref) {
        return new SecretValue(ref);
    }
    /**
     * Obtain the secret value through a CloudFormation parameter.
     *
     * Generally, this is not a recommended approach. AWS Secrets Manager is the
     * recommended way to reference secrets.
     *
     * @param param The CloudFormation parameter to use.
     */
    static cfnParameter(param) {
        if (!param.noEcho) {
            throw new Error('CloudFormation parameter must be configured with "NoEcho"');
        }
        return new SecretValue(param.value);
    }
}
exports.SecretValue = SecretValue;
//# sourceMappingURL=data:application/json;base64,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