"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareApp = void 0;
const constructs_1 = require("constructs");
const cfn_resource_1 = require("../cfn-resource");
const stack_1 = require("../stack");
const stage_1 = require("../stage");
const refs_1 = require("./refs");
/**
 * Prepares the app for synthesis. This function is called by the root `prepare`
 * (normally this the App, but if a Stack is a root, it is called by the stack),
 * which means it's the last 'prepare' that executes.
 *
 * It takes care of reifying cross-references between stacks (or nested stacks),
 * and of creating assets for nested stack templates.
 *
 * @param root The root of the construct tree.
 */
function prepareApp(root) {
    if (root.node.scope && !stage_1.Stage.isStage(root)) {
        throw new Error('prepareApp can only be called on a stage or a root construct');
    }
    // apply dependencies between resources in depending subtrees
    for (const dependency of root.node.dependencies) {
        const targetCfnResources = findCfnResources(dependency.target);
        const sourceCfnResources = findCfnResources(dependency.source);
        for (const target of targetCfnResources) {
            for (const source of sourceCfnResources) {
                source.addDependsOn(target);
            }
        }
    }
    // depth-first (children first) queue of nested stacks. We will pop a stack
    // from the head of this queue to prepare its template asset.
    const queue = findAllNestedStacks(root);
    while (true) {
        refs_1.resolveReferences(root);
        const nested = queue.shift();
        if (!nested) {
            break;
        }
        defineNestedStackAsset(nested);
    }
}
exports.prepareApp = prepareApp;
/**
 * Prepares the assets for nested stacks in this app.
 * @returns `true` if assets were added to the parent of a nested stack, which
 * implies that another round of reference resolution is in order. If this
 * function returns `false`, we know we are done.
 */
function defineNestedStackAsset(nestedStack) {
    // this is needed temporarily until we move NestedStack to '@aws-cdk/core'.
    const nested = nestedStack;
    nested._prepareTemplateAsset();
}
function findAllNestedStacks(root) {
    const result = new Array();
    const includeStack = (stack) => {
        if (!stack_1.Stack.isStack(stack)) {
            return false;
        }
        if (!stack.nested) {
            return false;
        }
        // test: if we are not within a stage, then include it.
        if (!stage_1.Stage.of(stack)) {
            return true;
        }
        return stage_1.Stage.of(stack) === root;
    };
    // create a list of all nested stacks in depth-first post order this means
    // that we first prepare the leaves and then work our way up.
    for (const stack of root.node.findAll(constructs_1.ConstructOrder.POSTORDER /* <== important */)) {
        if (includeStack(stack)) {
            result.push(stack);
        }
    }
    return result;
}
/**
 * Find all resources in a set of constructs
 */
function findCfnResources(root) {
    return root.node.findAll().filter(cfn_resource_1.CfnResource.isCfnResource);
}
//# sourceMappingURL=data:application/json;base64,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