"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LogicalIDs = void 0;
/**
 * Class that keeps track of the logical IDs that are assigned to resources
 *
 * Supports renaming the generated IDs.
 */
class LogicalIDs {
    constructor() {
        /**
         * The rename table (old to new)
         */
        this.renames = {};
        /**
         * All assigned names (new to old, may be identical)
         *
         * This is used to ensure that:
         *
         * - No 2 resources end up with the same final logical ID, unless they were the same to begin with.
         * - All renames have been used at the end of renaming.
         */
        this.reverse = {};
    }
    /**
     * Rename a logical ID from an old ID to a new ID
     */
    addRename(oldId, newId) {
        if (oldId in this.renames) {
            throw new Error(`A rename has already been registered for '${oldId}'`);
        }
        this.renames[oldId] = newId;
    }
    /**
     * Return the renamed version of an ID or the original ID.
     */
    applyRename(oldId) {
        let newId = oldId;
        if (oldId in this.renames) {
            newId = this.renames[oldId];
        }
        // If this newId has already been used, it must have been with the same oldId
        if (newId in this.reverse && this.reverse[newId] !== oldId) {
            // eslint-disable-next-line max-len
            throw new Error(`Two objects have been assigned the same Logical ID: '${this.reverse[newId]}' and '${oldId}' are now both named '${newId}'.`);
        }
        this.reverse[newId] = oldId;
        validateLogicalId(newId);
        return newId;
    }
    /**
     * Throw an error if not all renames have been used
     *
     * This is to assure that users didn't make typoes when registering renames.
     */
    assertAllRenamesApplied() {
        const keys = new Set();
        Object.keys(this.renames).forEach(keys.add.bind(keys));
        Object.keys(this.reverse).map(newId => {
            keys.delete(this.reverse[newId]);
        });
        if (keys.size !== 0) {
            const unusedRenames = Array.from(keys.values());
            throw new Error(`The following Logical IDs were attempted to be renamed, but not found: ${unusedRenames.join(', ')}`);
        }
    }
}
exports.LogicalIDs = LogicalIDs;
const VALID_LOGICALID_REGEX = /^[A-Za-z][A-Za-z0-9]{1,254}$/;
/**
 * Validate logical ID is valid for CloudFormation
 */
function validateLogicalId(logicalId) {
    if (!VALID_LOGICALID_REGEX.test(logicalId)) {
        throw new Error(`Logical ID must adhere to the regular expression: ${VALID_LOGICALID_REGEX.toString()}, got '${logicalId}'`);
    }
}
//# sourceMappingURL=data:application/json;base64,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