"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NestedStack = void 0;
const crypto = require("crypto");
const assets_1 = require("./assets");
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cloudformation_generated_1 = require("./cloudformation.generated");
const construct_compat_1 = require("./construct-compat");
const lazy_1 = require("./lazy");
const stack_1 = require("./stack");
const stack_synthesizers_1 = require("./stack-synthesizers");
const token_1 = require("./token");
const NESTED_STACK_SYMBOL = Symbol.for('@aws-cdk/core.NestedStack');
/**
 * A CloudFormation nested stack.
 *
 * When you apply template changes to update a top-level stack, CloudFormation
 * updates the top-level stack and initiates an update to its nested stacks.
 * CloudFormation updates the resources of modified nested stacks, but does not
 * update the resources of unmodified nested stacks.
 *
 * Furthermore, this stack will not be treated as an independent deployment
 * artifact (won't be listed in "cdk list" or deployable through "cdk deploy"),
 * but rather only synthesized as a template and uploaded as an asset to S3.
 *
 * Cross references of resource attributes between the parent stack and the
 * nested stack will automatically be translated to stack parameters and
 * outputs.
 *
 * @experimental
 */
class NestedStack extends stack_1.Stack {
    constructor(scope, id, props = {}) {
        const parentStack = findParentStack(scope);
        super(scope, id, {
            env: { account: parentStack.account, region: parentStack.region },
            synthesizer: new stack_synthesizers_1.NestedStackSynthesizer(parentStack.synthesizer),
        });
        this._parentStack = parentStack;
        // @deprecate: remove this in v2.0 (redundent)
        const parentScope = new construct_compat_1.Construct(scope, id + '.NestedStack');
        Object.defineProperty(this, NESTED_STACK_SYMBOL, { value: true });
        // this is the file name of the synthesized template file within the cloud assembly
        this.templateFile = `${this.node.uniqueId}.nested.template.json`;
        this.parameters = props.parameters || {};
        this.resource = new cloudformation_generated_1.CfnStack(parentScope, `${id}.NestedStackResource`, {
            templateUrl: lazy_1.Lazy.stringValue({ produce: () => this._templateUrl || '<unresolved>' }),
            parameters: lazy_1.Lazy.anyValue({ produce: () => Object.keys(this.parameters).length > 0 ? this.parameters : undefined }),
            notificationArns: props.notificationArns,
            timeoutInMinutes: props.timeout ? props.timeout.toMinutes() : undefined,
        });
        this.nestedStackResource = this.resource;
        // context-aware stack name: if resolved from within this stack, return AWS::StackName
        // if resolved from the outer stack, use the { Ref } of the AWS::CloudFormation::Stack resource
        // which resolves the ARN of the stack. We need to extract the stack name, which is the second
        // component after splitting by "/"
        this._contextualStackName = this.contextualAttribute(cfn_pseudo_1.Aws.STACK_NAME, cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.split('/', this.resource.ref)));
        this._contextualStackId = this.contextualAttribute(cfn_pseudo_1.Aws.STACK_ID, this.resource.ref);
    }
    /**
     * Checks if `x` is an object of type `NestedStack`.
     */
    static isNestedStack(x) {
        return x != null && typeof (x) === 'object' && NESTED_STACK_SYMBOL in x;
    }
    /**
     * An attribute that represents the name of the nested stack.
     *
     * This is a context aware attribute:
     * - If this is referenced from the parent stack, it will return a token that parses the name from the stack ID.
     * - If this is referenced from the context of the nested stack, it will return `{ "Ref": "AWS::StackName" }`
     *
     * @attribute
     * @example mystack-mynestedstack-sggfrhxhum7w
     */
    get stackName() {
        return this._contextualStackName;
    }
    /**
     * An attribute that represents the ID of the stack.
     *
     * This is a context aware attribute:
     * - If this is referenced from the parent stack, it will return `{ "Ref": "LogicalIdOfNestedStackResource" }`.
     * - If this is referenced from the context of the nested stack, it will return `{ "Ref": "AWS::StackId" }`
     *
     * @attribute
     * @example "arn:aws:cloudformation:us-east-2:123456789012:stack/mystack-mynestedstack-sggfrhxhum7w/f449b250-b969-11e0-a185-5081d0136786"
     */
    get stackId() {
        return this._contextualStackId;
    }
    /**
     * Assign a value to one of the nested stack parameters.
     * @param name The parameter name (ID)
     * @param value The value to assign
     */
    setParameter(name, value) {
        this.parameters[name] = value;
    }
    /**
     * Defines an asset at the parent stack which represents the template of this
     * nested stack.
     *
     * This private API is used by `App.prepare()` within a loop that rectifies
     * references every time an asset is added. This is because (at the moment)
     * assets are addressed using CloudFormation parameters.
     *
     * @returns `true` if a new asset was added or `false` if an asset was
     * previously added. When this returns `true`, App will do another reference
     * rectification cycle.
     *
     * @internal
     */
    _prepareTemplateAsset() {
        if (this._templateUrl) {
            return false;
        }
        const cfn = JSON.stringify(this._toCloudFormation());
        const templateHash = crypto.createHash('sha256').update(cfn).digest('hex');
        const templateLocation = this._parentStack.addFileAsset({
            packaging: assets_1.FileAssetPackaging.FILE,
            sourceHash: templateHash,
            fileName: this.templateFile,
        });
        // if bucketName/objectKey are cfn parameters from a stack other than the parent stack, they will
        // be resolved as cross-stack references like any other (see "multi" tests).
        this._templateUrl = `https://s3.${this._parentStack.region}.${this._parentStack.urlSuffix}/${templateLocation.bucketName}/${templateLocation.objectKey}`;
        return true;
    }
    contextualAttribute(innerValue, outerValue) {
        return token_1.Token.asString({
            resolve: (context) => {
                if (stack_1.Stack.of(context.scope) === this) {
                    return innerValue;
                }
                else {
                    return outerValue;
                }
            },
        });
    }
}
exports.NestedStack = NestedStack;
/**
 * Validates the scope for a nested stack. Nested stacks must be defined within the scope of another `Stack`.
 */
function findParentStack(scope) {
    if (!scope) {
        throw new Error('Nested stacks cannot be defined as a root construct');
    }
    const parentStack = scope.node.scopes.reverse().find(p => stack_1.Stack.isStack(p));
    if (!parentStack) {
        throw new Error('Nested stacks must be defined within scope of another non-nested stack');
    }
    return parentStack;
}
//# sourceMappingURL=data:application/json;base64,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