"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lazy = void 0;
const stack_trace_1 = require("./stack-trace");
const token_1 = require("./token");
/**
 * Lazily produce a value
 *
 * Can be used to return a string, list or numeric value whose actual value
 * will only be calculated later, during synthesis.
 */
class Lazy {
    static stringValue(producer, options = {}) {
        return token_1.Token.asString(new LazyString(producer), options);
    }
    static numberValue(producer) {
        return token_1.Token.asNumber(new LazyNumber(producer));
    }
    static listValue(producer, options = {}) {
        return token_1.Token.asList(new LazyList(producer, options), options);
    }
    static anyValue(producer, options = {}) {
        return new LazyAny(producer, options);
    }
    constructor() {
    }
}
exports.Lazy = Lazy;
class LazyBase {
    constructor() {
        this.creationStack = stack_trace_1.captureStackTrace();
    }
    toString() {
        return token_1.Token.asString(this);
    }
    /**
     * Turn this Token into JSON
     *
     * Called automatically when JSON.stringify() is called on a Token.
     */
    toJSON() {
        return '<unresolved-lazy>';
    }
}
class LazyString extends LazyBase {
    constructor(producer) {
        super();
        this.producer = producer;
    }
    resolve(context) {
        return this.producer.produce(context);
    }
}
class LazyNumber extends LazyBase {
    constructor(producer) {
        super();
        this.producer = producer;
    }
    resolve(context) {
        return this.producer.produce(context);
    }
}
class LazyList extends LazyBase {
    constructor(producer, options = {}) {
        super();
        this.producer = producer;
        this.options = options;
    }
    resolve(context) {
        const ret = this.producer.produce(context);
        if (ret !== undefined && ret.length === 0 && this.options.omitEmpty) {
            return undefined;
        }
        return ret;
    }
}
class LazyAny extends LazyBase {
    constructor(producer, options = {}) {
        super();
        this.producer = producer;
        this.options = options;
    }
    resolve(context) {
        const ret = this.producer.produce(context);
        if (Array.isArray(ret) && ret.length === 0 && this.options.omitEmptyArray) {
            return undefined;
        }
        return ret;
    }
}
//# sourceMappingURL=data:application/json;base64,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