"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContextProvider = void 0;
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const stack_1 = require("./stack");
const token_1 = require("./token");
/**
 * Base class for the model side of context providers
 *
 * Instances of this class communicate with context provider plugins in the 'cdk
 * toolkit' via context variables (input), outputting specialized queries for
 * more context variables (output).
 *
 * ContextProvider needs access to a Construct to hook into the context mechanism.
 *
 * @experimental
 */
class ContextProvider {
    /**
     * @returns the context key or undefined if a key cannot be rendered (due to tokens used in any of the props)
     */
    static getKey(scope, options) {
        const stack = stack_1.Stack.of(scope);
        const props = {
            account: stack.account,
            region: stack.region,
            ...options.props || {},
        };
        if (Object.values(props).find(x => token_1.Token.isUnresolved(x))) {
            throw new Error(`Cannot determine scope for context provider ${options.provider}.\n` +
                'This usually happens when one or more of the provider props have unresolved tokens');
        }
        const propStrings = propsToArray(props);
        return {
            key: `${options.provider}:${propStrings.join(':')}`,
            props,
        };
    }
    static getValue(scope, options) {
        const stack = stack_1.Stack.of(scope);
        if (token_1.Token.isUnresolved(stack.account) || token_1.Token.isUnresolved(stack.region)) {
            throw new Error(`Cannot retrieve value from context provider ${options.provider} since account/region ` +
                'are not specified at the stack level. Either configure "env" with explicit account and region when ' +
                'you define your stack, or use the environment variables "CDK_DEFAULT_ACCOUNT" and "CDK_DEFAULT_REGION" ' +
                'to inherit environment information from the CLI (not recommended for production stacks)');
        }
        const { key, props } = this.getKey(scope, options);
        const value = scope.node.tryGetContext(key);
        const providerError = extractProviderError(value);
        // if context is missing or an error occurred during context retrieval,
        // report and return a dummy value.
        if (value === undefined || providerError !== undefined) {
            stack.reportMissingContext({
                key,
                provider: options.provider,
                props: props,
            });
            if (providerError !== undefined) {
                scope.node.addError(providerError);
            }
            return { value: options.dummyValue };
        }
        return { value };
    }
    constructor() { }
}
exports.ContextProvider = ContextProvider;
/**
 * If the context value represents an error, return the error message
 */
function extractProviderError(value) {
    if (typeof value === 'object' && value !== null) {
        return value[cxapi.PROVIDER_ERROR_KEY];
    }
    return undefined;
}
/**
 * Quote colons in all strings so that we can undo the quoting at a later point
 *
 * We'll use $ as a quoting character, for no particularly good reason other
 * than that \ is going to lead to quoting hell when the keys are stored in JSON.
 */
function colonQuote(xs) {
    return xs.replace('$', '$$').replace(':', '$:');
}
function propsToArray(props, keyPrefix = '') {
    const ret = [];
    for (const key of Object.keys(props)) {
        // skip undefined values
        if (props[key] === undefined) {
            continue;
        }
        switch (typeof props[key]) {
            case 'object': {
                ret.push(...propsToArray(props[key], `${keyPrefix}${key}.`));
                break;
            }
            case 'string': {
                ret.push(`${keyPrefix}${key}=${colonQuote(props[key])}`);
                break;
            }
            default: {
                ret.push(`${keyPrefix}${key}=${JSON.stringify(props[key])}`);
                break;
            }
        }
    }
    ret.sort();
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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