"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConstructNode = exports.ConstructOrder = exports.Construct = void 0;
/**
 * Constructs compatibility layer.
 *
 * This file includes types that shadow types in the "constructs" module in
 * order to allow backwards-compatiblity in the AWS CDK v1.0 release line.
 *
 * There are pretty ugly hacks here, which mostly involve downcasting types to
 * adhere to legacy AWS CDK APIs.
 *
 * This file, in its entirety, is expected to be removed in v2.0.
 */
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const constructs = require("constructs");
const token_1 = require("./token");
const ORIGINAL_CONSTRUCT_NODE_SYMBOL = Symbol.for('@aws-cdk/core.ConstructNode');
const CONSTRUCT_SYMBOL = Symbol.for('@aws-cdk/core.Construct');
/**
 * Represents the building block of the construct graph.
 *
 * All constructs besides the root construct must be created within the scope of
 * another construct.
 */
class Construct extends constructs.Construct {
    constructor(scope, id) {
        super(scope, id, {
            nodeFactory: {
                createNode: (h, s, i) => new ConstructNode(h, s, i)._actualNode,
            },
        });
        if (token_1.Token.isUnresolved(id)) {
            throw new Error(`Cannot use tokens in construct ID: ${id}`);
        }
        Object.defineProperty(this, CONSTRUCT_SYMBOL, { value: true });
        this.node = ConstructNode._unwrap(constructs.Node.of(this));
        const disableTrace = this.node.tryGetContext(cxapi.DISABLE_METADATA_STACK_TRACE) ||
            this.node.tryGetContext(constructs.ConstructMetadata.DISABLE_STACK_TRACE_IN_METADATA) ||
            process.env.CDK_DISABLE_STACK_TRACE;
        if (disableTrace) {
            this.node.setContext(cxapi.DISABLE_METADATA_STACK_TRACE, true);
            this.node.setContext(constructs.ConstructMetadata.DISABLE_STACK_TRACE_IN_METADATA, true);
            process.env.CDK_DISABLE_STACK_TRACE = '1';
        }
    }
    /**
     * Return whether the given object is a Construct
     */
    static isConstruct(x) {
        return typeof x === 'object' && x !== null && CONSTRUCT_SYMBOL in x;
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @returns An array of validation error messages, or an empty array if the construct is valid.
     */
    onValidate() {
        return this.validate();
    }
    /**
     * Perform final modifications before synthesis
     *
     * This method can be implemented by derived constructs in order to perform
     * final changes before synthesis. prepare() will be called after child
     * constructs have been prepared.
     *
     * This is an advanced framework feature. Only use this if you
     * understand the implications.
     */
    onPrepare() {
        this.prepare();
    }
    /**
     * Allows this construct to emit artifacts into the cloud assembly during synthesis.
     *
     * This method is usually implemented by framework-level constructs such as `Stack` and `Asset`
     * as they participate in synthesizing the cloud assembly.
     *
     * @param session The synthesis session.
     */
    onSynthesize(session) {
        this.synthesize({
            outdir: session.outdir,
            assembly: session.assembly,
        });
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @returns An array of validation error messages, or an empty array if the construct is valid.
     */
    validate() {
        return [];
    }
    /**
     * Perform final modifications before synthesis
     *
     * This method can be implemented by derived constructs in order to perform
     * final changes before synthesis. prepare() will be called after child
     * constructs have been prepared.
     *
     * This is an advanced framework feature. Only use this if you
     * understand the implications.
     */
    prepare() {
        return;
    }
    /**
     * Allows this construct to emit artifacts into the cloud assembly during synthesis.
     *
     * This method is usually implemented by framework-level constructs such as `Stack` and `Asset`
     * as they participate in synthesizing the cloud assembly.
     *
     * @param session The synthesis session.
     */
    synthesize(session) {
        ignore(session);
    }
}
exports.Construct = Construct;
/**
 * In what order to return constructs
 */
var ConstructOrder;
(function (ConstructOrder) {
    /**
     * Depth-first, pre-order
     */
    ConstructOrder[ConstructOrder["PREORDER"] = 0] = "PREORDER";
    /**
     * Depth-first, post-order (leaf nodes first)
     */
    ConstructOrder[ConstructOrder["POSTORDER"] = 1] = "POSTORDER";
})(ConstructOrder = exports.ConstructOrder || (exports.ConstructOrder = {}));
/**
 * Represents the construct node in the scope tree.
 */
class ConstructNode {
    constructor(host, scope, id) {
        this._actualNode = new constructs.Node(host, scope, id);
        // store a back reference on _actualNode so we can our ConstructNode from it
        Object.defineProperty(this._actualNode, ORIGINAL_CONSTRUCT_NODE_SYMBOL, {
            value: this,
            configurable: false,
            enumerable: false,
        });
    }
    /**
     * Returns the wrapping `@aws-cdk/core.ConstructNode` instance from a `constructs.ConstructNode`.
     *
     * @internal
     */
    static _unwrap(c) {
        const x = c[ORIGINAL_CONSTRUCT_NODE_SYMBOL];
        if (!x) {
            throw new Error('invalid ConstructNode type');
        }
        return x;
    }
    /**
     * Synthesizes a CloudAssembly from a construct tree.
     * @param node The root of the construct tree.
     * @param options Synthesis options.
     * @deprecated Use `app.synth()` or `stage.synth()` instead
     */
    static synth(node, options = {}) {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const a = require('./private/synthesis');
        return a.synthesize(node.root, options);
    }
    /**
     * Invokes "prepare" on all constructs (depth-first, post-order) in the tree under `node`.
     * @param node The root node
     * @deprecated Use `app.synth()` instead
     */
    static prepare(node) {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const p = require('./private/prepare-app');
        p.prepareApp(node.root); // resolve cross refs and nested stack assets.
        return node._actualNode.prepare();
    }
    /**
     * Invokes "validate" on all constructs in the tree (depth-first, pre-order) and returns
     * the list of all errors. An empty list indicates that there are no errors.
     *
     * @param node The root node
     */
    static validate(node) {
        return node._actualNode.validate().map(e => ({ source: e.source, message: e.message }));
    }
    /**
     * Returns the scope in which this construct is defined.
     *
     * The value is `undefined` at the root of the construct scope tree.
     */
    get scope() {
        return this._actualNode.scope;
    }
    /**
     * The id of this construct within the current scope.
     *
     * This is a a scope-unique id. To obtain an app-unique id for this construct, use `uniqueId`.
     */
    get id() { return this._actualNode.id; }
    /**
     * The full, absolute path of this construct in the tree.
     *
     * Components are separated by '/'.
     */
    get path() { return this._actualNode.path; }
    /**
     * A tree-global unique alphanumeric identifier for this construct.
     * Includes all components of the tree.
     */
    get uniqueId() { return this._actualNode.uniqueId; }
    /**
     * Return a direct child by id, or undefined
     *
     * @param id Identifier of direct child
     * @returns the child if found, or undefined
     */
    tryFindChild(id) { return this._actualNode.tryFindChild(id); }
    /**
     * Return a direct child by id
     *
     * Throws an error if the child is not found.
     *
     * @param id Identifier of direct child
     * @returns Child with the given id.
     */
    findChild(id) { return this._actualNode.findChild(id); }
    /**
     * Returns the child construct that has the id `Default` or `Resource"`.
     * This is usually the construct that provides the bulk of the underlying functionality.
     * Useful for modifications of the underlying construct that are not available at the higher levels.
     *
     * @throws if there is more than one child
     * @returns a construct or undefined if there is no default child
     */
    get defaultChild() { return this._actualNode.defaultChild; }
    /**
     * Override the defaultChild property.
     *
     * This should only be used in the cases where the correct
     * default child is not named 'Resource' or 'Default' as it
     * should be.
     *
     * If you set this to undefined, the default behavior of finding
     * the child named 'Resource' or 'Default' will be used.
     */
    set defaultChild(value) { this._actualNode.defaultChild = value; }
    /**
     * All direct children of this construct.
     */
    get children() { return this._actualNode.children; }
    /**
     * Return this construct and all of its children in the given order
     */
    findAll(order = ConstructOrder.PREORDER) { return this._actualNode.findAll(order); }
    /**
     * This can be used to set contextual values.
     * Context must be set before any children are added, since children may consult context info during construction.
     * If the key already exists, it will be overridden.
     * @param key The context key
     * @param value The context value
     */
    setContext(key, value) {
        if (token_1.Token.isUnresolved(key)) {
            throw new Error('Invalid context key: context keys can\'t include tokens');
        }
        this._actualNode.setContext(key, value);
    }
    /**
     * Retrieves a value from tree context.
     *
     * Context is usually initialized at the root, but can be overridden at any point in the tree.
     *
     * @param key The context key
     * @returns The context value or `undefined` if there is no context value for thie key.
     */
    tryGetContext(key) {
        if (token_1.Token.isUnresolved(key)) {
            throw new Error('Invalid context key: context keys can\'t include tokens');
        }
        return this._actualNode.tryGetContext(key);
    }
    /**
     * An immutable array of metadata objects associated with this construct.
     * This can be used, for example, to implement support for deprecation notices, source mapping, etc.
     */
    get metadata() { return this._actualNode.metadata; }
    /**
     * Adds a metadata entry to this construct.
     * Entries are arbitrary values and will also include a stack trace to allow tracing back to
     * the code location for when the entry was added. It can be used, for example, to include source
     * mapping in CloudFormation templates to improve diagnostics.
     *
     * @param type a string denoting the type of metadata
     * @param data the value of the metadata (can be a Token). If null/undefined, metadata will not be added.
     * @param fromFunction a function under which to restrict the metadata entry's stack trace (defaults to this.addMetadata)
     */
    addMetadata(type, data, fromFunction) { this._actualNode.addMetadata(type, data, fromFunction); }
    /**
     * Adds a { "info": <message> } metadata entry to this construct.
     * The toolkit will display the info message when apps are synthesized.
     * @param message The info message.
     */
    addInfo(message) {
        this._actualNode.addMetadata(cxschema.ArtifactMetadataEntryType.INFO, message);
    }
    /**
     * Adds a { "warning": <message> } metadata entry to this construct.
     * The toolkit will display the warning when an app is synthesized, or fail
     * if run in --strict mode.
     * @param message The warning message.
     */
    addWarning(message) {
        this._actualNode.addMetadata(cxschema.ArtifactMetadataEntryType.WARN, message);
    }
    /**
     * Adds an { "error": <message> } metadata entry to this construct.
     * The toolkit will fail synthesis when errors are reported.
     * @param message The error message.
     */
    addError(message) {
        this._actualNode.addMetadata(cxschema.ArtifactMetadataEntryType.ERROR, message);
    }
    /**
     * Applies the aspect to this Constructs node
     */
    applyAspect(aspect) { this._actualNode.applyAspect(aspect); }
    /**
     * All parent scopes of this construct.
     *
     * @returns a list of parent scopes. The last element in the list will always
     * be the current construct and the first element will be the root of the
     * tree.
     */
    get scopes() { return this._actualNode.scopes; }
    /**
     * @returns The root of the construct tree.
     */
    get root() { return this._actualNode.root; }
    /**
     * Returns true if this construct or the scopes in which it is defined are
     * locked.
     */
    get locked() { return this._actualNode.locked; }
    /**
     * Add an ordering dependency on another Construct.
     *
     * All constructs in the dependency's scope will be deployed before any
     * construct in this construct's scope.
     */
    addDependency(...dependencies) { this._actualNode.addDependency(...dependencies); }
    /**
     * Return all dependencies registered on this node or any of its children
     */
    get dependencies() { return this._actualNode.dependencies; }
    /**
     * Remove the child with the given name, if present.
     *
     * @returns Whether a child with the given name was deleted.
     * @experimental
     */
    tryRemoveChild(childName) { return this._actualNode.tryRemoveChild(childName); }
}
exports.ConstructNode = ConstructNode;
/**
 * Separator used to delimit construct path components.
 */
ConstructNode.PATH_SEP = '/';
function ignore(_x) {
    return;
}
//# sourceMappingURL=data:application/json;base64,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