"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ScopedAws = exports.Aws = void 0;
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
const AWS_ACCOUNTID = 'AWS::AccountId';
const AWS_URLSUFFIX = 'AWS::URLSuffix';
const AWS_NOTIFICATIONARNS = 'AWS::NotificationARNs';
const AWS_PARTITION = 'AWS::Partition';
const AWS_REGION = 'AWS::Region';
const AWS_STACKID = 'AWS::StackId';
const AWS_STACKNAME = 'AWS::StackName';
const AWS_NOVALUE = 'AWS::NoValue';
/**
 * Accessor for pseudo parameters
 *
 * Since pseudo parameters need to be anchored to a stack somewhere in the
 * construct tree, this class takes an scope parameter; the pseudo parameter
 * values can be obtained as properties from an scoped object.
 */
class Aws {
    constructor() { }
}
exports.Aws = Aws;
Aws.ACCOUNT_ID = pseudoString(AWS_ACCOUNTID);
Aws.URL_SUFFIX = pseudoString(AWS_URLSUFFIX);
Aws.NOTIFICATION_ARNS = token_1.Token.asList({ Ref: AWS_NOTIFICATIONARNS }, { displayHint: AWS_NOTIFICATIONARNS });
Aws.PARTITION = pseudoString(AWS_PARTITION);
Aws.REGION = pseudoString(AWS_REGION);
Aws.STACK_ID = pseudoString(AWS_STACKID);
Aws.STACK_NAME = pseudoString(AWS_STACKNAME);
Aws.NO_VALUE = pseudoString(AWS_NOVALUE);
/**
 * Accessor for scoped pseudo parameters
 *
 * These pseudo parameters are anchored to a stack somewhere in the construct
 * tree, and their values will be exported automatically.
 */
class ScopedAws {
    constructor(scope) {
        this.scope = scope;
    }
    get accountId() {
        return this.asString(AWS_ACCOUNTID);
    }
    get urlSuffix() {
        return this.asString(AWS_URLSUFFIX);
    }
    get notificationArns() {
        return token_1.Token.asList(cfn_reference_1.CfnReference.forPseudo(AWS_NOTIFICATIONARNS, this.scope), {
            displayHint: AWS_NOTIFICATIONARNS,
        });
    }
    get partition() {
        return this.asString(AWS_PARTITION);
    }
    get region() {
        return this.asString(AWS_REGION);
    }
    get stackId() {
        return this.asString(AWS_STACKID);
    }
    get stackName() {
        return this.asString(AWS_STACKNAME);
    }
    asString(name) {
        return token_1.Token.asString(cfn_reference_1.CfnReference.forPseudo(name, this.scope), { displayHint: name });
    }
}
exports.ScopedAws = ScopedAws;
function pseudoString(name) {
    return token_1.Token.asString({ Ref: name }, { displayHint: name });
}
//# sourceMappingURL=data:application/json;base64,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