"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.readYamlSync = exports.readJsonSync = void 0;
const fs = require("fs");
const yaml = require("yaml");
function readJsonSync(filePath) {
    const fileContents = fs.readFileSync(filePath);
    return JSON.parse(fileContents.toString());
}
exports.readJsonSync = readJsonSync;
function readYamlSync(filePath) {
    const fileContents = fs.readFileSync(filePath);
    return parseYamlStrWithCfnTags(fileContents.toString());
}
exports.readYamlSync = readYamlSync;
function makeTagForCfnIntrinsic(intrinsicName, addFnPrefix = true, resolveFun) {
    return {
        identify(value) { return typeof value === 'string'; },
        tag: `!${intrinsicName}`,
        resolve: resolveFun || ((_doc, cstNode) => {
            const ret = {};
            ret[addFnPrefix ? `Fn::${intrinsicName}` : intrinsicName] =
                // the +1 is to account for the ! the short form begins with
                parseYamlStrWithCfnTags(cstNode.toString().substring(intrinsicName.length + 1));
            return ret;
        }),
    };
}
const shortForms = [
    'Base64', 'Cidr', 'FindInMap', 'GetAZs', 'ImportValue', 'Join',
    'Select', 'Split', 'Transform', 'And', 'Equals', 'If', 'Not', 'Or',
].map(name => makeTagForCfnIntrinsic(name)).concat(
// ToDo: special logic for ImportValue will be needed when support for Fn::Sub is added. See
// https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-importvalue.html
makeTagForCfnIntrinsic('Ref', false), makeTagForCfnIntrinsic('GetAtt', true, (_doc, cstNode) => {
    // The position of the leftmost period and opening bracket tell us what syntax is being used
    // If no brackets are found, then the dot notation is being used; the leftmost dot separates the
    // logical ID from the attribute.
    //
    // If a bracket is found, then the list notation is being used; if present, the leftmost dot separates the
    // logical ID from the attribute.
    const firstDot = cstNode.toString().indexOf('.');
    const firstBracket = cstNode.toString().indexOf('[');
    return {
        'Fn::GetAtt': firstDot !== -1 && firstBracket === -1
            ? [
                cstNode.toString().substring('!GetAtt '.length, firstDot),
                parseYamlStrWithCfnTags((cstNode.toString().substring(firstDot + 1))),
            ]
            : parseYamlStrWithCfnTags(cstNode.toString().substring('!GetAtt'.length)),
    };
}));
function parseYamlStrWithCfnTags(text) {
    return yaml.parse(text, {
        customTags: shortForms,
        schema: 'yaml-1.1',
    });
}
//# sourceMappingURL=data:application/json;base64,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