"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StateGraph = void 0;
/**
 * A collection of connected states
 *
 * A StateGraph is used to keep track of all states that are connected (have
 * transitions between them). It does not include the substatemachines in
 * a Parallel's branches: those are their own StateGraphs, but the graphs
 * themselves have a hierarchical relationship as well.
 *
 * By assigning states to a definitive StateGraph, we verify that no state
 * machines are constructed. In particular:
 *
 * - Every state object can only ever be in 1 StateGraph, and not inadvertently
 *   be used in two graphs.
 * - Every stateId must be unique across all states in the entire state
 *   machine.
 *
 * All policy statements in all states in all substatemachines are bubbled so
 * that the top-level StateMachine instantiation can read them all and add
 * them to the IAM Role.
 *
 * You do not need to instantiate this class; it is used internally.
 */
class StateGraph {
    /**
     * @param startState state that gets executed when the state machine is launched
     * @param graphDescription description of the state machine
     */
    constructor(startState, graphDescription) {
        this.startState = startState;
        this.graphDescription = graphDescription;
        /**
         * The accumulated policy statements
         */
        this.policyStatements = new Array();
        /**
         * All states in this graph
         */
        this.allStates = new Set();
        /**
         * A mapping of stateId -> Graph for all states in this graph and subgraphs
         */
        this.allContainedStates = new Map();
        this.allStates.add(startState);
        startState.bindToGraph(this);
    }
    /**
     * Register a state as part of this graph
     *
     * Called by State.bindToGraph().
     */
    registerState(state) {
        this.registerContainedState(state.stateId, this);
        this.allStates.add(state);
    }
    /**
     * Register a Policy Statement used by states in this graph
     */
    registerPolicyStatement(statement) {
        if (this.superGraph) {
            this.superGraph.registerPolicyStatement(statement);
        }
        else {
            this.policyStatements.push(statement);
        }
    }
    /**
     * Register this graph as a child of the given graph
     *
     * Resource changes will be bubbled up to the given graph.
     */
    registerSuperGraph(graph) {
        if (this.superGraph === graph) {
            return;
        }
        if (this.superGraph) {
            throw new Error('Every StateGraph can only be registered into one other StateGraph');
        }
        this.superGraph = graph;
        this.pushContainedStatesUp(graph);
        this.pushPolicyStatementsUp(graph);
    }
    /**
     * Return the Amazon States Language JSON for this graph
     */
    toGraphJson() {
        const states = {};
        for (const state of this.allStates) {
            states[state.stateId] = state.toStateJson();
        }
        return {
            StartAt: this.startState.stateId,
            States: states,
            TimeoutSeconds: this.timeout && this.timeout.toSeconds(),
        };
    }
    /**
     * Return a string description of this graph
     */
    toString() {
        const someNodes = Array.from(this.allStates).slice(0, 3).map(x => x.stateId);
        if (this.allStates.size > 3) {
            someNodes.push('...');
        }
        return `${this.graphDescription} (${someNodes.join(', ')})`;
    }
    /**
     * Register a stateId and graph where it was registered
     */
    registerContainedState(stateId, graph) {
        if (this.superGraph) {
            this.superGraph.registerContainedState(stateId, graph);
        }
        else {
            const existingGraph = this.allContainedStates.get(stateId);
            if (existingGraph) {
                throw new Error(`State with name '${stateId}' occurs in both ${graph} and ${existingGraph}. All states must have unique names.`);
            }
            this.allContainedStates.set(stateId, graph);
        }
    }
    /**
     * Push all contained state info up to the given super graph
     */
    pushContainedStatesUp(superGraph) {
        for (const [stateId, graph] of this.allContainedStates) {
            superGraph.registerContainedState(stateId, graph);
        }
    }
    /**
     * Push all policy statements to into the given super graph
     */
    pushPolicyStatementsUp(superGraph) {
        for (const policyStatement of this.policyStatements) {
            superGraph.registerPolicyStatement(policyStatement);
        }
    }
}
exports.StateGraph = StateGraph;
//# sourceMappingURL=data:application/json;base64,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