"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FieldUtils = exports.Context = exports.Data = exports.JsonPath = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const json_path_1 = require("./json-path");
/**
 * Extract a field from the State Machine data or context
 * that gets passed around between states
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-paths.html
 */
class JsonPath {
    constructor() { }
    /**
     * Instead of using a literal string, get the value from a JSON path
     */
    static stringAt(path) {
        validateJsonPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * Instead of using a literal string list, get the value from a JSON path
     */
    static listAt(path) {
        // does not apply to task context
        validateDataPath(path);
        return core_1.Token.asList(new json_path_1.JsonPathToken(path));
    }
    /**
     * Instead of using a literal number, get the value from a JSON path
     */
    static numberAt(path) {
        validateJsonPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * Use the entire data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entirePayload() {
        return new json_path_1.JsonPathToken('$').toString();
    }
    /**
     * Determines if the indicated string is an encoded JSON path
     *
     * @param value string to be evaluated
     */
    static isEncodedJsonPath(value) {
        return !!json_path_1.jsonPathString(value);
    }
    /**
     * Return the Task Token field
     *
     * External actions will need this token to report step completion
     * back to StepFunctions using the `SendTaskSuccess` or `SendTaskFailure`
     * calls.
     */
    static get taskToken() {
        return new json_path_1.JsonPathToken('$$.Task.Token').toString();
    }
    /**
     * Use the entire context data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entireContext() {
        return new json_path_1.JsonPathToken('$$').toString();
    }
}
exports.JsonPath = JsonPath;
/**
 * Special string value to discard state input, output or result
 */
JsonPath.DISCARD = 'DISCARD';
/**
 * Extract a field from the State Machine data that gets passed around between states
 *
 * @deprecated replaced by `JsonPath`
 */
class Data {
    /**
     * Instead of using a literal string, get the value from a JSON path
     */
    static stringAt(path) {
        validateDataPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * Instead of using a literal string list, get the value from a JSON path
     */
    static listAt(path) {
        validateDataPath(path);
        return core_1.Token.asList(new json_path_1.JsonPathToken(path));
    }
    /**
     * Instead of using a literal number, get the value from a JSON path
     */
    static numberAt(path) {
        validateDataPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * Use the entire data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entirePayload() {
        return new json_path_1.JsonPathToken('$').toString();
    }
    /**
     * Determines if the indicated string is an encoded JSON path
     *
     * @param value string to be evaluated
     */
    static isJsonPathString(value) {
        return !!json_path_1.jsonPathString(value);
    }
    constructor() { }
}
exports.Data = Data;
/**
 * Extract a field from the State Machine Context data
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#wait-token-contextobject
 *
 * @deprecated replaced by `JsonPath`
 */
class Context {
    /**
     * Instead of using a literal string, get the value from a JSON path
     */
    static stringAt(path) {
        validateContextPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * Instead of using a literal number, get the value from a JSON path
     */
    static numberAt(path) {
        validateContextPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * Return the Task Token field
     *
     * External actions will need this token to report step completion
     * back to StepFunctions using the `SendTaskSuccess` or `SendTaskFailure`
     * calls.
     */
    static get taskToken() {
        return new json_path_1.JsonPathToken('$$.Task.Token').toString();
    }
    /**
     * Use the entire context data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entireContext() {
        return new json_path_1.JsonPathToken('$$').toString();
    }
    constructor() { }
}
exports.Context = Context;
/**
 * Helper functions to work with structures containing fields
 */
class FieldUtils {
    /**
     * Render a JSON structure containing fields to the right StepFunctions structure
     */
    static renderObject(obj) {
        return json_path_1.renderObject(obj);
    }
    /**
     * Return all JSON paths used in the given structure
     */
    static findReferencedPaths(obj) {
        return Array.from(json_path_1.findReferencedPaths(obj)).sort();
    }
    /**
     * Returns whether the given task structure contains the TaskToken field anywhere
     *
     * The field is considered included if the field itself or one of its containing
     * fields occurs anywhere in the payload.
     */
    static containsTaskToken(obj) {
        const paths = json_path_1.findReferencedPaths(obj);
        return paths.has('$$.Task.Token') || paths.has('$$.Task') || paths.has('$$');
    }
    constructor() { }
}
exports.FieldUtils = FieldUtils;
function validateJsonPath(path) {
    if (path !== '$' && !path.startsWith('$.') && path !== '$$' && !path.startsWith('$$.')) {
        throw new Error(`JSON path values must be exactly '$', '$$', start with '$.' or start with '$$.' Received: ${path}`);
    }
}
function validateDataPath(path) {
    if (path !== '$' && !path.startsWith('$.')) {
        throw new Error("Data JSON path values must either be exactly equal to '$' or start with '$.'");
    }
}
function validateContextPath(path) {
    if (path !== '$$' && !path.startsWith('$$.')) {
        throw new Error("Context JSON path values must either be exactly equal to '$$' or start with '$$.'");
    }
}
//# sourceMappingURL=data:application/json;base64,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