"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssembleWith = exports.SplitType = exports.BatchStrategy = exports.CompressionType = exports.InputMode = exports.RecordWrapperType = exports.S3DataDistributionType = exports.S3DataType = exports.DockerImage = exports.S3Location = void 0;
const aws_ecr_assets_1 = require("../../../aws-ecr-assets"); // Automatically re-written from '@aws-cdk/aws-ecr-assets'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
/**
 * Constructs `IS3Location` objects.
 *
 * @experimental
 */
class S3Location {
    /**
     * An `IS3Location` built with a determined bucket and key prefix.
     *
     * @param bucket    is the bucket where the objects are to be stored.
     * @param keyPrefix is the key prefix used by the location.
     */
    static fromBucket(bucket, keyPrefix) {
        return new StandardS3Location({ bucket, keyPrefix, uri: bucket.urlForObject(keyPrefix) });
    }
    /**
     * An `IS3Location` determined fully by a JSON Path from the task input.
     *
     * Due to the dynamic nature of those locations, the IAM grants that will be set by `grantRead` and `grantWrite`
     * apply to the `*` resource.
     *
     * @param expression the JSON expression resolving to an S3 location URI.
     */
    static fromJsonExpression(expression) {
        return new StandardS3Location({ uri: sfn.JsonPath.stringAt(expression) });
    }
}
exports.S3Location = S3Location;
/**
 * Creates `IDockerImage` instances.
 *
 * @experimental
 */
class DockerImage {
    /**
     * Reference a Docker image stored in an ECR repository.
     *
     * @param repository the ECR repository where the image is hosted.
     * @param tag an optional `tag`
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new StandardDockerImage({ repository, imageUri: repository.repositoryUriForTag(tag) });
    }
    /**
     * Reference a Docker image which URI is obtained from the task's input.
     *
     * @param expression           the JSON path expression with the task input.
     * @param allowAnyEcrImagePull whether ECR access should be permitted (set to `false` if the image will never be in ECR).
     */
    static fromJsonExpression(expression, allowAnyEcrImagePull = true) {
        return new StandardDockerImage({ imageUri: expression, allowAnyEcrImagePull });
    }
    /**
     * Reference a Docker image by it's URI.
     *
     * When referencing ECR images, prefer using `inEcr`.
     *
     * @param imageUri the URI to the docker image.
     */
    static fromRegistry(imageUri) {
        return new StandardDockerImage({ imageUri });
    }
    /**
     * Reference a Docker image that is provided as an Asset in the current app.
     *
     * @param scope the scope in which to create the Asset.
     * @param id    the ID for the asset in the construct tree.
     * @param props the configuration props of the asset.
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new StandardDockerImage({ repository: asset.repository, imageUri: asset.imageUri });
    }
}
exports.DockerImage = DockerImage;
/**
 * S3 Data Type.
 *
 * @experimental
 */
var S3DataType;
(function (S3DataType) {
    /**
     * Manifest File Data Type
     */
    S3DataType["MANIFEST_FILE"] = "ManifestFile";
    /**
     * S3 Prefix Data Type
     */
    S3DataType["S3_PREFIX"] = "S3Prefix";
    /**
     * Augmented Manifest File Data Type
     */
    S3DataType["AUGMENTED_MANIFEST_FILE"] = "AugmentedManifestFile";
})(S3DataType = exports.S3DataType || (exports.S3DataType = {}));
/**
 * S3 Data Distribution Type.
 *
 * @experimental
 */
var S3DataDistributionType;
(function (S3DataDistributionType) {
    /**
     * Fully replicated S3 Data Distribution Type
     */
    S3DataDistributionType["FULLY_REPLICATED"] = "FullyReplicated";
    /**
     * Sharded By S3 Key Data Distribution Type
     */
    S3DataDistributionType["SHARDED_BY_S3_KEY"] = "ShardedByS3Key";
})(S3DataDistributionType = exports.S3DataDistributionType || (exports.S3DataDistributionType = {}));
/**
 * Define the format of the input data.
 *
 * @experimental
 */
var RecordWrapperType;
(function (RecordWrapperType) {
    /**
     * None record wrapper type
     */
    RecordWrapperType["NONE"] = "None";
    /**
     * RecordIO record wrapper type
     */
    RecordWrapperType["RECORD_IO"] = "RecordIO";
})(RecordWrapperType = exports.RecordWrapperType || (exports.RecordWrapperType = {}));
/**
 *  Input mode that the algorithm supports.
 *
 * @experimental
 */
var InputMode;
(function (InputMode) {
    /**
     * Pipe mode
     */
    InputMode["PIPE"] = "Pipe";
    /**
     * File mode.
     */
    InputMode["FILE"] = "File";
})(InputMode = exports.InputMode || (exports.InputMode = {}));
/**
 * Compression type of the data.
 *
 * @experimental
 */
var CompressionType;
(function (CompressionType) {
    /**
     * None compression type
     */
    CompressionType["NONE"] = "None";
    /**
     * Gzip compression type
     */
    CompressionType["GZIP"] = "Gzip";
})(CompressionType = exports.CompressionType || (exports.CompressionType = {}));
/**
 * Specifies the number of records to include in a mini-batch for an HTTP inference request.
 *
 * @experimental
 */
var BatchStrategy;
(function (BatchStrategy) {
    /**
     * Fits multiple records in a mini-batch.
     */
    BatchStrategy["MULTI_RECORD"] = "MultiRecord";
    /**
     * Use a single record when making an invocation request.
     */
    BatchStrategy["SINGLE_RECORD"] = "SingleRecord";
})(BatchStrategy = exports.BatchStrategy || (exports.BatchStrategy = {}));
/**
 * Method to use to split the transform job's data files into smaller batches.
 *
 * @experimental
 */
var SplitType;
(function (SplitType) {
    /**
     * Input data files are not split,
     */
    SplitType["NONE"] = "None";
    /**
     * Split records on a newline character boundary.
     */
    SplitType["LINE"] = "Line";
    /**
     * Split using MXNet RecordIO format.
     */
    SplitType["RECORD_IO"] = "RecordIO";
    /**
     * Split using TensorFlow TFRecord format.
     */
    SplitType["TF_RECORD"] = "TFRecord";
})(SplitType = exports.SplitType || (exports.SplitType = {}));
/**
 * How to assemble the results of the transform job as a single S3 object.
 *
 * @experimental
 */
var AssembleWith;
(function (AssembleWith) {
    /**
     * Concatenate the results in binary format.
     */
    AssembleWith["NONE"] = "None";
    /**
     * Add a newline character at the end of every transformed record.
     */
    AssembleWith["LINE"] = "Line";
})(AssembleWith = exports.AssembleWith || (exports.AssembleWith = {}));
class StandardDockerImage extends DockerImage {
    constructor(opts) {
        super();
        this.allowAnyEcrImagePull = !!opts.allowAnyEcrImagePull;
        this.imageUri = opts.imageUri;
        this.repository = opts.repository;
    }
    bind(task) {
        if (this.repository) {
            this.repository.grantPull(task);
        }
        if (this.allowAnyEcrImagePull) {
            task.grantPrincipal.addToPolicy(new iam.PolicyStatement({
                actions: [
                    'ecr:BatchCheckLayerAvailability',
                    'ecr:GetDownloadUrlForLayer',
                    'ecr:BatchGetImage',
                ],
                resources: ['*'],
            }));
        }
        return {
            imageUri: this.imageUri,
        };
    }
}
class StandardS3Location extends S3Location {
    constructor(opts) {
        super();
        this.bucket = opts.bucket;
        this.keyGlob = `${opts.keyPrefix || ''}*`;
        this.uri = opts.uri;
    }
    bind(task, opts) {
        if (this.bucket) {
            if (opts.forReading) {
                this.bucket.grantRead(task, this.keyGlob);
            }
            if (opts.forWriting) {
                this.bucket.grantWrite(task, this.keyGlob);
            }
        }
        else {
            const actions = new Array();
            if (opts.forReading) {
                actions.push('s3:GetObject', 's3:ListBucket');
            }
            if (opts.forWriting) {
                actions.push('s3:PutObject');
            }
            task.grantPrincipal.addToPolicy(new iam.PolicyStatement({ actions, resources: ['*'] }));
        }
        return { uri: this.uri };
    }
}
//# sourceMappingURL=data:application/json;base64,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