"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsRunTask = exports.EcsEc2LaunchTarget = exports.EcsFargateLaunchTarget = void 0;
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const ecs = require("../../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
/**
 * Configuration for running an ECS task on Fargate
 *
 * @see https://docs.aws.amazon.com/AmazonECS/latest/userguide/launch_types.html#launch-type-fargate
 */
class EcsFargateLaunchTarget {
    constructor(options) {
        this.options = options;
    }
    /**
     * Called when the Fargate launch type configured on RunTask
     */
    bind(_task, launchTargetOptions) {
        var _a;
        if (!launchTargetOptions.taskDefinition.isFargateCompatible) {
            throw new Error('Supplied TaskDefinition is not compatible with Fargate');
        }
        return {
            parameters: {
                LaunchType: 'FARGATE',
                PlatformVersion: (_a = this.options) === null || _a === void 0 ? void 0 : _a.platformVersion,
            },
        };
    }
}
exports.EcsFargateLaunchTarget = EcsFargateLaunchTarget;
/**
 * Configuration for running an ECS task on EC2
 *
 * @see https://docs.aws.amazon.com/AmazonECS/latest/userguide/launch_types.html#launch-type-ec2
 */
class EcsEc2LaunchTarget {
    constructor(options) {
        this.options = options;
    }
    /**
     * Called when the EC2 launch type is configured on RunTask
     */
    bind(_task, launchTargetOptions) {
        var _a, _b, _c, _d, _e;
        if (!launchTargetOptions.taskDefinition.isEc2Compatible) {
            throw new Error('Supplied TaskDefinition is not compatible with EC2');
        }
        if (!((_a = launchTargetOptions.cluster) === null || _a === void 0 ? void 0 : _a.hasEc2Capacity)) {
            throw new Error('Cluster for this service needs Ec2 capacity. Call addCapacity() on the cluster.');
        }
        return {
            parameters: {
                LaunchType: 'EC2',
                // takes an array of placement constraints each of which contain a single item array of constraints, flattens it
                // and renders the Json to be passed as a parameter in the state machine.
                // input: [ecs.PlacementConstraint.distinctInstances()] - distinctInstances() returns [{ type: 'distinctInstance' }]
                // output: {Type: 'distinctInstance'}
                PlacementConstraints: noEmpty(flatten(((_c = (_b = this.options) === null || _b === void 0 ? void 0 : _b.placementConstraints) !== null && _c !== void 0 ? _c : []).map((c) => c.toJson().map(uppercaseKeys)))),
                PlacementStrategy: noEmpty(flatten(((_e = (_d = this.options) === null || _d === void 0 ? void 0 : _d.placementStrategies) !== null && _e !== void 0 ? _e : []).map((c) => c.toJson().map(uppercaseKeys)))),
            },
        };
        function uppercaseKeys(obj) {
            const ret = {};
            for (const key of Object.keys(obj)) {
                ret[key.slice(0, 1).toUpperCase() + key.slice(1)] = obj[key];
            }
            return ret;
        }
        function flatten(xs) {
            return Array.prototype.concat([], ...xs);
        }
        function noEmpty(xs) {
            if (xs.length === 0) {
                return undefined;
            }
            return xs;
        }
    }
}
exports.EcsEc2LaunchTarget = EcsEc2LaunchTarget;
/**
 * Run a Task on ECS or Fargate
 */
class EcsRunTask extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id, props);
        this.props = props;
        /**
         * Manage allowed network traffic for this service
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, EcsRunTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.containerOverrides)) {
            throw new Error('Task Token is required in `containerOverrides` for callback. Use Context.taskToken to set the token.');
        }
        if (!this.props.taskDefinition.defaultContainer) {
            throw new Error('A TaskDefinition must have at least one essential container');
        }
        if (this.props.taskDefinition.networkMode === ecs.NetworkMode.AWS_VPC) {
            this.configureAwsVpcNetworking();
        }
        else {
            // Either None, Bridge or Host networking. Copy SecurityGroup from ASG.
            this.validateNoNetworkingProps();
            this.connections.addSecurityGroup(...this.props.cluster.connections.securityGroups);
        }
        for (const override of (_b = this.props.containerOverrides) !== null && _b !== void 0 ? _b : []) {
            const name = override.containerDefinition.containerName;
            if (!cdk.Token.isUnresolved(name)) {
                const cont = this.props.taskDefinition.node.tryFindChild(name);
                if (!cont) {
                    throw new Error(`Overrides mention container with name '${name}', but no such container in task definition`);
                }
            }
        }
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('ecs', 'runTask', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                Cluster: this.props.cluster.clusterArn,
                TaskDefinition: this.props.taskDefinition.taskDefinitionArn,
                NetworkConfiguration: this.networkConfiguration,
                Overrides: renderOverrides(this.props.containerOverrides),
                ...this.props.launchTarget.bind(this, { taskDefinition: this.props.taskDefinition, cluster: this.props.cluster }).parameters,
            }),
        };
    }
    configureAwsVpcNetworking() {
        var _a, _b;
        const subnetSelection = (_a = this.props.subnets) !== null && _a !== void 0 ? _a : { subnetType: this.props.assignPublicIp ? ec2.SubnetType.PUBLIC : ec2.SubnetType.PRIVATE };
        this.networkConfiguration = {
            AwsvpcConfiguration: {
                AssignPublicIp: this.props.assignPublicIp ? (this.props.assignPublicIp ? 'ENABLED' : 'DISABLED') : undefined,
                Subnets: this.props.cluster.vpc.selectSubnets(subnetSelection).subnetIds,
                SecurityGroups: cdk.Lazy.listValue({ produce: () => { var _a; return (_a = this.securityGroups) === null || _a === void 0 ? void 0 : _a.map(sg => sg.securityGroupId); } }),
            },
        };
        // Make sure we have a security group if we're using AWSVPC networking
        this.securityGroups = (_b = this.props.securityGroups) !== null && _b !== void 0 ? _b : [new ec2.SecurityGroup(this, 'SecurityGroup', { vpc: this.props.cluster.vpc })];
        this.connections.addSecurityGroup(...this.securityGroups);
    }
    validateNoNetworkingProps() {
        if (this.props.subnets !== undefined || this.props.securityGroups !== undefined) {
            throw new Error(`Supplied TaskDefinition must have 'networkMode' of 'AWS_VPC' to use 'vpcSubnets' and 'securityGroup'. Received: ${this.props.taskDefinition.networkMode}`);
        }
    }
    makePolicyStatements() {
        const stack = cdk.Stack.of(this);
        // https://docs.aws.amazon.com/step-functions/latest/dg/ecs-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['ecs:RunTask'],
                resources: [this.props.taskDefinition.taskDefinitionArn],
            }),
            new iam.PolicyStatement({
                actions: ['ecs:StopTask', 'ecs:DescribeTasks'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: this.taskExecutionRoles().map((r) => r.roleArn),
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForECSTaskRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
    taskExecutionRoles() {
        // Need to be able to pass both Task and Execution role, apparently
        const ret = new Array();
        ret.push(this.props.taskDefinition.taskRole);
        if (this.props.taskDefinition.executionRole) {
            ret.push(this.props.taskDefinition.executionRole);
        }
        return ret;
    }
}
exports.EcsRunTask = EcsRunTask;
EcsRunTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
function renderOverrides(containerOverrides) {
    var _a;
    if (!containerOverrides || containerOverrides.length === 0) {
        return undefined;
    }
    const ret = new Array();
    for (const override of containerOverrides) {
        ret.push({
            Name: override.containerDefinition.containerName,
            Command: override.command,
            Cpu: override.cpu,
            Memory: override.memoryLimit,
            MemoryReservation: override.memoryReservation,
            Environment: (_a = override.environment) === null || _a === void 0 ? void 0 : _a.map((e) => ({
                Name: e.name,
                Value: e.value,
            })),
        });
    }
    return { ContainerOverrides: ret };
}
//# sourceMappingURL=data:application/json;base64,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