"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Queue = exports.QueueEncryption = void 0;
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const queue_base_1 = require("./queue-base");
const sqs_generated_1 = require("./sqs.generated");
const validate_props_1 = require("./validate-props");
/**
 * What kind of encryption to apply to this queue
 */
var QueueEncryption;
(function (QueueEncryption) {
    /**
     * Messages in the queue are not encrypted
     */
    QueueEncryption["UNENCRYPTED"] = "NONE";
    /**
     * Server-side KMS encryption with a master key managed by SQS.
     */
    QueueEncryption["KMS_MANAGED"] = "MANAGED";
    /**
     * Server-side encryption with a KMS key managed by the user.
     *
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    QueueEncryption["KMS"] = "KMS";
})(QueueEncryption = exports.QueueEncryption || (exports.QueueEncryption = {}));
/**
 * A new Amazon SQS queue
 */
class Queue extends queue_base_1.QueueBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.queueName,
        });
        this.autoCreatePolicy = true;
        validate_props_1.validateProps(props);
        const redrivePolicy = props.deadLetterQueue
            ? {
                deadLetterTargetArn: props.deadLetterQueue.queue.queueArn,
                maxReceiveCount: props.deadLetterQueue.maxReceiveCount,
            }
            : undefined;
        const { encryptionMasterKey, encryptionProps } = _determineEncryptionProps.call(this);
        const fifoProps = this.determineFifoProps(props);
        this.fifo = fifoProps.fifoQueue || false;
        const queue = new sqs_generated_1.CfnQueue(this, 'Resource', {
            queueName: this.physicalName,
            ...fifoProps,
            ...encryptionProps,
            redrivePolicy,
            delaySeconds: props.deliveryDelay && props.deliveryDelay.toSeconds(),
            maximumMessageSize: props.maxMessageSizeBytes,
            messageRetentionPeriod: props.retentionPeriod && props.retentionPeriod.toSeconds(),
            receiveMessageWaitTimeSeconds: props.receiveMessageWaitTime && props.receiveMessageWaitTime.toSeconds(),
            visibilityTimeout: props.visibilityTimeout && props.visibilityTimeout.toSeconds(),
        });
        this.queueArn = this.getResourceArnAttribute(queue.attrArn, {
            service: 'sqs',
            resource: this.physicalName,
        });
        this.queueName = this.getResourceNameAttribute(queue.attrQueueName);
        this.encryptionMasterKey = encryptionMasterKey;
        this.queueUrl = queue.ref;
        function _determineEncryptionProps() {
            let encryption = props.encryption || QueueEncryption.UNENCRYPTED;
            if (encryption !== QueueEncryption.KMS && props.encryptionMasterKey) {
                encryption = QueueEncryption.KMS; // KMS is implied by specifying an encryption key
            }
            if (encryption === QueueEncryption.UNENCRYPTED) {
                return { encryptionProps: {} };
            }
            if (encryption === QueueEncryption.KMS_MANAGED) {
                return {
                    encryptionProps: {
                        kmsMasterKeyId: 'alias/aws/sqs',
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds(),
                    },
                };
            }
            if (encryption === QueueEncryption.KMS) {
                const masterKey = props.encryptionMasterKey || new kms.Key(this, 'Key', {
                    description: `Created by ${this.node.path}`,
                });
                return {
                    encryptionMasterKey: masterKey,
                    encryptionProps: {
                        kmsMasterKeyId: masterKey.keyArn,
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds(),
                    },
                };
            }
            throw new Error(`Unexpected 'encryptionType': ${encryption}`);
        }
    }
    /**
     * Import an existing SQS queue provided an ARN
     *
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param queueArn queue ARN (i.e. arn:aws:sqs:us-east-2:444455556666:queue1)
     */
    static fromQueueArn(scope, id, queueArn) {
        return Queue.fromQueueAttributes(scope, id, { queueArn });
    }
    /**
     * Import an existing queue
     */
    static fromQueueAttributes(scope, id, attrs) {
        const stack = core_1.Stack.of(scope);
        const queueName = attrs.queueName || stack.parseArn(attrs.queueArn).resource;
        const queueUrl = attrs.queueUrl || `https://sqs.${stack.region}.${stack.urlSuffix}/${stack.account}/${queueName}`;
        class Import extends queue_base_1.QueueBase {
            constructor() {
                super(...arguments);
                this.queueArn = attrs.queueArn; // arn:aws:sqs:us-east-1:123456789012:queue1
                this.queueUrl = queueUrl;
                this.queueName = queueName;
                this.encryptionMasterKey = attrs.keyArn
                    ? kms.Key.fromKeyArn(this, 'Key', attrs.keyArn)
                    : undefined;
                this.fifo = queueName.endsWith('.fifo') ? true : false;
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Look at the props, see if the FIFO props agree, and return the correct subset of props
     */
    determineFifoProps(props) {
        // Check if any of the signals that we have say that this is a FIFO queue.
        let fifoQueue = props.fifo;
        const queueName = props.queueName;
        if (typeof fifoQueue === 'undefined' && queueName && !core_1.Token.isUnresolved(queueName) && queueName.endsWith('.fifo')) {
            fifoQueue = true;
        }
        if (typeof fifoQueue === 'undefined' && props.contentBasedDeduplication) {
            fifoQueue = true;
        }
        // If we have a name, see that it agrees with the FIFO setting
        if (typeof queueName === 'string') {
            if (fifoQueue && !queueName.endsWith('.fifo')) {
                throw new Error("FIFO queue names must end in '.fifo'");
            }
            if (!fifoQueue && queueName.endsWith('.fifo')) {
                throw new Error("Non-FIFO queue name may not end in '.fifo'");
            }
        }
        if (props.contentBasedDeduplication && !fifoQueue) {
            throw new Error('Content-based deduplication can only be defined for FIFO queues');
        }
        return {
            contentBasedDeduplication: props.contentBasedDeduplication,
            fifoQueue,
        };
    }
}
exports.Queue = Queue;
//# sourceMappingURL=data:application/json;base64,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