"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.QueueBase = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const policy_1 = require("./policy");
/**
 * Reference to a new or existing Amazon SQS queue
 */
class QueueBase extends core_1.Resource {
    /**
     * Adds a statement to the IAM resource policy associated with this queue.
     *
     * If this queue was created in this stack (`new Queue`), a queue policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the queue is improted (`Queue.import`), then this is a no-op.
     */
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.QueuePolicy(this, 'Policy', { queues: [this] });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    /**
     * Grant permissions to consume messages from a queue
     *
     * This will grant the following permissions:
     *
     *   - sqs:ChangeMessageVisibility
     *   - sqs:DeleteMessage
     *   - sqs:ReceiveMessage
     *   - sqs:GetQueueAttributes
     *   - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant consume rights to
     */
    grantConsumeMessages(grantee) {
        const ret = this.grant(grantee, 'sqs:ReceiveMessage', 'sqs:ChangeMessageVisibility', 'sqs:GetQueueUrl', 'sqs:DeleteMessage', 'sqs:GetQueueAttributes');
        if (this.encryptionMasterKey) {
            this.encryptionMasterKey.grantDecrypt(grantee);
        }
        return ret;
    }
    /**
     * Grant access to send messages to a queue to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - sqs:SendMessage
     *  - sqs:GetQueueAttributes
     *  - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to
     */
    grantSendMessages(grantee) {
        const ret = this.grant(grantee, 'sqs:SendMessage', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
        if (this.encryptionMasterKey) {
            // kms:Decrypt necessary to execute grantsendMessages to an SSE enabled SQS queue
            this.encryptionMasterKey.grantEncryptDecrypt(grantee);
        }
        return ret;
    }
    /**
     * Grant an IAM principal permissions to purge all messages from the queue.
     *
     * This will grant the following permissions:
     *
     *  - sqs:PurgeQueue
     *  - sqs:GetQueueAttributes
     *  - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to
     */
    grantPurge(grantee) {
        return this.grant(grantee, 'sqs:PurgeQueue', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
    }
    /**
     * Grant the actions defined in queueActions to the identity Principal given
     * on this SQS queue resource.
     *
     * @param grantee Principal to grant right to
     * @param actions The actions to grant
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions,
            resourceArns: [this.queueArn],
            resource: this,
        });
    }
}
exports.QueueBase = QueueBase;
//# sourceMappingURL=data:application/json;base64,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