"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TopicBase = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const policy_1 = require("./policy");
const subscription_1 = require("./subscription");
/**
 * Either a new or imported Topic
 */
class TopicBase extends core_1.Resource {
    /**
     * Subscribe some endpoint to this topic
     */
    addSubscription(subscription) {
        const subscriptionConfig = subscription.bind(this);
        const scope = subscriptionConfig.subscriberScope || this;
        let id = subscriptionConfig.subscriberId;
        if (core_1.Token.isUnresolved(subscriptionConfig.subscriberId)) {
            id = this.nextTokenId(scope);
        }
        // We use the subscriber's id as the construct id. There's no meaning
        // to subscribing the same subscriber twice on the same topic.
        if (scope.node.tryFindChild(id)) {
            throw new Error(`A subscription with id "${id}" already exists under the scope ${scope.node.path}`);
        }
        new subscription_1.Subscription(scope, id, {
            topic: this,
            ...subscriptionConfig,
        });
    }
    /**
     * Adds a statement to the IAM resource policy associated with this topic.
     *
     * If this topic was created in this stack (`new Topic`), a topic policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the topic is improted (`Topic.import`), then this is a no-op.
     */
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.TopicPolicy(this, 'Policy', { topics: [this] });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    /**
     * Grant topic publishing permissions to the given identity
     */
    grantPublish(grantee) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['sns:Publish'],
            resourceArns: [this.topicArn],
            resource: this,
        });
    }
    nextTokenId(scope) {
        let nextSuffix = 1;
        const re = /TokenSubscription:([\d]*)/gm;
        // Search through the construct and all of its children
        // for previous subscriptions that match our regex pattern
        for (const source of scope.node.findAll()) {
            const m = re.exec(source.node.id); // Use regex to find a match
            if (m !== null) { // if we found a match
                const matchSuffix = parseInt(m[1], 10); // get the suffix for that match (as integer)
                if (matchSuffix >= nextSuffix) { // check if the match suffix is larger or equal to currently proposed suffix
                    nextSuffix = matchSuffix + 1; // increment the suffix
                }
            }
        }
        return `TokenSubscription:${nextSuffix}`;
    }
}
exports.TopicBase = TopicBase;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidG9waWMtYmFzZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRvcGljLWJhc2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEscUNBQXFDLENBQUMsbURBQW1EO0FBQ3pGLHFDQUFtRSxDQUFDLGdEQUFnRDtBQUNwSCxxQ0FBdUM7QUFFdkMsaURBQThDO0FBa0M5Qzs7R0FFRztBQUNILE1BQXNCLFNBQVUsU0FBUSxlQUFRO0lBVTVDOztPQUVHO0lBQ0ksZUFBZSxDQUFDLFlBQWdDO1FBQ25ELE1BQU0sa0JBQWtCLEdBQUcsWUFBWSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNuRCxNQUFNLEtBQUssR0FBRyxrQkFBa0IsQ0FBQyxlQUFlLElBQUksSUFBSSxDQUFDO1FBQ3pELElBQUksRUFBRSxHQUFHLGtCQUFrQixDQUFDLFlBQVksQ0FBQztRQUN6QyxJQUFJLFlBQUssQ0FBQyxZQUFZLENBQUMsa0JBQWtCLENBQUMsWUFBWSxDQUFDLEVBQUU7WUFDckQsRUFBRSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDLENBQUM7U0FDaEM7UUFDRCxxRUFBcUU7UUFDckUsOERBQThEO1FBQzlELElBQUksS0FBSyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRSxDQUFDLEVBQUU7WUFDN0IsTUFBTSxJQUFJLEtBQUssQ0FBQywyQkFBMkIsRUFBRSxvQ0FBb0MsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1NBQ3ZHO1FBQ0QsSUFBSSwyQkFBWSxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDeEIsS0FBSyxFQUFFLElBQUk7WUFDWCxHQUFHLGtCQUFrQjtTQUN4QixDQUFDLENBQUM7SUFDUCxDQUFDO0lBQ0Q7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CLENBQUMsU0FBOEI7UUFDckQsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLElBQUksSUFBSSxDQUFDLGdCQUFnQixFQUFFO1lBQ3ZDLElBQUksQ0FBQyxNQUFNLEdBQUcsSUFBSSxvQkFBVyxDQUFDLElBQUksRUFBRSxRQUFRLEVBQUUsRUFBRSxNQUFNLEVBQUUsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7U0FDckU7UUFDRCxJQUFJLElBQUksQ0FBQyxNQUFNLEVBQUU7WUFDYixJQUFJLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxhQUFhLENBQUMsU0FBUyxDQUFDLENBQUM7WUFDOUMsT0FBTyxFQUFFLGNBQWMsRUFBRSxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLE1BQU0sRUFBRSxDQUFDO1NBQ2xFO1FBQ0QsT0FBTyxFQUFFLGNBQWMsRUFBRSxLQUFLLEVBQUUsQ0FBQztJQUNyQyxDQUFDO0lBQ0Q7O09BRUc7SUFDSSxZQUFZLENBQUMsT0FBdUI7UUFDdkMsT0FBTyxHQUFHLENBQUMsS0FBSyxDQUFDLHdCQUF3QixDQUFDO1lBQ3RDLE9BQU87WUFDUCxPQUFPLEVBQUUsQ0FBQyxhQUFhLENBQUM7WUFDeEIsWUFBWSxFQUFFLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQztZQUM3QixRQUFRLEVBQUUsSUFBSTtTQUNqQixDQUFDLENBQUM7SUFDUCxDQUFDO0lBQ08sV0FBVyxDQUFDLEtBQWdCO1FBQ2hDLElBQUksVUFBVSxHQUFHLENBQUMsQ0FBQztRQUNuQixNQUFNLEVBQUUsR0FBRyw2QkFBNkIsQ0FBQztRQUN6Qyx1REFBdUQ7UUFDdkQsMERBQTBEO1FBQzFELEtBQUssTUFBTSxNQUFNLElBQUksS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsRUFBRTtZQUN2QyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyw0QkFBNEI7WUFDL0QsSUFBSSxDQUFDLEtBQUssSUFBSSxFQUFFLEVBQUUsc0JBQXNCO2dCQUNwQyxNQUFNLFdBQVcsR0FBRyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsNkNBQTZDO2dCQUNyRixJQUFJLFdBQVcsSUFBSSxVQUFVLEVBQUUsRUFBRSw0RUFBNEU7b0JBQ3pHLFVBQVUsR0FBRyxXQUFXLEdBQUcsQ0FBQyxDQUFDLENBQUMsdUJBQXVCO2lCQUN4RDthQUNKO1NBQ0o7UUFDRCxPQUFPLHFCQUFxQixVQUFVLEVBQUUsQ0FBQztJQUM3QyxDQUFDO0NBQ0o7QUExRUQsOEJBMEVDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgaWFtIGZyb20gXCIuLi8uLi9hd3MtaWFtXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtaWFtJ1xuaW1wb3J0IHsgQ29uc3RydWN0LCBJUmVzb3VyY2UsIFJlc291cmNlLCBUb2tlbiB9IGZyb20gXCIuLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0IHsgVG9waWNQb2xpY3kgfSBmcm9tICcuL3BvbGljeSc7XG5pbXBvcnQgeyBJVG9waWNTdWJzY3JpcHRpb24gfSBmcm9tICcuL3N1YnNjcmliZXInO1xuaW1wb3J0IHsgU3Vic2NyaXB0aW9uIH0gZnJvbSAnLi9zdWJzY3JpcHRpb24nO1xuLyoqXG4gKiBSZXByZXNlbnRzIGFuIFNOUyB0b3BpY1xuICovXG5leHBvcnQgaW50ZXJmYWNlIElUb3BpYyBleHRlbmRzIElSZXNvdXJjZSB7XG4gICAgLyoqXG4gICAgICogVGhlIEFSTiBvZiB0aGUgdG9waWNcbiAgICAgKlxuICAgICAqIEBhdHRyaWJ1dGVcbiAgICAgKi9cbiAgICByZWFkb25seSB0b3BpY0Fybjogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBuYW1lIG9mIHRoZSB0b3BpY1xuICAgICAqXG4gICAgICogQGF0dHJpYnV0ZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHRvcGljTmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFN1YnNjcmliZSBzb21lIGVuZHBvaW50IHRvIHRoaXMgdG9waWNcbiAgICAgKi9cbiAgICBhZGRTdWJzY3JpcHRpb24oc3Vic2NyaXB0aW9uOiBJVG9waWNTdWJzY3JpcHRpb24pOiB2b2lkO1xuICAgIC8qKlxuICAgICAqIEFkZHMgYSBzdGF0ZW1lbnQgdG8gdGhlIElBTSByZXNvdXJjZSBwb2xpY3kgYXNzb2NpYXRlZCB3aXRoIHRoaXMgdG9waWMuXG4gICAgICpcbiAgICAgKiBJZiB0aGlzIHRvcGljIHdhcyBjcmVhdGVkIGluIHRoaXMgc3RhY2sgKGBuZXcgVG9waWNgKSwgYSB0b3BpYyBwb2xpY3lcbiAgICAgKiB3aWxsIGJlIGF1dG9tYXRpY2FsbHkgY3JlYXRlZCB1cG9uIHRoZSBmaXJzdCBjYWxsIHRvIGBhZGRUb1BvbGljeWAuIElmXG4gICAgICogdGhlIHRvcGljIGlzIGltcHJvdGVkIChgVG9waWMuaW1wb3J0YCksIHRoZW4gdGhpcyBpcyBhIG5vLW9wLlxuICAgICAqL1xuICAgIGFkZFRvUmVzb3VyY2VQb2xpY3koc3RhdGVtZW50OiBpYW0uUG9saWN5U3RhdGVtZW50KTogaWFtLkFkZFRvUmVzb3VyY2VQb2xpY3lSZXN1bHQ7XG4gICAgLyoqXG4gICAgICogR3JhbnQgdG9waWMgcHVibGlzaGluZyBwZXJtaXNzaW9ucyB0byB0aGUgZ2l2ZW4gaWRlbnRpdHlcbiAgICAgKi9cbiAgICBncmFudFB1Ymxpc2goaWRlbnRpdHk6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50O1xufVxuLyoqXG4gKiBFaXRoZXIgYSBuZXcgb3IgaW1wb3J0ZWQgVG9waWNcbiAqL1xuZXhwb3J0IGFic3RyYWN0IGNsYXNzIFRvcGljQmFzZSBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVRvcGljIHtcbiAgICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgdG9waWNBcm46IHN0cmluZztcbiAgICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgdG9waWNOYW1lOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogQ29udHJvbHMgYXV0b21hdGljIGNyZWF0aW9uIG9mIHBvbGljeSBvYmplY3RzLlxuICAgICAqXG4gICAgICogU2V0IGJ5IHN1YmNsYXNzZXMuXG4gICAgICovXG4gICAgcHJvdGVjdGVkIGFic3RyYWN0IHJlYWRvbmx5IGF1dG9DcmVhdGVQb2xpY3k6IGJvb2xlYW47XG4gICAgcHJpdmF0ZSBwb2xpY3k/OiBUb3BpY1BvbGljeTtcbiAgICAvKipcbiAgICAgKiBTdWJzY3JpYmUgc29tZSBlbmRwb2ludCB0byB0aGlzIHRvcGljXG4gICAgICovXG4gICAgcHVibGljIGFkZFN1YnNjcmlwdGlvbihzdWJzY3JpcHRpb246IElUb3BpY1N1YnNjcmlwdGlvbikge1xuICAgICAgICBjb25zdCBzdWJzY3JpcHRpb25Db25maWcgPSBzdWJzY3JpcHRpb24uYmluZCh0aGlzKTtcbiAgICAgICAgY29uc3Qgc2NvcGUgPSBzdWJzY3JpcHRpb25Db25maWcuc3Vic2NyaWJlclNjb3BlIHx8IHRoaXM7XG4gICAgICAgIGxldCBpZCA9IHN1YnNjcmlwdGlvbkNvbmZpZy5zdWJzY3JpYmVySWQ7XG4gICAgICAgIGlmIChUb2tlbi5pc1VucmVzb2x2ZWQoc3Vic2NyaXB0aW9uQ29uZmlnLnN1YnNjcmliZXJJZCkpIHtcbiAgICAgICAgICAgIGlkID0gdGhpcy5uZXh0VG9rZW5JZChzY29wZSk7XG4gICAgICAgIH1cbiAgICAgICAgLy8gV2UgdXNlIHRoZSBzdWJzY3JpYmVyJ3MgaWQgYXMgdGhlIGNvbnN0cnVjdCBpZC4gVGhlcmUncyBubyBtZWFuaW5nXG4gICAgICAgIC8vIHRvIHN1YnNjcmliaW5nIHRoZSBzYW1lIHN1YnNjcmliZXIgdHdpY2Ugb24gdGhlIHNhbWUgdG9waWMuXG4gICAgICAgIGlmIChzY29wZS5ub2RlLnRyeUZpbmRDaGlsZChpZCkpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgQSBzdWJzY3JpcHRpb24gd2l0aCBpZCBcIiR7aWR9XCIgYWxyZWFkeSBleGlzdHMgdW5kZXIgdGhlIHNjb3BlICR7c2NvcGUubm9kZS5wYXRofWApO1xuICAgICAgICB9XG4gICAgICAgIG5ldyBTdWJzY3JpcHRpb24oc2NvcGUsIGlkLCB7XG4gICAgICAgICAgICB0b3BpYzogdGhpcyxcbiAgICAgICAgICAgIC4uLnN1YnNjcmlwdGlvbkNvbmZpZyxcbiAgICAgICAgfSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZHMgYSBzdGF0ZW1lbnQgdG8gdGhlIElBTSByZXNvdXJjZSBwb2xpY3kgYXNzb2NpYXRlZCB3aXRoIHRoaXMgdG9waWMuXG4gICAgICpcbiAgICAgKiBJZiB0aGlzIHRvcGljIHdhcyBjcmVhdGVkIGluIHRoaXMgc3RhY2sgKGBuZXcgVG9waWNgKSwgYSB0b3BpYyBwb2xpY3lcbiAgICAgKiB3aWxsIGJlIGF1dG9tYXRpY2FsbHkgY3JlYXRlZCB1cG9uIHRoZSBmaXJzdCBjYWxsIHRvIGBhZGRUb1BvbGljeWAuIElmXG4gICAgICogdGhlIHRvcGljIGlzIGltcHJvdGVkIChgVG9waWMuaW1wb3J0YCksIHRoZW4gdGhpcyBpcyBhIG5vLW9wLlxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRUb1Jlc291cmNlUG9saWN5KHN0YXRlbWVudDogaWFtLlBvbGljeVN0YXRlbWVudCk6IGlhbS5BZGRUb1Jlc291cmNlUG9saWN5UmVzdWx0IHtcbiAgICAgICAgaWYgKCF0aGlzLnBvbGljeSAmJiB0aGlzLmF1dG9DcmVhdGVQb2xpY3kpIHtcbiAgICAgICAgICAgIHRoaXMucG9saWN5ID0gbmV3IFRvcGljUG9saWN5KHRoaXMsICdQb2xpY3knLCB7IHRvcGljczogW3RoaXNdIH0pO1xuICAgICAgICB9XG4gICAgICAgIGlmICh0aGlzLnBvbGljeSkge1xuICAgICAgICAgICAgdGhpcy5wb2xpY3kuZG9jdW1lbnQuYWRkU3RhdGVtZW50cyhzdGF0ZW1lbnQpO1xuICAgICAgICAgICAgcmV0dXJuIHsgc3RhdGVtZW50QWRkZWQ6IHRydWUsIHBvbGljeURlcGVuZGFibGU6IHRoaXMucG9saWN5IH07XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHsgc3RhdGVtZW50QWRkZWQ6IGZhbHNlIH07XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEdyYW50IHRvcGljIHB1Ymxpc2hpbmcgcGVybWlzc2lvbnMgdG8gdGhlIGdpdmVuIGlkZW50aXR5XG4gICAgICovXG4gICAgcHVibGljIGdyYW50UHVibGlzaChncmFudGVlOiBpYW0uSUdyYW50YWJsZSkge1xuICAgICAgICByZXR1cm4gaWFtLkdyYW50LmFkZFRvUHJpbmNpcGFsT3JSZXNvdXJjZSh7XG4gICAgICAgICAgICBncmFudGVlLFxuICAgICAgICAgICAgYWN0aW9uczogWydzbnM6UHVibGlzaCddLFxuICAgICAgICAgICAgcmVzb3VyY2VBcm5zOiBbdGhpcy50b3BpY0Fybl0sXG4gICAgICAgICAgICByZXNvdXJjZTogdGhpcyxcbiAgICAgICAgfSk7XG4gICAgfVxuICAgIHByaXZhdGUgbmV4dFRva2VuSWQoc2NvcGU6IENvbnN0cnVjdCkge1xuICAgICAgICBsZXQgbmV4dFN1ZmZpeCA9IDE7XG4gICAgICAgIGNvbnN0IHJlID0gL1Rva2VuU3Vic2NyaXB0aW9uOihbXFxkXSopL2dtO1xuICAgICAgICAvLyBTZWFyY2ggdGhyb3VnaCB0aGUgY29uc3RydWN0IGFuZCBhbGwgb2YgaXRzIGNoaWxkcmVuXG4gICAgICAgIC8vIGZvciBwcmV2aW91cyBzdWJzY3JpcHRpb25zIHRoYXQgbWF0Y2ggb3VyIHJlZ2V4IHBhdHRlcm5cbiAgICAgICAgZm9yIChjb25zdCBzb3VyY2Ugb2Ygc2NvcGUubm9kZS5maW5kQWxsKCkpIHtcbiAgICAgICAgICAgIGNvbnN0IG0gPSByZS5leGVjKHNvdXJjZS5ub2RlLmlkKTsgLy8gVXNlIHJlZ2V4IHRvIGZpbmQgYSBtYXRjaFxuICAgICAgICAgICAgaWYgKG0gIT09IG51bGwpIHsgLy8gaWYgd2UgZm91bmQgYSBtYXRjaFxuICAgICAgICAgICAgICAgIGNvbnN0IG1hdGNoU3VmZml4ID0gcGFyc2VJbnQobVsxXSwgMTApOyAvLyBnZXQgdGhlIHN1ZmZpeCBmb3IgdGhhdCBtYXRjaCAoYXMgaW50ZWdlcilcbiAgICAgICAgICAgICAgICBpZiAobWF0Y2hTdWZmaXggPj0gbmV4dFN1ZmZpeCkgeyAvLyBjaGVjayBpZiB0aGUgbWF0Y2ggc3VmZml4IGlzIGxhcmdlciBvciBlcXVhbCB0byBjdXJyZW50bHkgcHJvcG9zZWQgc3VmZml4XG4gICAgICAgICAgICAgICAgICAgIG5leHRTdWZmaXggPSBtYXRjaFN1ZmZpeCArIDE7IC8vIGluY3JlbWVudCB0aGUgc3VmZml4XG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIHJldHVybiBgVG9rZW5TdWJzY3JpcHRpb246JHtuZXh0U3VmZml4fWA7XG4gICAgfVxufVxuIl19