"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqsSubscription = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sns = require("../../aws-sns"); // Automatically re-written from '@aws-cdk/aws-sns'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Use an SQS queue as a subscription target
 */
class SqsSubscription {
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
    }
    /**
     * Returns a configuration for an SQS queue to subscribe to an SNS topic
     */
    bind(topic) {
        // Create subscription under *consuming* construct to make sure it ends up
        // in the correct stack in cases of cross-stack subscriptions.
        if (!core_1.Construct.isConstruct(this.queue)) {
            throw new Error('The supplied Queue object must be an instance of Construct');
        }
        // add a statement to the queue resource policy which allows this topic
        // to send messages to the queue.
        this.queue.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.queue.queueArn],
            actions: ['sqs:SendMessage'],
            principals: [new iam.ServicePrincipal('sns.amazonaws.com')],
            conditions: {
                ArnEquals: { 'aws:SourceArn': topic.topicArn },
            },
        }));
        return {
            subscriberScope: this.queue,
            subscriberId: topic.node.uniqueId,
            endpoint: this.queue.queueArn,
            protocol: sns.SubscriptionProtocol.SQS,
            rawMessageDelivery: this.props.rawMessageDelivery,
            filterPolicy: this.props.filterPolicy,
            region: this.regionFromArn(topic),
            deadLetterQueue: this.props.deadLetterQueue,
        };
    }
    regionFromArn(topic) {
        // no need to specify `region` for topics defined within the same stack
        if (topic instanceof sns.Topic) {
            return undefined;
        }
        return core_1.Stack.of(topic).parseArn(topic.topicArn).region;
    }
}
exports.SqsSubscription = SqsSubscription;
//# sourceMappingURL=data:application/json;base64,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