"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3 = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Saves the received message to an Amazon S3 bucket and, optionally, publishes
 * a notification to Amazon SNS.
 */
class S3 {
    constructor(props) {
        this.props = props;
    }
    bind(rule) {
        // Allow SES to write to S3 bucket
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-s3
        const keyPattern = this.props.objectKeyPrefix || '';
        const s3Statement = new iam.PolicyStatement({
            actions: ['s3:PutObject'],
            principals: [new iam.ServicePrincipal('ses.amazonaws.com')],
            resources: [this.props.bucket.arnForObjects(`${keyPattern}*`)],
            conditions: {
                StringEquals: {
                    'aws:Referer': cdk.Aws.ACCOUNT_ID,
                },
            },
        });
        this.props.bucket.addToResourcePolicy(s3Statement);
        const policy = this.props.bucket.node.tryFindChild('Policy');
        if (policy) { // The bucket could be imported
            rule.node.addDependency(policy);
        }
        else {
            rule.node.addWarning('This rule is using a S3 action with an imported bucket. Ensure permission is given to SES to write to that bucket.');
        }
        // Allow SES to use KMS master key
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-kms
        if (this.props.kmsKey && !/alias\/aws\/ses$/.test(this.props.kmsKey.keyArn)) {
            const kmsStatement = new iam.PolicyStatement({
                actions: ['km:Encrypt', 'kms:GenerateDataKey'],
                principals: [new iam.ServicePrincipal('ses.amazonaws.com')],
                resources: ['*'],
                conditions: {
                    Null: {
                        'kms:EncryptionContext:aws:ses:rule-name': 'false',
                        'kms:EncryptionContext:aws:ses:message-id': 'false',
                    },
                    StringEquals: {
                        'kms:EncryptionContext:aws:ses:source-account': cdk.Aws.ACCOUNT_ID,
                    },
                },
            });
            this.props.kmsKey.addToResourcePolicy(kmsStatement);
        }
        return {
            s3Action: {
                bucketName: this.props.bucket.bucketName,
                kmsKeyArn: this.props.kmsKey ? this.props.kmsKey.keyArn : undefined,
                objectKeyPrefix: this.props.objectKeyPrefix,
                topicArn: this.props.topic ? this.props.topic.topicArn : undefined,
            },
        };
    }
}
exports.S3 = S3;
//# sourceMappingURL=data:application/json;base64,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