"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IpInstance = void 0;
const instance_1 = require("./instance");
const service_1 = require("./service");
const servicediscovery_generated_1 = require("./servicediscovery.generated");
/**
 * Instance that is accessible using an IP address.
 *
 * @resource AWS::ServiceDiscovery::Instance
 */
class IpInstance extends instance_1.InstanceBase {
    constructor(scope, id, props) {
        super(scope, id);
        const dnsRecordType = props.service.dnsRecordType;
        if (dnsRecordType === service_1.DnsRecordType.CNAME) {
            throw new Error('Service must support `A`, `AAAA` or `SRV` records to register this instance type.');
        }
        if (dnsRecordType === service_1.DnsRecordType.SRV) {
            if (!props.port) {
                throw new Error('A `port` must be specified for a service using a `SRV` record.');
            }
            if (!props.ipv4 && !props.ipv6) {
                throw new Error('At least `ipv4` or `ipv6` must be specified for a service using a `SRV` record.');
            }
        }
        if (!props.ipv4 && (dnsRecordType === service_1.DnsRecordType.A || dnsRecordType === service_1.DnsRecordType.A_AAAA)) {
            throw new Error('An `ipv4` must be specified for a service using a `A` record.');
        }
        if (!props.ipv6 &&
            (dnsRecordType === service_1.DnsRecordType.AAAA || dnsRecordType === service_1.DnsRecordType.A_AAAA)) {
            throw new Error('An `ipv6` must be specified for a service using a `AAAA` record.');
        }
        const port = props.port || 80;
        const resource = new servicediscovery_generated_1.CfnInstance(this, 'Resource', {
            instanceAttributes: {
                AWS_INSTANCE_IPV4: props.ipv4,
                AWS_INSTANCE_IPV6: props.ipv6,
                AWS_INSTANCE_PORT: port.toString(),
                ...props.customAttributes,
            },
            instanceId: props.instanceId || this.uniqueInstanceId(),
            serviceId: props.service.serviceId,
        });
        this.service = props.service;
        this.instanceId = resource.ref;
        this.ipv4 = props.ipv4 || '';
        this.ipv6 = props.ipv6 || '';
        this.port = port;
    }
}
exports.IpInstance = IpInstance;
//# sourceMappingURL=data:application/json;base64,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