"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cluster = exports.ClusterType = exports.NodeType = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const redshift_generated_1 = require("./redshift.generated");
/**
 * Possible Node Types to use in the cluster
 * used for defining {@link ClusterProps.nodeType}.
 */
var NodeType;
(function (NodeType) {
    /**
     * ds2.xlarge
     */
    NodeType["DS2_XLARGE"] = "ds2.xlarge";
    /**
     * ds2.8xlarge
     */
    NodeType["DS2_8XLARGE"] = "ds2.8xlarge";
    /**
     * dc1.large
     */
    NodeType["DC1_LARGE"] = "dc1.large";
    /**
     * dc1.8xlarge
     */
    NodeType["DC1_8XLARGE"] = "dc1.8xlarge";
    /**
     * dc2.large
     */
    NodeType["DC2_LARGE"] = "dc2.large";
    /**
     * dc2.8xlarge
     */
    NodeType["DC2_8XLARGE"] = "dc2.8xlarge";
    /**
     * ra3.16xlarge
     */
    NodeType["RA3_16XLARGE"] = "ra3.16xlarge";
})(NodeType = exports.NodeType || (exports.NodeType = {}));
/**
 * What cluster type to use.
 * Used by {@link ClusterProps.clusterType}
 */
var ClusterType;
(function (ClusterType) {
    /**
     * single-node cluster, the {@link ClusterProps.numberOfNodes} parameter is not required
     */
    ClusterType["SINGLE_NODE"] = "single-node";
    /**
     * multi-node cluster, set the amount of nodes using {@link ClusterProps.numberOfNodes} parameter
     */
    ClusterType["MULTI_NODE"] = "multi-node";
})(ClusterType = exports.ClusterType || (exports.ClusterType = {}));
/**
 * A new or imported clustered database.
 */
class ClusterBase extends core_1.Resource {
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterName,
            targetType: secretsmanager.AttachmentTargetType.REDSHIFT_CLUSTER,
        };
    }
}
/**
 * Create a Redshift cluster a given number of nodes.
 *
 * @resource AWS::Redshift::Cluster
 */
class Cluster extends ClusterBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.vpc = props.vpc;
        this.vpcSubnets = props.vpcSubnets ? props.vpcSubnets : {
            subnetType: ec2.SubnetType.PRIVATE,
        };
        const removalPolicy = props.removalPolicy ? props.removalPolicy : core_1.RemovalPolicy.RETAIN;
        const { subnetIds } = this.vpc.selectSubnets(this.vpcSubnets);
        const subnetGroup = new redshift_generated_1.CfnClusterSubnetGroup(this, 'Subnets', {
            description: `Subnets for ${id} Redshift cluster`,
            subnetIds,
        });
        subnetGroup.applyRemovalPolicy(removalPolicy, {
            applyToUpdateReplacePolicy: true,
        });
        const securityGroups = props.securityGroups !== undefined ?
            props.securityGroups : [new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'Redshift security group',
                vpc: this.vpc,
                securityGroupName: 'redshift SG',
            })];
        const securityGroupIds = securityGroups.map(sg => sg.securityGroupId);
        let secret;
        if (!props.masterUser.masterPassword) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUser.masterUsername,
                encryptionKey: props.masterUser.encryptionKey,
            });
        }
        const clusterType = props.clusterType || ClusterType.MULTI_NODE;
        const nodeCount = props.numberOfNodes !== undefined ? props.numberOfNodes : (clusterType === ClusterType.MULTI_NODE ? 2 : 1);
        if (clusterType === ClusterType.MULTI_NODE && nodeCount < 2) {
            throw new Error('Number of nodes for cluster type multi-node must be at least 2');
        }
        if (props.encrypted === false && props.encryptionKey !== undefined) {
            throw new Error('Cannot set property encryptionKey without enabling encryption!');
        }
        this.singleUserRotationApplication = secretsmanager.SecretRotationApplication.REDSHIFT_ROTATION_SINGLE_USER;
        this.multiUserRotationApplication = secretsmanager.SecretRotationApplication.REDSHIFT_ROTATION_MULTI_USER;
        let loggingProperties;
        if (props.loggingBucket) {
            loggingProperties = {
                bucketName: props.loggingBucket.bucketName,
                s3KeyPrefix: props.loggingKeyPrefix,
            };
        }
        const cluster = new redshift_generated_1.CfnCluster(this, 'Resource', {
            // Basic
            allowVersionUpgrade: true,
            automatedSnapshotRetentionPeriod: 1,
            clusterType,
            clusterIdentifier: props.clusterName,
            clusterSubnetGroupName: subnetGroup.ref,
            vpcSecurityGroupIds: securityGroupIds,
            port: props.port,
            clusterParameterGroupName: props.parameterGroup && props.parameterGroup.clusterParameterGroupName,
            // Admin
            masterUsername: secret ? secret.secretValueFromJson('username').toString() : props.masterUser.masterUsername,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : (props.masterUser.masterPassword
                    ? props.masterUser.masterPassword.toString()
                    : 'default'),
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            nodeType: props.nodeType || NodeType.DC2_LARGE,
            numberOfNodes: nodeCount,
            loggingProperties,
            iamRoles: props.roles ? props.roles.map(role => role.roleArn) : undefined,
            dbName: props.defaultDatabaseName || 'default_db',
            publiclyAccessible: false,
            // Encryption
            kmsKeyId: props.encryptionKey && props.encryptionKey.keyArn,
            encrypted: props.encrypted !== undefined ? props.encrypted : true,
        });
        cluster.applyRemovalPolicy(removalPolicy, {
            applyToUpdateReplacePolicy: true,
        });
        this.clusterName = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        if (secret) {
            this.secret = secret.attach(this);
        }
        const defaultPort = ec2.Port.tcp(this.clusterEndpoint.port);
        this.connections = new ec2.Connections({ securityGroups, defaultPort });
    }
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromClusterAttributes(scope, id, attrs) {
        class Import extends ClusterBase {
            constructor() {
                super(...arguments);
                this.connections = new ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: ec2.Port.tcp(attrs.clusterEndpointPort),
                });
                this.clusterName = attrs.clusterName;
                this.instanceIdentifiers = [];
                this.clusterEndpoint = new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.clusterEndpointPort);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id, options) {
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
}
exports.Cluster = Cluster;
//# sourceMappingURL=data:application/json;base64,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