"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseProxy = exports.ProxyTarget = exports.SessionPinningFilter = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const rds_generated_1 = require("./rds.generated");
/**
 * SessionPinningFilter
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html#rds-proxy-pinning
 */
class SessionPinningFilter {
    constructor(
    /**
     * Filter name
     */
    filterName) {
        this.filterName = filterName;
    }
    /**
     * custom filter
     */
    static of(filterName) {
        return new SessionPinningFilter(filterName);
    }
}
exports.SessionPinningFilter = SessionPinningFilter;
/**
 * You can opt out of session pinning for the following kinds of application statements:
 *
 * - Setting session variables and configuration settings.
 */
SessionPinningFilter.EXCLUDE_VARIABLE_SETS = new SessionPinningFilter('EXCLUDE_VARIABLE_SETS');
/**
 * Proxy target: Instance or Cluster
 *
 * A target group is a collection of databases that the proxy can connect to.
 * Currently, you can specify only one RDS DB instance or Aurora DB cluster.
 */
class ProxyTarget {
    constructor(dbInstance, dbCluster) {
        this.dbInstance = dbInstance;
        this.dbCluster = dbCluster;
    }
    /**
     * From instance
     *
     * @param instance RDS database instance
     */
    static fromInstance(instance) {
        return new ProxyTarget(instance);
    }
    /**
     * From cluster
     *
     * @param cluster RDS database cluster
     */
    static fromCluster(cluster) {
        return new ProxyTarget(undefined, cluster);
    }
    /**
     * Bind this target to the specified database proxy.
     */
    bind(_) {
        let engine;
        if (this.dbCluster && this.dbInstance) {
            throw new Error('Proxy cannot target both database cluster and database instance.');
        }
        else if (this.dbCluster) {
            engine = this.dbCluster.node.defaultChild.engine;
        }
        else if (this.dbInstance) {
            engine = this.dbInstance.node.defaultChild.engine;
        }
        let engineFamily;
        switch (engine) {
            case 'aurora':
            case 'aurora-mysql':
            case 'mysql':
                engineFamily = 'MYSQL';
                break;
            case 'aurora-postgresql':
            case 'postgres':
                engineFamily = 'POSTGRESQL';
                break;
            default:
                throw new Error(`Unsupported engine type - ${engine}`);
        }
        return {
            engineFamily,
            dbClusters: this.dbCluster ? [this.dbCluster] : undefined,
            dbInstances: this.dbInstance ? [this.dbInstance] : undefined,
        };
    }
}
exports.ProxyTarget = ProxyTarget;
/**
 * RDS Database Proxy
 *
 * @resource AWS::RDS::DBProxy
 */
class DatabaseProxy extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, { physicalName: props.dbProxyName || id });
        const role = props.role || new iam.Role(this, 'IAMRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const secret of props.secrets) {
            secret.grantRead(role);
        }
        this.connections = new ec2.Connections({ securityGroups: props.securityGroups });
        const bindResult = props.proxyTarget.bind(this);
        if (props.secrets.length < 1) {
            throw new Error('One or more secrets are required.');
        }
        this.resource = new rds_generated_1.CfnDBProxy(this, 'Resource', {
            auth: props.secrets.map(_ => {
                return {
                    authScheme: 'SECRETS',
                    iamAuth: props.iamAuth ? 'REQUIRED' : 'DISABLED',
                    secretArn: _.secretArn,
                };
            }),
            dbProxyName: this.physicalName,
            debugLogging: props.debugLogging,
            engineFamily: bindResult.engineFamily,
            idleClientTimeout: (_a = props.idleClientTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            requireTls: (_b = props.requireTLS) !== null && _b !== void 0 ? _b : true,
            roleArn: role.roleArn,
            vpcSecurityGroupIds: (_c = props.securityGroups) === null || _c === void 0 ? void 0 : _c.map(_ => _.securityGroupId),
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds,
        });
        this.dbProxyName = this.resource.ref;
        this.dbProxyArn = this.resource.attrDbProxyArn;
        this.endpoint = this.resource.attrEndpoint;
        let dbInstanceIdentifiers;
        if (bindResult.dbInstances) {
            // support for only single instance
            dbInstanceIdentifiers = [bindResult.dbInstances[0].instanceIdentifier];
        }
        let dbClusterIdentifiers;
        if (bindResult.dbClusters) {
            dbClusterIdentifiers = bindResult.dbClusters.map((c) => c.clusterIdentifier);
        }
        if (!!dbInstanceIdentifiers && !!dbClusterIdentifiers) {
            throw new Error('Cannot specify both dbInstanceIdentifiers and dbClusterIdentifiers');
        }
        const proxyTargetGroup = new rds_generated_1.CfnDBProxyTargetGroup(this, 'ProxyTargetGroup', {
            dbProxyName: this.dbProxyName,
            dbInstanceIdentifiers,
            dbClusterIdentifiers,
            connectionPoolConfigurationInfo: toConnectionPoolConfigurationInfo(props),
        });
        // Currently(2020-07-04), this property must be set to default.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#TargetGroupName-fn::getatt
        proxyTargetGroup.addOverride('Properties.TargetGroupName', 'default');
    }
    /**
     * Import an existing database proxy.
     */
    static fromDatabaseProxyAttributes(scope, id, attrs) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.dbProxyName = attrs.dbProxyName;
                this.dbProxyArn = attrs.dbProxyArn;
                this.endpoint = attrs.endpoint;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.dbProxyName,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_PROXY,
        };
    }
}
exports.DatabaseProxy = DatabaseProxy;
/**
 * ConnectionPoolConfiguration (L2 => L1)
 */
function toConnectionPoolConfigurationInfo(props) {
    var _a, _b;
    return {
        connectionBorrowTimeout: (_a = props.borrowTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
        initQuery: props.initQuery,
        maxConnectionsPercent: props.maxConnectionsPercent,
        maxIdleConnectionsPercent: props.maxIdleConnectionsPercent,
        sessionPinningFilters: (_b = props.sessionPinningFilters) === null || _b === void 0 ? void 0 : _b.map(_ => _.filterName),
    };
}
//# sourceMappingURL=data:application/json;base64,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