"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OptionGroup = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const rds_generated_1 = require("./rds.generated");
/**
 * An option group
 */
class OptionGroup extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
         * The connections object for the options.
         */
        this.optionConnections = {};
        const majorEngineVersion = (_a = props.engine.engineVersion) === null || _a === void 0 ? void 0 : _a.majorVersion;
        if (!majorEngineVersion) {
            throw new Error("OptionGroup cannot be used with an engine that doesn't specify a version");
        }
        const optionGroup = new rds_generated_1.CfnOptionGroup(this, 'Resource', {
            engineName: props.engine.engineType,
            majorEngineVersion,
            optionGroupDescription: props.description || `Option group for ${props.engine.engineType} ${majorEngineVersion}`,
            optionConfigurations: this.renderConfigurations(props.configurations),
        });
        this.optionGroupName = optionGroup.ref;
    }
    /**
     * Import an existing option group.
     */
    static fromOptionGroupName(scope, id, optionGroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.optionGroupName = optionGroupName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Renders the option configurations specifications.
     */
    renderConfigurations(configurations) {
        const configs = [];
        for (const config of configurations) {
            let configuration = {
                optionName: config.name,
                optionSettings: config.settings && Object.entries(config.settings).map(([name, value]) => ({ name, value })),
                optionVersion: config.version,
            };
            if (config.port) {
                if (!config.vpc) {
                    throw new Error('`port` and `vpc` must be specified together.');
                }
                const securityGroup = new ec2.SecurityGroup(this, `SecurityGroup${config.name}`, {
                    description: `Security group for ${config.name} option`,
                    vpc: config.vpc,
                });
                this.optionConnections[config.name] = new ec2.Connections({
                    securityGroups: [securityGroup],
                    defaultPort: ec2.Port.tcp(config.port),
                });
                configuration = {
                    ...configuration,
                    port: config.port,
                    vpcSecurityGroupMemberships: [securityGroup.securityGroupId],
                };
            }
            configs.push(configuration);
        }
        return configs;
    }
}
exports.OptionGroup = OptionGroup;
//# sourceMappingURL=data:application/json;base64,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