"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseCluster = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const proxy_1 = require("./proxy");
const rds_generated_1 = require("./rds.generated");
/**
 * A new or imported clustered database.
 */
class DatabaseClusterBase extends core_1.Resource {
    /**
     * Add a new db proxy to this cluster.
     */
    addProxy(id, options) {
        return new proxy_1.DatabaseProxy(this, id, {
            proxyTarget: proxy_1.ProxyTarget.fromCluster(this),
            ...options,
        });
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_CLUSTER,
        };
    }
}
/**
 * Create a clustered database with a given number of instances.
 *
 * @resource AWS::RDS::DBCluster
 */
class DatabaseCluster extends DatabaseClusterBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g;
        super(scope, id);
        /**
         * Identifiers of the replicas
         */
        this.instanceIdentifiers = [];
        /**
         * Endpoints which address each individual replica.
         */
        this.instanceEndpoints = [];
        this.vpc = props.instanceProps.vpc;
        this.vpcSubnets = props.instanceProps.vpcSubnets;
        const { subnetIds } = props.instanceProps.vpc.selectSubnets(props.instanceProps.vpcSubnets);
        // Cannot test whether the subnets are in different AZs, but at least we can test the amount.
        if (subnetIds.length < 2) {
            this.node.addError(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        const subnetGroup = new rds_generated_1.CfnDBSubnetGroup(this, 'Subnets', {
            dbSubnetGroupDescription: `Subnets for ${id} database`,
            subnetIds,
        });
        if (props.removalPolicy === core_1.RemovalPolicy.RETAIN) {
            subnetGroup.applyRemovalPolicy(core_1.RemovalPolicy.RETAIN);
        }
        const securityGroups = (_a = props.instanceProps.securityGroups) !== null && _a !== void 0 ? _a : [
            new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'RDS security group',
                vpc: props.instanceProps.vpc,
            }),
        ];
        let secret;
        if (!props.masterUser.password) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUser.username,
                encryptionKey: props.masterUser.encryptionKey,
            });
        }
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        let s3ImportRole = props.s3ImportRole;
        if (props.s3ImportBuckets && props.s3ImportBuckets.length > 0) {
            if (props.s3ImportRole) {
                throw new Error('Only one of s3ImportRole or s3ImportBuckets must be specified, not both.');
            }
            s3ImportRole = new aws_iam_1.Role(this, 'S3ImportRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
            });
            for (const bucket of props.s3ImportBuckets) {
                bucket.grantRead(s3ImportRole);
            }
        }
        let s3ExportRole = props.s3ExportRole;
        if (props.s3ExportBuckets && props.s3ExportBuckets.length > 0) {
            if (props.s3ExportRole) {
                throw new Error('Only one of s3ExportRole or s3ExportBuckets must be specified, not both.');
            }
            s3ExportRole = new aws_iam_1.Role(this, 'S3ExportRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
            });
            for (const bucket of props.s3ExportBuckets) {
                bucket.grantReadWrite(s3ExportRole);
            }
        }
        const clusterAssociatedRoles = [];
        if (s3ImportRole || s3ExportRole) {
            if (s3ImportRole) {
                clusterAssociatedRoles.push({ roleArn: s3ImportRole.roleArn });
            }
            if (s3ExportRole) {
                clusterAssociatedRoles.push({ roleArn: s3ExportRole.roleArn });
            }
        }
        // bind the engine to the Cluster
        const clusterEngineBindConfig = props.engine.bindToCluster(this, {
            s3ImportRole,
            s3ExportRole,
            parameterGroup: props.parameterGroup,
        });
        const clusterParameterGroup = (_b = props.parameterGroup) !== null && _b !== void 0 ? _b : clusterEngineBindConfig.parameterGroup;
        const clusterParameterGroupConfig = clusterParameterGroup === null || clusterParameterGroup === void 0 ? void 0 : clusterParameterGroup.bindToCluster({});
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            // Basic
            engine: props.engine.engineType,
            engineVersion: (_c = props.engine.engineVersion) === null || _c === void 0 ? void 0 : _c.fullVersion,
            dbClusterIdentifier: props.clusterIdentifier,
            dbSubnetGroupName: subnetGroup.ref,
            vpcSecurityGroupIds: securityGroups.map(sg => sg.securityGroupId),
            port: (_d = props.port) !== null && _d !== void 0 ? _d : clusterEngineBindConfig.port,
            dbClusterParameterGroupName: clusterParameterGroupConfig === null || clusterParameterGroupConfig === void 0 ? void 0 : clusterParameterGroupConfig.parameterGroupName,
            associatedRoles: clusterAssociatedRoles.length > 0 ? clusterAssociatedRoles : undefined,
            // Admin
            masterUsername: secret ? secret.secretValueFromJson('username').toString() : props.masterUser.username,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : (props.masterUser.password
                    ? props.masterUser.password.toString()
                    : undefined),
            backupRetentionPeriod: props.backup && props.backup.retention && props.backup.retention.toDays(),
            preferredBackupWindow: props.backup && props.backup.preferredWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            databaseName: props.defaultDatabaseName,
            // Encryption
            kmsKeyId: props.storageEncryptionKey && props.storageEncryptionKey.keyArn,
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        // if removalPolicy was not specified,
        // leave it as the default, which is Snapshot
        if (props.removalPolicy) {
            cluster.applyRemovalPolicy(props.removalPolicy);
        }
        else {
            // The CFN default makes sense for DeletionPolicy,
            // but doesn't cover UpdateReplacePolicy.
            // Fix that here.
            cluster.cfnOptions.updateReplacePolicy = core_1.CfnDeletionPolicy.SNAPSHOT;
        }
        this.clusterIdentifier = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        if (secret) {
            this.secret = secret.attach(this);
        }
        const instanceCount = props.instances != null ? props.instances : 2;
        if (instanceCount < 1) {
            throw new Error('At least one instance is required');
        }
        // Get the actual subnet objects so we can depend on internet connectivity.
        const internetConnected = props.instanceProps.vpc.selectSubnets(props.instanceProps.vpcSubnets).internetConnectivityEstablished;
        let monitoringRole;
        if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
            monitoringRole = props.monitoringRole || new aws_iam_1.Role(this, 'MonitoringRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('monitoring.rds.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole'),
                ],
            });
        }
        const instanceType = (_e = props.instanceProps.instanceType) !== null && _e !== void 0 ? _e : ec2.InstanceType.of(ec2.InstanceClass.T3, ec2.InstanceSize.MEDIUM);
        const instanceParameterGroupConfig = (_f = props.instanceProps.parameterGroup) === null || _f === void 0 ? void 0 : _f.bindToInstance({});
        for (let i = 0; i < instanceCount; i++) {
            const instanceIndex = i + 1;
            const instanceIdentifier = props.instanceIdentifierBase != null ? `${props.instanceIdentifierBase}${instanceIndex}` :
                props.clusterIdentifier != null ? `${props.clusterIdentifier}instance${instanceIndex}` :
                    undefined;
            const publiclyAccessible = props.instanceProps.vpcSubnets && props.instanceProps.vpcSubnets.subnetType === ec2.SubnetType.PUBLIC;
            const instance = new rds_generated_1.CfnDBInstance(this, `Instance${instanceIndex}`, {
                // Link to cluster
                engine: props.engine.engineType,
                engineVersion: (_g = props.engine.engineVersion) === null || _g === void 0 ? void 0 : _g.fullVersion,
                dbClusterIdentifier: cluster.ref,
                dbInstanceIdentifier: instanceIdentifier,
                // Instance properties
                dbInstanceClass: databaseInstanceType(instanceType),
                publiclyAccessible,
                // This is already set on the Cluster. Unclear to me whether it should be repeated or not. Better yes.
                dbSubnetGroupName: subnetGroup.ref,
                dbParameterGroupName: instanceParameterGroupConfig === null || instanceParameterGroupConfig === void 0 ? void 0 : instanceParameterGroupConfig.parameterGroupName,
                monitoringInterval: props.monitoringInterval && props.monitoringInterval.toSeconds(),
                monitoringRoleArn: monitoringRole && monitoringRole.roleArn,
            });
            // If removalPolicy isn't explicitly set,
            // it's Snapshot for Cluster.
            // Because of that, in this case,
            // we can safely use the CFN default of Delete for DbInstances with dbClusterIdentifier set.
            if (props.removalPolicy) {
                instance.applyRemovalPolicy(props.removalPolicy);
            }
            // We must have a dependency on the NAT gateway provider here to create
            // things in the right order.
            instance.node.addDependency(internetConnected);
            this.instanceIdentifiers.push(instance.ref);
            this.instanceEndpoints.push(new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute));
        }
        const defaultPort = ec2.Port.tcp(this.clusterEndpoint.port);
        this.connections = new ec2.Connections({ securityGroups, defaultPort });
    }
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromDatabaseClusterAttributes(scope, id, attrs) {
        class Import extends DatabaseClusterBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: this.defaultPort,
                });
                this.clusterIdentifier = attrs.clusterIdentifier;
                this.instanceIdentifiers = [];
                this.clusterEndpoint = new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port);
                this.clusterReadEndpoint = new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port);
                this.instanceEndpoints = attrs.instanceEndpointAddresses.map(a => new endpoint_1.Endpoint(a, attrs.port));
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id, options) {
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
}
exports.DatabaseCluster = DatabaseCluster;
/**
 * Turn a regular instance type into a database instance type
 */
function databaseInstanceType(instanceType) {
    return 'db.' + instanceType.toString();
}
//# sourceMappingURL=data:application/json;base64,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