"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SpaceDelimitedTextPattern = exports.FilterPattern = exports.JsonPattern = void 0;
/**
 * Base class for patterns that only match JSON log events.
 */
class JsonPattern {
    // This is a separate class so we have some type safety where users can't
    // combine text patterns and JSON patterns with an 'and' operation.
    constructor(jsonPatternString) {
        this.jsonPatternString = jsonPatternString;
    }
    get logPatternString() {
        return '{ ' + this.jsonPatternString + ' }';
    }
}
exports.JsonPattern = JsonPattern;
/**
 * A collection of static methods to generate appropriate ILogPatterns
 */
class FilterPattern {
    /**
     * Use the given string as log pattern.
     *
     * See https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/FilterAndPatternSyntax.html
     * for information on writing log patterns.
     *
     * @param logPatternString The pattern string to use.
     */
    static literal(logPatternString) {
        return new LiteralLogPattern(logPatternString);
    }
    /**
     * A log pattern that matches all events.
     */
    static allEvents() {
        return new LiteralLogPattern('');
    }
    /**
     * A log pattern that matches if all the strings given appear in the event.
     *
     * @param terms The words to search for. All terms must match.
     */
    static allTerms(...terms) {
        return new TextLogPattern([terms]);
    }
    /**
     * A log pattern that matches if any of the strings given appear in the event.
     *
     * @param terms The words to search for. Any terms must match.
     */
    static anyTerm(...terms) {
        return new TextLogPattern(terms.map(t => [t]));
    }
    /**
     * A log pattern that matches if any of the given term groups matches the event.
     *
     * A term group matches an event if all the terms in it appear in the event string.
     *
     * @param termGroups A list of term groups to search for. Any one of the clauses must match.
     */
    static anyTermGroup(...termGroups) {
        return new TextLogPattern(termGroups);
    }
    /**
     * A JSON log pattern that compares string values.
     *
     * This pattern only matches if the event is a JSON event, and the indicated field inside
     * compares with the string value.
     *
     * Use '$' to indicate the root of the JSON structure. The comparison operator can only
     * compare equality or inequality. The '*' wildcard may appear in the value may at the
     * start or at the end.
     *
     * For more information, see:
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/FilterAndPatternSyntax.html
     *
     * @param jsonField Field inside JSON. Example: "$.myField"
     * @param comparison Comparison to carry out. Either = or !=.
     * @param value The string value to compare to. May use '*' as wildcard at start or end of string.
     */
    static stringValue(jsonField, comparison, value) {
        return new JSONStringPattern(jsonField, comparison, value);
    }
    /**
     * A JSON log pattern that compares numerical values.
     *
     * This pattern only matches if the event is a JSON event, and the indicated field inside
     * compares with the value in the indicated way.
     *
     * Use '$' to indicate the root of the JSON structure. The comparison operator can only
     * compare equality or inequality. The '*' wildcard may appear in the value may at the
     * start or at the end.
     *
     * For more information, see:
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/FilterAndPatternSyntax.html
     *
     * @param jsonField Field inside JSON. Example: "$.myField"
     * @param comparison Comparison to carry out. One of =, !=, <, <=, >, >=.
     * @param value The numerical value to compare to
     */
    static numberValue(jsonField, comparison, value) {
        return new JSONNumberPattern(jsonField, comparison, value);
    }
    /**
     * A JSON log pattern that matches if the field exists and has the special value 'null'.
     *
     * @param jsonField Field inside JSON. Example: "$.myField"
     */
    static isNull(jsonField) {
        return new JSONPostfixPattern(jsonField, 'IS NULL');
    }
    /**
     * A JSON log pattern that matches if the field does not exist.
     *
     * @param jsonField Field inside JSON. Example: "$.myField"
     */
    static notExists(jsonField) {
        return new JSONPostfixPattern(jsonField, 'NOT EXISTS');
    }
    /**
     * A JSON log patter that matches if the field exists.
     *
     * This is a readable convenience wrapper over 'field = *'
     *
     * @param jsonField Field inside JSON. Example: "$.myField"
     */
    static exists(jsonField) {
        return new JSONStringPattern(jsonField, '=', '*');
    }
    /**
     * A JSON log pattern that matches if the field exists and equals the boolean value.
     *
     * @param jsonField Field inside JSON. Example: "$.myField"
     * @param value The value to match
     */
    static booleanValue(jsonField, value) {
        return new JSONPostfixPattern(jsonField, value ? 'IS TRUE' : 'IS FALSE');
    }
    /**
     * A JSON log pattern that matches if all given JSON log patterns match
     */
    static all(...patterns) {
        if (patterns.length === 0) {
            throw new Error('Must supply at least one pattern, or use allEvents() to match all events.');
        }
        if (patterns.length === 1) {
            return patterns[0];
        }
        return new JSONAggregatePattern('&&', patterns);
    }
    /**
     * A JSON log pattern that matches if any of the given JSON log patterns match
     */
    static any(...patterns) {
        if (patterns.length === 0) {
            throw new Error('Must supply at least one pattern');
        }
        if (patterns.length === 1) {
            return patterns[0];
        }
        return new JSONAggregatePattern('||', patterns);
    }
    /**
     * A space delimited log pattern matcher.
     *
     * The log event is divided into space-delimited columns (optionally
     * enclosed by "" or [] to capture spaces into column values), and names
     * are given to each column.
     *
     * '...' may be specified once to match any number of columns.
     *
     * Afterwards, conditions may be added to individual columns.
     *
     * @param columns The columns in the space-delimited log stream.
     */
    static spaceDelimited(...columns) {
        return SpaceDelimitedTextPattern.construct(columns);
    }
}
exports.FilterPattern = FilterPattern;
/**
 * Use a string literal as a log pattern
 */
class LiteralLogPattern {
    constructor(logPatternString) {
        this.logPatternString = logPatternString;
    }
}
/**
 * Search for a set of set of terms
 */
class TextLogPattern {
    constructor(clauses) {
        const quotedClauses = clauses.map(terms => terms.map(quoteTerm).join(' '));
        if (quotedClauses.length === 1) {
            this.logPatternString = quotedClauses[0];
        }
        else {
            this.logPatternString = quotedClauses.map(alt => '?' + alt).join(' ');
        }
    }
}
/**
 * A string comparison for JSON values
 */
class JSONStringPattern extends JsonPattern {
    constructor(jsonField, comparison, value) {
        comparison = validateStringOperator(comparison);
        super(`${jsonField} ${comparison} ${quoteTerm(value)}`);
    }
}
/**
 * A number comparison for JSON values
 */
class JSONNumberPattern extends JsonPattern {
    constructor(jsonField, comparison, value) {
        comparison = validateNumericalOperator(comparison);
        super(`${jsonField} ${comparison} ${value}`);
    }
}
/**
 * A postfix operator for JSON patterns
 */
class JSONPostfixPattern extends JsonPattern {
    constructor(jsonField, postfix) {
        // No validation, we assume these are generated by trusted factory functions
        super(`${jsonField} ${postfix}`);
    }
}
/**
 * Combines multiple other JSON patterns with an operator
 */
class JSONAggregatePattern extends JsonPattern {
    constructor(operator, patterns) {
        if (operator !== '&&' && operator !== '||') {
            throw new Error('Operator must be one of && or ||');
        }
        const clauses = patterns.map(p => '(' + p.jsonPatternString + ')');
        super(clauses.join(` ${operator} `));
    }
}
const COL_ELLIPSIS = '...';
/**
 * Space delimited text pattern
 */
class SpaceDelimitedTextPattern {
    // TODO: Temporarily changed from private to protected to unblock build. We need to think
    //     about how to handle jsii types with private constructors.
    constructor(columns, restrictions) {
        this.columns = columns;
        this.restrictions = restrictions;
        // Private constructor so we validate in the .construct() factory function
    }
    /**
     * Construct a new instance of a space delimited text pattern
     *
     * Since this class must be public, we can't rely on the user only creating it through
     * the `LogPattern.spaceDelimited()` factory function. We must therefore validate the
     * argument in the constructor. Since we're returning a copy on every mutation, and we
     * don't want to re-validate the same things on every construction, we provide a limited
     * set of mutator functions and only validate the new data every time.
     */
    static construct(columns) {
        // Validation happens here because a user could instantiate this object directly without
        // going through the factory
        for (const column of columns) {
            if (!validColumnName(column)) {
                throw new Error(`Invalid column name: ${column}`);
            }
        }
        if (sum(columns.map(c => c === COL_ELLIPSIS ? 1 : 0)) > 1) {
            throw new Error("Can use at most one '...' column");
        }
        return new SpaceDelimitedTextPattern(columns, {});
    }
    /**
     * Restrict where the pattern applies
     */
    whereString(columnName, comparison, value) {
        if (columnName === COL_ELLIPSIS) {
            throw new Error("Can't use '...' in a restriction");
        }
        if (this.columns.indexOf(columnName) === -1) {
            throw new Error(`Column in restrictions that is not in columns: ${columnName}`);
        }
        comparison = validateStringOperator(comparison);
        return new SpaceDelimitedTextPattern(this.columns, this.addRestriction(columnName, {
            comparison,
            stringValue: value,
        }));
    }
    /**
     * Restrict where the pattern applies
     */
    whereNumber(columnName, comparison, value) {
        if (columnName === COL_ELLIPSIS) {
            throw new Error("Can't use '...' in a restriction");
        }
        if (this.columns.indexOf(columnName) === -1) {
            throw new Error(`Column in restrictions that is not in columns: ${columnName}`);
        }
        comparison = validateNumericalOperator(comparison);
        return new SpaceDelimitedTextPattern(this.columns, this.addRestriction(columnName, {
            comparison,
            numberValue: value,
        }));
    }
    get logPatternString() {
        return '[' + this.columns.map(this.columnExpression.bind(this)).join(', ') + ']';
    }
    /**
     * Return the column expression for the given column
     */
    columnExpression(column) {
        const restrictions = this.restrictions[column];
        if (!restrictions) {
            return column;
        }
        return restrictions.map(r => renderRestriction(column, r)).join(' && ');
    }
    /**
     * Make a copy of the current restrictions and add one
     */
    addRestriction(columnName, restriction) {
        const ret = {};
        for (const key of Object.keys(this.restrictions)) {
            ret[key] = this.restrictions[key].slice();
        }
        if (!(columnName in ret)) {
            ret[columnName] = [];
        }
        ret[columnName].push(restriction);
        return ret;
    }
}
exports.SpaceDelimitedTextPattern = SpaceDelimitedTextPattern;
/**
 * Quote a term for use in a pattern expression
 *
 * It's never wrong to quote a string term, and required if the term
 * contains non-alphanumerical characters, so we just always do it.
 *
 * Inner double quotes are escaped using a backslash.
 */
function quoteTerm(term) {
    return '"' + term.replace(/\\/g, '\\\\').replace(/"/g, '\\"') + '"';
}
/**
 * Return whether the given column name is valid in a space-delimited table
 */
function validColumnName(column) {
    return column === COL_ELLIPSIS || /^[a-zA-Z0-9_-]+$/.exec(column);
}
/**
 * Validate and normalize the string comparison operator
 *
 * Correct for a common typo/confusion, treat '==' as '='
 */
function validateStringOperator(operator) {
    if (operator === '==') {
        operator = '=';
    }
    if (operator !== '=' && operator !== '!=') {
        throw new Error(`Invalid comparison operator ('${operator}'), must be either '=' or '!='`);
    }
    return operator;
}
const VALID_OPERATORS = ['=', '!=', '<', '<=', '>', '>='];
/**
 * Validate and normalize numerical comparison operators
 *
 * Correct for a common typo/confusion, treat '==' as '='
 */
function validateNumericalOperator(operator) {
    // Correct for a common typo, treat '==' as '='
    if (operator === '==') {
        operator = '=';
    }
    if (VALID_OPERATORS.indexOf(operator) === -1) {
        throw new Error(`Invalid comparison operator ('${operator}'), must be one of ${VALID_OPERATORS.join(', ')}`);
    }
    return operator;
}
/**
 * Render a table restriction
 */
function renderRestriction(column, restriction) {
    if (restriction.numberValue !== undefined) {
        return `${column} ${restriction.comparison} ${restriction.numberValue}`;
    }
    else if (restriction.stringValue) {
        return `${column} ${restriction.comparison} ${quoteTerm(restriction.stringValue)}`;
    }
    else {
        throw new Error('Invalid restriction');
    }
}
function sum(xs) {
    return xs.reduce((a, c) => a + c, 0);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicGF0dGVybi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInBhdHRlcm4udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBT0E7O0dBRUc7QUFDSCxNQUFzQixXQUFXO0lBQzdCLHlFQUF5RTtJQUN6RSxtRUFBbUU7SUFDbkUsWUFBNEIsaUJBQXlCO1FBQXpCLHNCQUFpQixHQUFqQixpQkFBaUIsQ0FBUTtJQUFJLENBQUM7SUFDMUQsSUFBVyxnQkFBZ0I7UUFDdkIsT0FBTyxJQUFJLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixHQUFHLElBQUksQ0FBQztJQUNoRCxDQUFDO0NBQ0o7QUFQRCxrQ0FPQztBQUNEOztHQUVHO0FBQ0gsTUFBYSxhQUFhO0lBQ3RCOzs7Ozs7O09BT0c7SUFDSSxNQUFNLENBQUMsT0FBTyxDQUFDLGdCQUF3QjtRQUMxQyxPQUFPLElBQUksaUJBQWlCLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuRCxDQUFDO0lBQ0Q7O09BRUc7SUFDSSxNQUFNLENBQUMsU0FBUztRQUNuQixPQUFPLElBQUksaUJBQWlCLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDckMsQ0FBQztJQUNEOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsUUFBUSxDQUFDLEdBQUcsS0FBZTtRQUNyQyxPQUFPLElBQUksY0FBYyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxPQUFPLENBQUMsR0FBRyxLQUFlO1FBQ3BDLE9BQU8sSUFBSSxjQUFjLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFDRDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsWUFBWSxDQUFDLEdBQUcsVUFBc0I7UUFDaEQsT0FBTyxJQUFJLGNBQWMsQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBQ0Q7Ozs7Ozs7Ozs7Ozs7Ozs7O09BaUJHO0lBQ0ksTUFBTSxDQUFDLFdBQVcsQ0FBQyxTQUFpQixFQUFFLFVBQWtCLEVBQUUsS0FBYTtRQUMxRSxPQUFPLElBQUksaUJBQWlCLENBQUMsU0FBUyxFQUFFLFVBQVUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUMvRCxDQUFDO0lBQ0Q7Ozs7Ozs7Ozs7Ozs7Ozs7O09BaUJHO0lBQ0ksTUFBTSxDQUFDLFdBQVcsQ0FBQyxTQUFpQixFQUFFLFVBQWtCLEVBQUUsS0FBYTtRQUMxRSxPQUFPLElBQUksaUJBQWlCLENBQUMsU0FBUyxFQUFFLFVBQVUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUMvRCxDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxNQUFNLENBQUMsU0FBaUI7UUFDbEMsT0FBTyxJQUFJLGtCQUFrQixDQUFDLFNBQVMsRUFBRSxTQUFTLENBQUMsQ0FBQztJQUN4RCxDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxTQUFTLENBQUMsU0FBaUI7UUFDckMsT0FBTyxJQUFJLGtCQUFrQixDQUFDLFNBQVMsRUFBRSxZQUFZLENBQUMsQ0FBQztJQUMzRCxDQUFDO0lBQ0Q7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLE1BQU0sQ0FBQyxTQUFpQjtRQUNsQyxPQUFPLElBQUksaUJBQWlCLENBQUMsU0FBUyxFQUFFLEdBQUcsRUFBRSxHQUFHLENBQUMsQ0FBQztJQUN0RCxDQUFDO0lBQ0Q7Ozs7O09BS0c7SUFDSSxNQUFNLENBQUMsWUFBWSxDQUFDLFNBQWlCLEVBQUUsS0FBYztRQUN4RCxPQUFPLElBQUksa0JBQWtCLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUM3RSxDQUFDO0lBQ0Q7O09BRUc7SUFDSSxNQUFNLENBQUMsR0FBRyxDQUFDLEdBQUcsUUFBdUI7UUFDeEMsSUFBSSxRQUFRLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUN2QixNQUFNLElBQUksS0FBSyxDQUFDLDJFQUEyRSxDQUFDLENBQUM7U0FDaEc7UUFDRCxJQUFJLFFBQVEsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3ZCLE9BQU8sUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDO1NBQ3RCO1FBQ0QsT0FBTyxJQUFJLG9CQUFvQixDQUFDLElBQUksRUFBRSxRQUFRLENBQUMsQ0FBQztJQUNwRCxDQUFDO0lBQ0Q7O09BRUc7SUFDSSxNQUFNLENBQUMsR0FBRyxDQUFDLEdBQUcsUUFBdUI7UUFDeEMsSUFBSSxRQUFRLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUN2QixNQUFNLElBQUksS0FBSyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7U0FDdkQ7UUFDRCxJQUFJLFFBQVEsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3ZCLE9BQU8sUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDO1NBQ3RCO1FBQ0QsT0FBTyxJQUFJLG9CQUFvQixDQUFDLElBQUksRUFBRSxRQUFRLENBQUMsQ0FBQztJQUNwRCxDQUFDO0lBQ0Q7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksTUFBTSxDQUFDLGNBQWMsQ0FBQyxHQUFHLE9BQWlCO1FBQzdDLE9BQU8seUJBQXlCLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQ3hELENBQUM7Q0FDSjtBQWpLRCxzQ0FpS0M7QUFDRDs7R0FFRztBQUNILE1BQU0saUJBQWlCO0lBQ25CLFlBQTRCLGdCQUF3QjtRQUF4QixxQkFBZ0IsR0FBaEIsZ0JBQWdCLENBQVE7SUFDcEQsQ0FBQztDQUNKO0FBQ0Q7O0dBRUc7QUFDSCxNQUFNLGNBQWM7SUFFaEIsWUFBWSxPQUFtQjtRQUMzQixNQUFNLGFBQWEsR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUMzRSxJQUFJLGFBQWEsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzVCLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDNUM7YUFDSTtZQUNELElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxhQUFhLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUN6RTtJQUNMLENBQUM7Q0FDSjtBQUNEOztHQUVHO0FBQ0gsTUFBTSxpQkFBa0IsU0FBUSxXQUFXO0lBQ3ZDLFlBQW1CLFNBQWlCLEVBQUUsVUFBa0IsRUFBRSxLQUFhO1FBQ25FLFVBQVUsR0FBRyxzQkFBc0IsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUNoRCxLQUFLLENBQUMsR0FBRyxTQUFTLElBQUksVUFBVSxJQUFJLFNBQVMsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDNUQsQ0FBQztDQUNKO0FBQ0Q7O0dBRUc7QUFDSCxNQUFNLGlCQUFrQixTQUFRLFdBQVc7SUFDdkMsWUFBbUIsU0FBaUIsRUFBRSxVQUFrQixFQUFFLEtBQWE7UUFDbkUsVUFBVSxHQUFHLHlCQUF5QixDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ25ELEtBQUssQ0FBQyxHQUFHLFNBQVMsSUFBSSxVQUFVLElBQUksS0FBSyxFQUFFLENBQUMsQ0FBQztJQUNqRCxDQUFDO0NBQ0o7QUFDRDs7R0FFRztBQUNILE1BQU0sa0JBQW1CLFNBQVEsV0FBVztJQUN4QyxZQUFtQixTQUFpQixFQUFFLE9BQWU7UUFDakQsNEVBQTRFO1FBQzVFLEtBQUssQ0FBQyxHQUFHLFNBQVMsSUFBSSxPQUFPLEVBQUUsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7Q0FDSjtBQUNEOztHQUVHO0FBQ0gsTUFBTSxvQkFBcUIsU0FBUSxXQUFXO0lBQzFDLFlBQW1CLFFBQWdCLEVBQUUsUUFBdUI7UUFDeEQsSUFBSSxRQUFRLEtBQUssSUFBSSxJQUFJLFFBQVEsS0FBSyxJQUFJLEVBQUU7WUFDeEMsTUFBTSxJQUFJLEtBQUssQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1NBQ3ZEO1FBQ0QsTUFBTSxPQUFPLEdBQUcsUUFBUSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEdBQUcsR0FBRyxDQUFDLENBQUMsaUJBQWlCLEdBQUcsR0FBRyxDQUFDLENBQUM7UUFDbkUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxRQUFRLEdBQUcsQ0FBQyxDQUFDLENBQUM7SUFDekMsQ0FBQztDQUNKO0FBSUQsTUFBTSxZQUFZLEdBQUcsS0FBSyxDQUFDO0FBQzNCOztHQUVHO0FBQ0gsTUFBYSx5QkFBeUI7SUF1QmxDLHlGQUF5RjtJQUN6RixnRUFBZ0U7SUFDaEUsWUFBdUMsT0FBaUIsRUFBbUIsWUFBNEI7UUFBaEUsWUFBTyxHQUFQLE9BQU8sQ0FBVTtRQUFtQixpQkFBWSxHQUFaLFlBQVksQ0FBZ0I7UUFDbkcsMEVBQTBFO0lBQzlFLENBQUM7SUExQkQ7Ozs7Ozs7O09BUUc7SUFDSSxNQUFNLENBQUMsU0FBUyxDQUFDLE9BQWlCO1FBQ3JDLHdGQUF3RjtRQUN4Riw0QkFBNEI7UUFDNUIsS0FBSyxNQUFNLE1BQU0sSUFBSSxPQUFPLEVBQUU7WUFDMUIsSUFBSSxDQUFDLGVBQWUsQ0FBQyxNQUFNLENBQUMsRUFBRTtnQkFDMUIsTUFBTSxJQUFJLEtBQUssQ0FBQyx3QkFBd0IsTUFBTSxFQUFFLENBQUMsQ0FBQzthQUNyRDtTQUNKO1FBQ0QsSUFBSSxHQUFHLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsS0FBSyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLEVBQUU7WUFDdkQsTUFBTSxJQUFJLEtBQUssQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1NBQ3ZEO1FBQ0QsT0FBTyxJQUFJLHlCQUF5QixDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQztJQUN0RCxDQUFDO0lBTUQ7O09BRUc7SUFDSSxXQUFXLENBQUMsVUFBa0IsRUFBRSxVQUFrQixFQUFFLEtBQWE7UUFDcEUsSUFBSSxVQUFVLEtBQUssWUFBWSxFQUFFO1lBQzdCLE1BQU0sSUFBSSxLQUFLLENBQUMsa0NBQWtDLENBQUMsQ0FBQztTQUN2RDtRQUNELElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUU7WUFDekMsTUFBTSxJQUFJLEtBQUssQ0FBQyxrREFBa0QsVUFBVSxFQUFFLENBQUMsQ0FBQztTQUNuRjtRQUNELFVBQVUsR0FBRyxzQkFBc0IsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUNoRCxPQUFPLElBQUkseUJBQXlCLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLFVBQVUsRUFBRTtZQUMvRSxVQUFVO1lBQ1YsV0FBVyxFQUFFLEtBQUs7U0FDckIsQ0FBQyxDQUFDLENBQUM7SUFDUixDQUFDO0lBQ0Q7O09BRUc7SUFDSSxXQUFXLENBQUMsVUFBa0IsRUFBRSxVQUFrQixFQUFFLEtBQWE7UUFDcEUsSUFBSSxVQUFVLEtBQUssWUFBWSxFQUFFO1lBQzdCLE1BQU0sSUFBSSxLQUFLLENBQUMsa0NBQWtDLENBQUMsQ0FBQztTQUN2RDtRQUNELElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUU7WUFDekMsTUFBTSxJQUFJLEtBQUssQ0FBQyxrREFBa0QsVUFBVSxFQUFFLENBQUMsQ0FBQztTQUNuRjtRQUNELFVBQVUsR0FBRyx5QkFBeUIsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUNuRCxPQUFPLElBQUkseUJBQXlCLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLFVBQVUsRUFBRTtZQUMvRSxVQUFVO1lBQ1YsV0FBVyxFQUFFLEtBQUs7U0FDckIsQ0FBQyxDQUFDLENBQUM7SUFDUixDQUFDO0lBQ0QsSUFBVyxnQkFBZ0I7UUFDdkIsT0FBTyxHQUFHLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxHQUFHLENBQUM7SUFDckYsQ0FBQztJQUNEOztPQUVHO0lBQ0ssZ0JBQWdCLENBQUMsTUFBYztRQUNuQyxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQy9DLElBQUksQ0FBQyxZQUFZLEVBQUU7WUFDZixPQUFPLE1BQU0sQ0FBQztTQUNqQjtRQUNELE9BQU8sWUFBWSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUM1RSxDQUFDO0lBQ0Q7O09BRUc7SUFDSyxjQUFjLENBQUMsVUFBa0IsRUFBRSxXQUE4QjtRQUNyRSxNQUFNLEdBQUcsR0FBbUIsRUFBRSxDQUFDO1FBQy9CLEtBQUssTUFBTSxHQUFHLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLEVBQUU7WUFDOUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxHQUFHLElBQUksQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLENBQUMsS0FBSyxFQUFFLENBQUM7U0FDN0M7UUFDRCxJQUFJLENBQUMsQ0FBQyxVQUFVLElBQUksR0FBRyxDQUFDLEVBQUU7WUFDdEIsR0FBRyxDQUFDLFVBQVUsQ0FBQyxHQUFHLEVBQUUsQ0FBQztTQUN4QjtRQUNELEdBQUcsQ0FBQyxVQUFVLENBQUMsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDbEMsT0FBTyxHQUFHLENBQUM7SUFDZixDQUFDO0NBQ0o7QUF2RkQsOERBdUZDO0FBbUJEOzs7Ozs7O0dBT0c7QUFDSCxTQUFTLFNBQVMsQ0FBQyxJQUFZO0lBQzNCLE9BQU8sR0FBRyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLEdBQUcsR0FBRyxDQUFDO0FBQ3hFLENBQUM7QUFDRDs7R0FFRztBQUNILFNBQVMsZUFBZSxDQUFDLE1BQWM7SUFDbkMsT0FBTyxNQUFNLEtBQUssWUFBWSxJQUFJLGtCQUFrQixDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztBQUN0RSxDQUFDO0FBQ0Q7Ozs7R0FJRztBQUNILFNBQVMsc0JBQXNCLENBQUMsUUFBZ0I7SUFDNUMsSUFBSSxRQUFRLEtBQUssSUFBSSxFQUFFO1FBQ25CLFFBQVEsR0FBRyxHQUFHLENBQUM7S0FDbEI7SUFDRCxJQUFJLFFBQVEsS0FBSyxHQUFHLElBQUksUUFBUSxLQUFLLElBQUksRUFBRTtRQUN2QyxNQUFNLElBQUksS0FBSyxDQUFDLGlDQUFpQyxRQUFRLGdDQUFnQyxDQUFDLENBQUM7S0FDOUY7SUFDRCxPQUFPLFFBQVEsQ0FBQztBQUNwQixDQUFDO0FBQ0QsTUFBTSxlQUFlLEdBQUcsQ0FBQyxHQUFHLEVBQUUsSUFBSSxFQUFFLEdBQUcsRUFBRSxJQUFJLEVBQUUsR0FBRyxFQUFFLElBQUksQ0FBQyxDQUFDO0FBQzFEOzs7O0dBSUc7QUFDSCxTQUFTLHlCQUF5QixDQUFDLFFBQWdCO0lBQy9DLCtDQUErQztJQUMvQyxJQUFJLFFBQVEsS0FBSyxJQUFJLEVBQUU7UUFDbkIsUUFBUSxHQUFHLEdBQUcsQ0FBQztLQUNsQjtJQUNELElBQUksZUFBZSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRTtRQUMxQyxNQUFNLElBQUksS0FBSyxDQUFDLGlDQUFpQyxRQUFRLHNCQUFzQixlQUFlLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztLQUNoSDtJQUNELE9BQU8sUUFBUSxDQUFDO0FBQ3BCLENBQUM7QUFDRDs7R0FFRztBQUNILFNBQVMsaUJBQWlCLENBQUMsTUFBYyxFQUFFLFdBQThCO0lBQ3JFLElBQUksV0FBVyxDQUFDLFdBQVcsS0FBSyxTQUFTLEVBQUU7UUFDdkMsT0FBTyxHQUFHLE1BQU0sSUFBSSxXQUFXLENBQUMsVUFBVSxJQUFJLFdBQVcsQ0FBQyxXQUFXLEVBQUUsQ0FBQztLQUMzRTtTQUNJLElBQUksV0FBVyxDQUFDLFdBQVcsRUFBRTtRQUM5QixPQUFPLEdBQUcsTUFBTSxJQUFJLFdBQVcsQ0FBQyxVQUFVLElBQUksU0FBUyxDQUFDLFdBQVcsQ0FBQyxXQUFXLENBQUMsRUFBRSxDQUFDO0tBQ3RGO1NBQ0k7UUFDRCxNQUFNLElBQUksS0FBSyxDQUFDLHFCQUFxQixDQUFDLENBQUM7S0FDMUM7QUFDTCxDQUFDO0FBQ0QsU0FBUyxHQUFHLENBQUMsRUFBWTtJQUNyQixPQUFPLEVBQUUsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO0FBQ3pDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvLyBJbXBsZW1lbnRhdGlvbiBvZiBtZXRyaWMgcGF0dGVybnNcbi8qKlxuICogSW50ZXJmYWNlIGZvciBvYmplY3RzIHRoYXQgY2FuIHJlbmRlciB0aGVtc2VsdmVzIHRvIGxvZyBwYXR0ZXJucy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJRmlsdGVyUGF0dGVybiB7XG4gICAgcmVhZG9ubHkgbG9nUGF0dGVyblN0cmluZzogc3RyaW5nO1xufVxuLyoqXG4gKiBCYXNlIGNsYXNzIGZvciBwYXR0ZXJucyB0aGF0IG9ubHkgbWF0Y2ggSlNPTiBsb2cgZXZlbnRzLlxuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgSnNvblBhdHRlcm4gaW1wbGVtZW50cyBJRmlsdGVyUGF0dGVybiB7XG4gICAgLy8gVGhpcyBpcyBhIHNlcGFyYXRlIGNsYXNzIHNvIHdlIGhhdmUgc29tZSB0eXBlIHNhZmV0eSB3aGVyZSB1c2VycyBjYW4ndFxuICAgIC8vIGNvbWJpbmUgdGV4dCBwYXR0ZXJucyBhbmQgSlNPTiBwYXR0ZXJucyB3aXRoIGFuICdhbmQnIG9wZXJhdGlvbi5cbiAgICBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkganNvblBhdHRlcm5TdHJpbmc6IHN0cmluZykgeyB9XG4gICAgcHVibGljIGdldCBsb2dQYXR0ZXJuU3RyaW5nKCk6IHN0cmluZyB7XG4gICAgICAgIHJldHVybiAneyAnICsgdGhpcy5qc29uUGF0dGVyblN0cmluZyArICcgfSc7XG4gICAgfVxufVxuLyoqXG4gKiBBIGNvbGxlY3Rpb24gb2Ygc3RhdGljIG1ldGhvZHMgdG8gZ2VuZXJhdGUgYXBwcm9wcmlhdGUgSUxvZ1BhdHRlcm5zXG4gKi9cbmV4cG9ydCBjbGFzcyBGaWx0ZXJQYXR0ZXJuIHtcbiAgICAvKipcbiAgICAgKiBVc2UgdGhlIGdpdmVuIHN0cmluZyBhcyBsb2cgcGF0dGVybi5cbiAgICAgKlxuICAgICAqIFNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaC9sYXRlc3QvbG9ncy9GaWx0ZXJBbmRQYXR0ZXJuU3ludGF4Lmh0bWxcbiAgICAgKiBmb3IgaW5mb3JtYXRpb24gb24gd3JpdGluZyBsb2cgcGF0dGVybnMuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gbG9nUGF0dGVyblN0cmluZyBUaGUgcGF0dGVybiBzdHJpbmcgdG8gdXNlLlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgbGl0ZXJhbChsb2dQYXR0ZXJuU3RyaW5nOiBzdHJpbmcpOiBJRmlsdGVyUGF0dGVybiB7XG4gICAgICAgIHJldHVybiBuZXcgTGl0ZXJhbExvZ1BhdHRlcm4obG9nUGF0dGVyblN0cmluZyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEEgbG9nIHBhdHRlcm4gdGhhdCBtYXRjaGVzIGFsbCBldmVudHMuXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBhbGxFdmVudHMoKTogSUZpbHRlclBhdHRlcm4ge1xuICAgICAgICByZXR1cm4gbmV3IExpdGVyYWxMb2dQYXR0ZXJuKCcnKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQSBsb2cgcGF0dGVybiB0aGF0IG1hdGNoZXMgaWYgYWxsIHRoZSBzdHJpbmdzIGdpdmVuIGFwcGVhciBpbiB0aGUgZXZlbnQuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gdGVybXMgVGhlIHdvcmRzIHRvIHNlYXJjaCBmb3IuIEFsbCB0ZXJtcyBtdXN0IG1hdGNoLlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgYWxsVGVybXMoLi4udGVybXM6IHN0cmluZ1tdKTogSUZpbHRlclBhdHRlcm4ge1xuICAgICAgICByZXR1cm4gbmV3IFRleHRMb2dQYXR0ZXJuKFt0ZXJtc10pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBIGxvZyBwYXR0ZXJuIHRoYXQgbWF0Y2hlcyBpZiBhbnkgb2YgdGhlIHN0cmluZ3MgZ2l2ZW4gYXBwZWFyIGluIHRoZSBldmVudC5cbiAgICAgKlxuICAgICAqIEBwYXJhbSB0ZXJtcyBUaGUgd29yZHMgdG8gc2VhcmNoIGZvci4gQW55IHRlcm1zIG11c3QgbWF0Y2guXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBhbnlUZXJtKC4uLnRlcm1zOiBzdHJpbmdbXSk6IElGaWx0ZXJQYXR0ZXJuIHtcbiAgICAgICAgcmV0dXJuIG5ldyBUZXh0TG9nUGF0dGVybih0ZXJtcy5tYXAodCA9PiBbdF0pKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQSBsb2cgcGF0dGVybiB0aGF0IG1hdGNoZXMgaWYgYW55IG9mIHRoZSBnaXZlbiB0ZXJtIGdyb3VwcyBtYXRjaGVzIHRoZSBldmVudC5cbiAgICAgKlxuICAgICAqIEEgdGVybSBncm91cCBtYXRjaGVzIGFuIGV2ZW50IGlmIGFsbCB0aGUgdGVybXMgaW4gaXQgYXBwZWFyIGluIHRoZSBldmVudCBzdHJpbmcuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gdGVybUdyb3VwcyBBIGxpc3Qgb2YgdGVybSBncm91cHMgdG8gc2VhcmNoIGZvci4gQW55IG9uZSBvZiB0aGUgY2xhdXNlcyBtdXN0IG1hdGNoLlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgYW55VGVybUdyb3VwKC4uLnRlcm1Hcm91cHM6IHN0cmluZ1tdW10pOiBJRmlsdGVyUGF0dGVybiB7XG4gICAgICAgIHJldHVybiBuZXcgVGV4dExvZ1BhdHRlcm4odGVybUdyb3Vwcyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEEgSlNPTiBsb2cgcGF0dGVybiB0aGF0IGNvbXBhcmVzIHN0cmluZyB2YWx1ZXMuXG4gICAgICpcbiAgICAgKiBUaGlzIHBhdHRlcm4gb25seSBtYXRjaGVzIGlmIHRoZSBldmVudCBpcyBhIEpTT04gZXZlbnQsIGFuZCB0aGUgaW5kaWNhdGVkIGZpZWxkIGluc2lkZVxuICAgICAqIGNvbXBhcmVzIHdpdGggdGhlIHN0cmluZyB2YWx1ZS5cbiAgICAgKlxuICAgICAqIFVzZSAnJCcgdG8gaW5kaWNhdGUgdGhlIHJvb3Qgb2YgdGhlIEpTT04gc3RydWN0dXJlLiBUaGUgY29tcGFyaXNvbiBvcGVyYXRvciBjYW4gb25seVxuICAgICAqIGNvbXBhcmUgZXF1YWxpdHkgb3IgaW5lcXVhbGl0eS4gVGhlICcqJyB3aWxkY2FyZCBtYXkgYXBwZWFyIGluIHRoZSB2YWx1ZSBtYXkgYXQgdGhlXG4gICAgICogc3RhcnQgb3IgYXQgdGhlIGVuZC5cbiAgICAgKlxuICAgICAqIEZvciBtb3JlIGluZm9ybWF0aW9uLCBzZWU6XG4gICAgICpcbiAgICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaC9sYXRlc3QvbG9ncy9GaWx0ZXJBbmRQYXR0ZXJuU3ludGF4Lmh0bWxcbiAgICAgKlxuICAgICAqIEBwYXJhbSBqc29uRmllbGQgRmllbGQgaW5zaWRlIEpTT04uIEV4YW1wbGU6IFwiJC5teUZpZWxkXCJcbiAgICAgKiBAcGFyYW0gY29tcGFyaXNvbiBDb21wYXJpc29uIHRvIGNhcnJ5IG91dC4gRWl0aGVyID0gb3IgIT0uXG4gICAgICogQHBhcmFtIHZhbHVlIFRoZSBzdHJpbmcgdmFsdWUgdG8gY29tcGFyZSB0by4gTWF5IHVzZSAnKicgYXMgd2lsZGNhcmQgYXQgc3RhcnQgb3IgZW5kIG9mIHN0cmluZy5cbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHN0cmluZ1ZhbHVlKGpzb25GaWVsZDogc3RyaW5nLCBjb21wYXJpc29uOiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcpOiBKc29uUGF0dGVybiB7XG4gICAgICAgIHJldHVybiBuZXcgSlNPTlN0cmluZ1BhdHRlcm4oanNvbkZpZWxkLCBjb21wYXJpc29uLCB2YWx1ZSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEEgSlNPTiBsb2cgcGF0dGVybiB0aGF0IGNvbXBhcmVzIG51bWVyaWNhbCB2YWx1ZXMuXG4gICAgICpcbiAgICAgKiBUaGlzIHBhdHRlcm4gb25seSBtYXRjaGVzIGlmIHRoZSBldmVudCBpcyBhIEpTT04gZXZlbnQsIGFuZCB0aGUgaW5kaWNhdGVkIGZpZWxkIGluc2lkZVxuICAgICAqIGNvbXBhcmVzIHdpdGggdGhlIHZhbHVlIGluIHRoZSBpbmRpY2F0ZWQgd2F5LlxuICAgICAqXG4gICAgICogVXNlICckJyB0byBpbmRpY2F0ZSB0aGUgcm9vdCBvZiB0aGUgSlNPTiBzdHJ1Y3R1cmUuIFRoZSBjb21wYXJpc29uIG9wZXJhdG9yIGNhbiBvbmx5XG4gICAgICogY29tcGFyZSBlcXVhbGl0eSBvciBpbmVxdWFsaXR5LiBUaGUgJyonIHdpbGRjYXJkIG1heSBhcHBlYXIgaW4gdGhlIHZhbHVlIG1heSBhdCB0aGVcbiAgICAgKiBzdGFydCBvciBhdCB0aGUgZW5kLlxuICAgICAqXG4gICAgICogRm9yIG1vcmUgaW5mb3JtYXRpb24sIHNlZTpcbiAgICAgKlxuICAgICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoL2xhdGVzdC9sb2dzL0ZpbHRlckFuZFBhdHRlcm5TeW50YXguaHRtbFxuICAgICAqXG4gICAgICogQHBhcmFtIGpzb25GaWVsZCBGaWVsZCBpbnNpZGUgSlNPTi4gRXhhbXBsZTogXCIkLm15RmllbGRcIlxuICAgICAqIEBwYXJhbSBjb21wYXJpc29uIENvbXBhcmlzb24gdG8gY2Fycnkgb3V0LiBPbmUgb2YgPSwgIT0sIDwsIDw9LCA+LCA+PS5cbiAgICAgKiBAcGFyYW0gdmFsdWUgVGhlIG51bWVyaWNhbCB2YWx1ZSB0byBjb21wYXJlIHRvXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBudW1iZXJWYWx1ZShqc29uRmllbGQ6IHN0cmluZywgY29tcGFyaXNvbjogc3RyaW5nLCB2YWx1ZTogbnVtYmVyKTogSnNvblBhdHRlcm4ge1xuICAgICAgICByZXR1cm4gbmV3IEpTT05OdW1iZXJQYXR0ZXJuKGpzb25GaWVsZCwgY29tcGFyaXNvbiwgdmFsdWUpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBIEpTT04gbG9nIHBhdHRlcm4gdGhhdCBtYXRjaGVzIGlmIHRoZSBmaWVsZCBleGlzdHMgYW5kIGhhcyB0aGUgc3BlY2lhbCB2YWx1ZSAnbnVsbCcuXG4gICAgICpcbiAgICAgKiBAcGFyYW0ganNvbkZpZWxkIEZpZWxkIGluc2lkZSBKU09OLiBFeGFtcGxlOiBcIiQubXlGaWVsZFwiXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBpc051bGwoanNvbkZpZWxkOiBzdHJpbmcpOiBKc29uUGF0dGVybiB7XG4gICAgICAgIHJldHVybiBuZXcgSlNPTlBvc3RmaXhQYXR0ZXJuKGpzb25GaWVsZCwgJ0lTIE5VTEwnKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQSBKU09OIGxvZyBwYXR0ZXJuIHRoYXQgbWF0Y2hlcyBpZiB0aGUgZmllbGQgZG9lcyBub3QgZXhpc3QuXG4gICAgICpcbiAgICAgKiBAcGFyYW0ganNvbkZpZWxkIEZpZWxkIGluc2lkZSBKU09OLiBFeGFtcGxlOiBcIiQubXlGaWVsZFwiXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBub3RFeGlzdHMoanNvbkZpZWxkOiBzdHJpbmcpOiBKc29uUGF0dGVybiB7XG4gICAgICAgIHJldHVybiBuZXcgSlNPTlBvc3RmaXhQYXR0ZXJuKGpzb25GaWVsZCwgJ05PVCBFWElTVFMnKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQSBKU09OIGxvZyBwYXR0ZXIgdGhhdCBtYXRjaGVzIGlmIHRoZSBmaWVsZCBleGlzdHMuXG4gICAgICpcbiAgICAgKiBUaGlzIGlzIGEgcmVhZGFibGUgY29udmVuaWVuY2Ugd3JhcHBlciBvdmVyICdmaWVsZCA9IConXG4gICAgICpcbiAgICAgKiBAcGFyYW0ganNvbkZpZWxkIEZpZWxkIGluc2lkZSBKU09OLiBFeGFtcGxlOiBcIiQubXlGaWVsZFwiXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBleGlzdHMoanNvbkZpZWxkOiBzdHJpbmcpOiBKc29uUGF0dGVybiB7XG4gICAgICAgIHJldHVybiBuZXcgSlNPTlN0cmluZ1BhdHRlcm4oanNvbkZpZWxkLCAnPScsICcqJyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEEgSlNPTiBsb2cgcGF0dGVybiB0aGF0IG1hdGNoZXMgaWYgdGhlIGZpZWxkIGV4aXN0cyBhbmQgZXF1YWxzIHRoZSBib29sZWFuIHZhbHVlLlxuICAgICAqXG4gICAgICogQHBhcmFtIGpzb25GaWVsZCBGaWVsZCBpbnNpZGUgSlNPTi4gRXhhbXBsZTogXCIkLm15RmllbGRcIlxuICAgICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUgdG8gbWF0Y2hcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGJvb2xlYW5WYWx1ZShqc29uRmllbGQ6IHN0cmluZywgdmFsdWU6IGJvb2xlYW4pOiBKc29uUGF0dGVybiB7XG4gICAgICAgIHJldHVybiBuZXcgSlNPTlBvc3RmaXhQYXR0ZXJuKGpzb25GaWVsZCwgdmFsdWUgPyAnSVMgVFJVRScgOiAnSVMgRkFMU0UnKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQSBKU09OIGxvZyBwYXR0ZXJuIHRoYXQgbWF0Y2hlcyBpZiBhbGwgZ2l2ZW4gSlNPTiBsb2cgcGF0dGVybnMgbWF0Y2hcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGFsbCguLi5wYXR0ZXJuczogSnNvblBhdHRlcm5bXSk6IEpzb25QYXR0ZXJuIHtcbiAgICAgICAgaWYgKHBhdHRlcm5zLmxlbmd0aCA9PT0gMCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdNdXN0IHN1cHBseSBhdCBsZWFzdCBvbmUgcGF0dGVybiwgb3IgdXNlIGFsbEV2ZW50cygpIHRvIG1hdGNoIGFsbCBldmVudHMuJyk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHBhdHRlcm5zLmxlbmd0aCA9PT0gMSkge1xuICAgICAgICAgICAgcmV0dXJuIHBhdHRlcm5zWzBdO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBuZXcgSlNPTkFnZ3JlZ2F0ZVBhdHRlcm4oJyYmJywgcGF0dGVybnMpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBIEpTT04gbG9nIHBhdHRlcm4gdGhhdCBtYXRjaGVzIGlmIGFueSBvZiB0aGUgZ2l2ZW4gSlNPTiBsb2cgcGF0dGVybnMgbWF0Y2hcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGFueSguLi5wYXR0ZXJuczogSnNvblBhdHRlcm5bXSk6IEpzb25QYXR0ZXJuIHtcbiAgICAgICAgaWYgKHBhdHRlcm5zLmxlbmd0aCA9PT0gMCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdNdXN0IHN1cHBseSBhdCBsZWFzdCBvbmUgcGF0dGVybicpO1xuICAgICAgICB9XG4gICAgICAgIGlmIChwYXR0ZXJucy5sZW5ndGggPT09IDEpIHtcbiAgICAgICAgICAgIHJldHVybiBwYXR0ZXJuc1swXTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gbmV3IEpTT05BZ2dyZWdhdGVQYXR0ZXJuKCd8fCcsIHBhdHRlcm5zKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQSBzcGFjZSBkZWxpbWl0ZWQgbG9nIHBhdHRlcm4gbWF0Y2hlci5cbiAgICAgKlxuICAgICAqIFRoZSBsb2cgZXZlbnQgaXMgZGl2aWRlZCBpbnRvIHNwYWNlLWRlbGltaXRlZCBjb2x1bW5zIChvcHRpb25hbGx5XG4gICAgICogZW5jbG9zZWQgYnkgXCJcIiBvciBbXSB0byBjYXB0dXJlIHNwYWNlcyBpbnRvIGNvbHVtbiB2YWx1ZXMpLCBhbmQgbmFtZXNcbiAgICAgKiBhcmUgZ2l2ZW4gdG8gZWFjaCBjb2x1bW4uXG4gICAgICpcbiAgICAgKiAnLi4uJyBtYXkgYmUgc3BlY2lmaWVkIG9uY2UgdG8gbWF0Y2ggYW55IG51bWJlciBvZiBjb2x1bW5zLlxuICAgICAqXG4gICAgICogQWZ0ZXJ3YXJkcywgY29uZGl0aW9ucyBtYXkgYmUgYWRkZWQgdG8gaW5kaXZpZHVhbCBjb2x1bW5zLlxuICAgICAqXG4gICAgICogQHBhcmFtIGNvbHVtbnMgVGhlIGNvbHVtbnMgaW4gdGhlIHNwYWNlLWRlbGltaXRlZCBsb2cgc3RyZWFtLlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgc3BhY2VEZWxpbWl0ZWQoLi4uY29sdW1uczogc3RyaW5nW10pOiBTcGFjZURlbGltaXRlZFRleHRQYXR0ZXJuIHtcbiAgICAgICAgcmV0dXJuIFNwYWNlRGVsaW1pdGVkVGV4dFBhdHRlcm4uY29uc3RydWN0KGNvbHVtbnMpO1xuICAgIH1cbn1cbi8qKlxuICogVXNlIGEgc3RyaW5nIGxpdGVyYWwgYXMgYSBsb2cgcGF0dGVyblxuICovXG5jbGFzcyBMaXRlcmFsTG9nUGF0dGVybiBpbXBsZW1lbnRzIElGaWx0ZXJQYXR0ZXJuIHtcbiAgICBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgbG9nUGF0dGVyblN0cmluZzogc3RyaW5nKSB7XG4gICAgfVxufVxuLyoqXG4gKiBTZWFyY2ggZm9yIGEgc2V0IG9mIHNldCBvZiB0ZXJtc1xuICovXG5jbGFzcyBUZXh0TG9nUGF0dGVybiBpbXBsZW1lbnRzIElGaWx0ZXJQYXR0ZXJuIHtcbiAgICBwdWJsaWMgcmVhZG9ubHkgbG9nUGF0dGVyblN0cmluZzogc3RyaW5nO1xuICAgIGNvbnN0cnVjdG9yKGNsYXVzZXM6IHN0cmluZ1tdW10pIHtcbiAgICAgICAgY29uc3QgcXVvdGVkQ2xhdXNlcyA9IGNsYXVzZXMubWFwKHRlcm1zID0+IHRlcm1zLm1hcChxdW90ZVRlcm0pLmpvaW4oJyAnKSk7XG4gICAgICAgIGlmIChxdW90ZWRDbGF1c2VzLmxlbmd0aCA9PT0gMSkge1xuICAgICAgICAgICAgdGhpcy5sb2dQYXR0ZXJuU3RyaW5nID0gcXVvdGVkQ2xhdXNlc1swXTtcbiAgICAgICAgfVxuICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgIHRoaXMubG9nUGF0dGVyblN0cmluZyA9IHF1b3RlZENsYXVzZXMubWFwKGFsdCA9PiAnPycgKyBhbHQpLmpvaW4oJyAnKTtcbiAgICAgICAgfVxuICAgIH1cbn1cbi8qKlxuICogQSBzdHJpbmcgY29tcGFyaXNvbiBmb3IgSlNPTiB2YWx1ZXNcbiAqL1xuY2xhc3MgSlNPTlN0cmluZ1BhdHRlcm4gZXh0ZW5kcyBKc29uUGF0dGVybiB7XG4gICAgcHVibGljIGNvbnN0cnVjdG9yKGpzb25GaWVsZDogc3RyaW5nLCBjb21wYXJpc29uOiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcpIHtcbiAgICAgICAgY29tcGFyaXNvbiA9IHZhbGlkYXRlU3RyaW5nT3BlcmF0b3IoY29tcGFyaXNvbik7XG4gICAgICAgIHN1cGVyKGAke2pzb25GaWVsZH0gJHtjb21wYXJpc29ufSAke3F1b3RlVGVybSh2YWx1ZSl9YCk7XG4gICAgfVxufVxuLyoqXG4gKiBBIG51bWJlciBjb21wYXJpc29uIGZvciBKU09OIHZhbHVlc1xuICovXG5jbGFzcyBKU09OTnVtYmVyUGF0dGVybiBleHRlbmRzIEpzb25QYXR0ZXJuIHtcbiAgICBwdWJsaWMgY29uc3RydWN0b3IoanNvbkZpZWxkOiBzdHJpbmcsIGNvbXBhcmlzb246IHN0cmluZywgdmFsdWU6IG51bWJlcikge1xuICAgICAgICBjb21wYXJpc29uID0gdmFsaWRhdGVOdW1lcmljYWxPcGVyYXRvcihjb21wYXJpc29uKTtcbiAgICAgICAgc3VwZXIoYCR7anNvbkZpZWxkfSAke2NvbXBhcmlzb259ICR7dmFsdWV9YCk7XG4gICAgfVxufVxuLyoqXG4gKiBBIHBvc3RmaXggb3BlcmF0b3IgZm9yIEpTT04gcGF0dGVybnNcbiAqL1xuY2xhc3MgSlNPTlBvc3RmaXhQYXR0ZXJuIGV4dGVuZHMgSnNvblBhdHRlcm4ge1xuICAgIHB1YmxpYyBjb25zdHJ1Y3Rvcihqc29uRmllbGQ6IHN0cmluZywgcG9zdGZpeDogc3RyaW5nKSB7XG4gICAgICAgIC8vIE5vIHZhbGlkYXRpb24sIHdlIGFzc3VtZSB0aGVzZSBhcmUgZ2VuZXJhdGVkIGJ5IHRydXN0ZWQgZmFjdG9yeSBmdW5jdGlvbnNcbiAgICAgICAgc3VwZXIoYCR7anNvbkZpZWxkfSAke3Bvc3RmaXh9YCk7XG4gICAgfVxufVxuLyoqXG4gKiBDb21iaW5lcyBtdWx0aXBsZSBvdGhlciBKU09OIHBhdHRlcm5zIHdpdGggYW4gb3BlcmF0b3JcbiAqL1xuY2xhc3MgSlNPTkFnZ3JlZ2F0ZVBhdHRlcm4gZXh0ZW5kcyBKc29uUGF0dGVybiB7XG4gICAgcHVibGljIGNvbnN0cnVjdG9yKG9wZXJhdG9yOiBzdHJpbmcsIHBhdHRlcm5zOiBKc29uUGF0dGVybltdKSB7XG4gICAgICAgIGlmIChvcGVyYXRvciAhPT0gJyYmJyAmJiBvcGVyYXRvciAhPT0gJ3x8Jykge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdPcGVyYXRvciBtdXN0IGJlIG9uZSBvZiAmJiBvciB8fCcpO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IGNsYXVzZXMgPSBwYXR0ZXJucy5tYXAocCA9PiAnKCcgKyBwLmpzb25QYXR0ZXJuU3RyaW5nICsgJyknKTtcbiAgICAgICAgc3VwZXIoY2xhdXNlcy5qb2luKGAgJHtvcGVyYXRvcn0gYCkpO1xuICAgIH1cbn1cbmV4cG9ydCB0eXBlIFJlc3RyaWN0aW9uTWFwID0ge1xuICAgIFtjb2x1bW46IHN0cmluZ106IENvbHVtblJlc3RyaWN0aW9uW107XG59O1xuY29uc3QgQ09MX0VMTElQU0lTID0gJy4uLic7XG4vKipcbiAqIFNwYWNlIGRlbGltaXRlZCB0ZXh0IHBhdHRlcm5cbiAqL1xuZXhwb3J0IGNsYXNzIFNwYWNlRGVsaW1pdGVkVGV4dFBhdHRlcm4gaW1wbGVtZW50cyBJRmlsdGVyUGF0dGVybiB7XG4gICAgLyoqXG4gICAgICogQ29uc3RydWN0IGEgbmV3IGluc3RhbmNlIG9mIGEgc3BhY2UgZGVsaW1pdGVkIHRleHQgcGF0dGVyblxuICAgICAqXG4gICAgICogU2luY2UgdGhpcyBjbGFzcyBtdXN0IGJlIHB1YmxpYywgd2UgY2FuJ3QgcmVseSBvbiB0aGUgdXNlciBvbmx5IGNyZWF0aW5nIGl0IHRocm91Z2hcbiAgICAgKiB0aGUgYExvZ1BhdHRlcm4uc3BhY2VEZWxpbWl0ZWQoKWAgZmFjdG9yeSBmdW5jdGlvbi4gV2UgbXVzdCB0aGVyZWZvcmUgdmFsaWRhdGUgdGhlXG4gICAgICogYXJndW1lbnQgaW4gdGhlIGNvbnN0cnVjdG9yLiBTaW5jZSB3ZSdyZSByZXR1cm5pbmcgYSBjb3B5IG9uIGV2ZXJ5IG11dGF0aW9uLCBhbmQgd2VcbiAgICAgKiBkb24ndCB3YW50IHRvIHJlLXZhbGlkYXRlIHRoZSBzYW1lIHRoaW5ncyBvbiBldmVyeSBjb25zdHJ1Y3Rpb24sIHdlIHByb3ZpZGUgYSBsaW1pdGVkXG4gICAgICogc2V0IG9mIG11dGF0b3IgZnVuY3Rpb25zIGFuZCBvbmx5IHZhbGlkYXRlIHRoZSBuZXcgZGF0YSBldmVyeSB0aW1lLlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgY29uc3RydWN0KGNvbHVtbnM6IHN0cmluZ1tdKSB7XG4gICAgICAgIC8vIFZhbGlkYXRpb24gaGFwcGVucyBoZXJlIGJlY2F1c2UgYSB1c2VyIGNvdWxkIGluc3RhbnRpYXRlIHRoaXMgb2JqZWN0IGRpcmVjdGx5IHdpdGhvdXRcbiAgICAgICAgLy8gZ29pbmcgdGhyb3VnaCB0aGUgZmFjdG9yeVxuICAgICAgICBmb3IgKGNvbnN0IGNvbHVtbiBvZiBjb2x1bW5zKSB7XG4gICAgICAgICAgICBpZiAoIXZhbGlkQ29sdW1uTmFtZShjb2x1bW4pKSB7XG4gICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBJbnZhbGlkIGNvbHVtbiBuYW1lOiAke2NvbHVtbn1gKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICBpZiAoc3VtKGNvbHVtbnMubWFwKGMgPT4gYyA9PT0gQ09MX0VMTElQU0lTID8gMSA6IDApKSA+IDEpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihcIkNhbiB1c2UgYXQgbW9zdCBvbmUgJy4uLicgY29sdW1uXCIpO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBuZXcgU3BhY2VEZWxpbWl0ZWRUZXh0UGF0dGVybihjb2x1bW5zLCB7fSk7XG4gICAgfVxuICAgIC8vIFRPRE86IFRlbXBvcmFyaWx5IGNoYW5nZWQgZnJvbSBwcml2YXRlIHRvIHByb3RlY3RlZCB0byB1bmJsb2NrIGJ1aWxkLiBXZSBuZWVkIHRvIHRoaW5rXG4gICAgLy8gICAgIGFib3V0IGhvdyB0byBoYW5kbGUganNpaSB0eXBlcyB3aXRoIHByaXZhdGUgY29uc3RydWN0b3JzLlxuICAgIHByb3RlY3RlZCBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IGNvbHVtbnM6IHN0cmluZ1tdLCBwcml2YXRlIHJlYWRvbmx5IHJlc3RyaWN0aW9uczogUmVzdHJpY3Rpb25NYXApIHtcbiAgICAgICAgLy8gUHJpdmF0ZSBjb25zdHJ1Y3RvciBzbyB3ZSB2YWxpZGF0ZSBpbiB0aGUgLmNvbnN0cnVjdCgpIGZhY3RvcnkgZnVuY3Rpb25cbiAgICB9XG4gICAgLyoqXG4gICAgICogUmVzdHJpY3Qgd2hlcmUgdGhlIHBhdHRlcm4gYXBwbGllc1xuICAgICAqL1xuICAgIHB1YmxpYyB3aGVyZVN0cmluZyhjb2x1bW5OYW1lOiBzdHJpbmcsIGNvbXBhcmlzb246IHN0cmluZywgdmFsdWU6IHN0cmluZyk6IFNwYWNlRGVsaW1pdGVkVGV4dFBhdHRlcm4ge1xuICAgICAgICBpZiAoY29sdW1uTmFtZSA9PT0gQ09MX0VMTElQU0lTKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXCJDYW4ndCB1c2UgJy4uLicgaW4gYSByZXN0cmljdGlvblwiKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAodGhpcy5jb2x1bW5zLmluZGV4T2YoY29sdW1uTmFtZSkgPT09IC0xKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYENvbHVtbiBpbiByZXN0cmljdGlvbnMgdGhhdCBpcyBub3QgaW4gY29sdW1uczogJHtjb2x1bW5OYW1lfWApO1xuICAgICAgICB9XG4gICAgICAgIGNvbXBhcmlzb24gPSB2YWxpZGF0ZVN0cmluZ09wZXJhdG9yKGNvbXBhcmlzb24pO1xuICAgICAgICByZXR1cm4gbmV3IFNwYWNlRGVsaW1pdGVkVGV4dFBhdHRlcm4odGhpcy5jb2x1bW5zLCB0aGlzLmFkZFJlc3RyaWN0aW9uKGNvbHVtbk5hbWUsIHtcbiAgICAgICAgICAgIGNvbXBhcmlzb24sXG4gICAgICAgICAgICBzdHJpbmdWYWx1ZTogdmFsdWUsXG4gICAgICAgIH0pKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogUmVzdHJpY3Qgd2hlcmUgdGhlIHBhdHRlcm4gYXBwbGllc1xuICAgICAqL1xuICAgIHB1YmxpYyB3aGVyZU51bWJlcihjb2x1bW5OYW1lOiBzdHJpbmcsIGNvbXBhcmlzb246IHN0cmluZywgdmFsdWU6IG51bWJlcik6IFNwYWNlRGVsaW1pdGVkVGV4dFBhdHRlcm4ge1xuICAgICAgICBpZiAoY29sdW1uTmFtZSA9PT0gQ09MX0VMTElQU0lTKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXCJDYW4ndCB1c2UgJy4uLicgaW4gYSByZXN0cmljdGlvblwiKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAodGhpcy5jb2x1bW5zLmluZGV4T2YoY29sdW1uTmFtZSkgPT09IC0xKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYENvbHVtbiBpbiByZXN0cmljdGlvbnMgdGhhdCBpcyBub3QgaW4gY29sdW1uczogJHtjb2x1bW5OYW1lfWApO1xuICAgICAgICB9XG4gICAgICAgIGNvbXBhcmlzb24gPSB2YWxpZGF0ZU51bWVyaWNhbE9wZXJhdG9yKGNvbXBhcmlzb24pO1xuICAgICAgICByZXR1cm4gbmV3IFNwYWNlRGVsaW1pdGVkVGV4dFBhdHRlcm4odGhpcy5jb2x1bW5zLCB0aGlzLmFkZFJlc3RyaWN0aW9uKGNvbHVtbk5hbWUsIHtcbiAgICAgICAgICAgIGNvbXBhcmlzb24sXG4gICAgICAgICAgICBudW1iZXJWYWx1ZTogdmFsdWUsXG4gICAgICAgIH0pKTtcbiAgICB9XG4gICAgcHVibGljIGdldCBsb2dQYXR0ZXJuU3RyaW5nKCk6IHN0cmluZyB7XG4gICAgICAgIHJldHVybiAnWycgKyB0aGlzLmNvbHVtbnMubWFwKHRoaXMuY29sdW1uRXhwcmVzc2lvbi5iaW5kKHRoaXMpKS5qb2luKCcsICcpICsgJ10nO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZXR1cm4gdGhlIGNvbHVtbiBleHByZXNzaW9uIGZvciB0aGUgZ2l2ZW4gY29sdW1uXG4gICAgICovXG4gICAgcHJpdmF0ZSBjb2x1bW5FeHByZXNzaW9uKGNvbHVtbjogc3RyaW5nKSB7XG4gICAgICAgIGNvbnN0IHJlc3RyaWN0aW9ucyA9IHRoaXMucmVzdHJpY3Rpb25zW2NvbHVtbl07XG4gICAgICAgIGlmICghcmVzdHJpY3Rpb25zKSB7XG4gICAgICAgICAgICByZXR1cm4gY29sdW1uO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiByZXN0cmljdGlvbnMubWFwKHIgPT4gcmVuZGVyUmVzdHJpY3Rpb24oY29sdW1uLCByKSkuam9pbignICYmICcpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBNYWtlIGEgY29weSBvZiB0aGUgY3VycmVudCByZXN0cmljdGlvbnMgYW5kIGFkZCBvbmVcbiAgICAgKi9cbiAgICBwcml2YXRlIGFkZFJlc3RyaWN0aW9uKGNvbHVtbk5hbWU6IHN0cmluZywgcmVzdHJpY3Rpb246IENvbHVtblJlc3RyaWN0aW9uKSB7XG4gICAgICAgIGNvbnN0IHJldDogUmVzdHJpY3Rpb25NYXAgPSB7fTtcbiAgICAgICAgZm9yIChjb25zdCBrZXkgb2YgT2JqZWN0LmtleXModGhpcy5yZXN0cmljdGlvbnMpKSB7XG4gICAgICAgICAgICByZXRba2V5XSA9IHRoaXMucmVzdHJpY3Rpb25zW2tleV0uc2xpY2UoKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAoIShjb2x1bW5OYW1lIGluIHJldCkpIHtcbiAgICAgICAgICAgIHJldFtjb2x1bW5OYW1lXSA9IFtdO1xuICAgICAgICB9XG4gICAgICAgIHJldFtjb2x1bW5OYW1lXS5wdXNoKHJlc3RyaWN0aW9uKTtcbiAgICAgICAgcmV0dXJuIHJldDtcbiAgICB9XG59XG5leHBvcnQgaW50ZXJmYWNlIENvbHVtblJlc3RyaWN0aW9uIHtcbiAgICAvKipcbiAgICAgKiBDb21wYXJpc29uIG9wZXJhdG9yIHRvIHVzZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNvbXBhcmlzb246IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBTdHJpbmcgdmFsdWUgdG8gY29tcGFyZSB0b1xuICAgICAqXG4gICAgICogRXhhY3RseSBvbmUgb2YgJ3N0cmluZ1ZhbHVlJyBhbmQgJ251bWJlclZhbHVlJyBtdXN0IGJlIHNldC5cbiAgICAgKi9cbiAgICByZWFkb25seSBzdHJpbmdWYWx1ZT86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBOdW1iZXIgdmFsdWUgdG8gY29tcGFyZSB0b1xuICAgICAqXG4gICAgICogRXhhY3RseSBvbmUgb2YgJ3N0cmluZ1ZhbHVlJyBhbmQgJ251bWJlclZhbHVlJyBtdXN0IGJlIHNldC5cbiAgICAgKi9cbiAgICByZWFkb25seSBudW1iZXJWYWx1ZT86IG51bWJlcjtcbn1cbi8qKlxuICogUXVvdGUgYSB0ZXJtIGZvciB1c2UgaW4gYSBwYXR0ZXJuIGV4cHJlc3Npb25cbiAqXG4gKiBJdCdzIG5ldmVyIHdyb25nIHRvIHF1b3RlIGEgc3RyaW5nIHRlcm0sIGFuZCByZXF1aXJlZCBpZiB0aGUgdGVybVxuICogY29udGFpbnMgbm9uLWFscGhhbnVtZXJpY2FsIGNoYXJhY3RlcnMsIHNvIHdlIGp1c3QgYWx3YXlzIGRvIGl0LlxuICpcbiAqIElubmVyIGRvdWJsZSBxdW90ZXMgYXJlIGVzY2FwZWQgdXNpbmcgYSBiYWNrc2xhc2guXG4gKi9cbmZ1bmN0aW9uIHF1b3RlVGVybSh0ZXJtOiBzdHJpbmcpOiBzdHJpbmcge1xuICAgIHJldHVybiAnXCInICsgdGVybS5yZXBsYWNlKC9cXFxcL2csICdcXFxcXFxcXCcpLnJlcGxhY2UoL1wiL2csICdcXFxcXCInKSArICdcIic7XG59XG4vKipcbiAqIFJldHVybiB3aGV0aGVyIHRoZSBnaXZlbiBjb2x1bW4gbmFtZSBpcyB2YWxpZCBpbiBhIHNwYWNlLWRlbGltaXRlZCB0YWJsZVxuICovXG5mdW5jdGlvbiB2YWxpZENvbHVtbk5hbWUoY29sdW1uOiBzdHJpbmcpIHtcbiAgICByZXR1cm4gY29sdW1uID09PSBDT0xfRUxMSVBTSVMgfHwgL15bYS16QS1aMC05Xy1dKyQvLmV4ZWMoY29sdW1uKTtcbn1cbi8qKlxuICogVmFsaWRhdGUgYW5kIG5vcm1hbGl6ZSB0aGUgc3RyaW5nIGNvbXBhcmlzb24gb3BlcmF0b3JcbiAqXG4gKiBDb3JyZWN0IGZvciBhIGNvbW1vbiB0eXBvL2NvbmZ1c2lvbiwgdHJlYXQgJz09JyBhcyAnPSdcbiAqL1xuZnVuY3Rpb24gdmFsaWRhdGVTdHJpbmdPcGVyYXRvcihvcGVyYXRvcjogc3RyaW5nKSB7XG4gICAgaWYgKG9wZXJhdG9yID09PSAnPT0nKSB7XG4gICAgICAgIG9wZXJhdG9yID0gJz0nO1xuICAgIH1cbiAgICBpZiAob3BlcmF0b3IgIT09ICc9JyAmJiBvcGVyYXRvciAhPT0gJyE9Jykge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEludmFsaWQgY29tcGFyaXNvbiBvcGVyYXRvciAoJyR7b3BlcmF0b3J9JyksIG11c3QgYmUgZWl0aGVyICc9JyBvciAnIT0nYCk7XG4gICAgfVxuICAgIHJldHVybiBvcGVyYXRvcjtcbn1cbmNvbnN0IFZBTElEX09QRVJBVE9SUyA9IFsnPScsICchPScsICc8JywgJzw9JywgJz4nLCAnPj0nXTtcbi8qKlxuICogVmFsaWRhdGUgYW5kIG5vcm1hbGl6ZSBudW1lcmljYWwgY29tcGFyaXNvbiBvcGVyYXRvcnNcbiAqXG4gKiBDb3JyZWN0IGZvciBhIGNvbW1vbiB0eXBvL2NvbmZ1c2lvbiwgdHJlYXQgJz09JyBhcyAnPSdcbiAqL1xuZnVuY3Rpb24gdmFsaWRhdGVOdW1lcmljYWxPcGVyYXRvcihvcGVyYXRvcjogc3RyaW5nKSB7XG4gICAgLy8gQ29ycmVjdCBmb3IgYSBjb21tb24gdHlwbywgdHJlYXQgJz09JyBhcyAnPSdcbiAgICBpZiAob3BlcmF0b3IgPT09ICc9PScpIHtcbiAgICAgICAgb3BlcmF0b3IgPSAnPSc7XG4gICAgfVxuICAgIGlmIChWQUxJRF9PUEVSQVRPUlMuaW5kZXhPZihvcGVyYXRvcikgPT09IC0xKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCBjb21wYXJpc29uIG9wZXJhdG9yICgnJHtvcGVyYXRvcn0nKSwgbXVzdCBiZSBvbmUgb2YgJHtWQUxJRF9PUEVSQVRPUlMuam9pbignLCAnKX1gKTtcbiAgICB9XG4gICAgcmV0dXJuIG9wZXJhdG9yO1xufVxuLyoqXG4gKiBSZW5kZXIgYSB0YWJsZSByZXN0cmljdGlvblxuICovXG5mdW5jdGlvbiByZW5kZXJSZXN0cmljdGlvbihjb2x1bW46IHN0cmluZywgcmVzdHJpY3Rpb246IENvbHVtblJlc3RyaWN0aW9uKSB7XG4gICAgaWYgKHJlc3RyaWN0aW9uLm51bWJlclZhbHVlICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgcmV0dXJuIGAke2NvbHVtbn0gJHtyZXN0cmljdGlvbi5jb21wYXJpc29ufSAke3Jlc3RyaWN0aW9uLm51bWJlclZhbHVlfWA7XG4gICAgfVxuICAgIGVsc2UgaWYgKHJlc3RyaWN0aW9uLnN0cmluZ1ZhbHVlKSB7XG4gICAgICAgIHJldHVybiBgJHtjb2x1bW59ICR7cmVzdHJpY3Rpb24uY29tcGFyaXNvbn0gJHtxdW90ZVRlcm0ocmVzdHJpY3Rpb24uc3RyaW5nVmFsdWUpfWA7XG4gICAgfVxuICAgIGVsc2Uge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0ludmFsaWQgcmVzdHJpY3Rpb24nKTtcbiAgICB9XG59XG5mdW5jdGlvbiBzdW0oeHM6IG51bWJlcltdKTogbnVtYmVyIHtcbiAgICByZXR1cm4geHMucmVkdWNlKChhLCBjKSA9PiBhICsgYywgMCk7XG59XG4iXX0=