"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrossAccountDestination = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const logs_generated_1 = require("./logs.generated");
/**
 * A new CloudWatch Logs Destination for use in cross-account scenarios
 *
 * CrossAccountDestinations are used to subscribe a Kinesis stream in a
 * different account to a CloudWatch Subscription.
 *
 * Consumers will hardly ever need to use this class. Instead, directly
 * subscribe a Kinesis stream using the integration class in the
 * `@aws-cdk/aws-logs-destinations` package; if necessary, a
 * `CrossAccountDestination` will be created automatically.
 *
 * @resource AWS::Logs::Destination
 */
class CrossAccountDestination extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.destinationName ||
                // In the underlying model, the name is not optional, but we make it so anyway.
                cdk.Lazy.stringValue({ produce: () => this.generateUniqueName() }),
        });
        /**
         * Policy object of this CrossAccountDestination object
         */
        this.policyDocument = new iam.PolicyDocument();
        this.resource = new logs_generated_1.CfnDestination(this, 'Resource', {
            destinationName: this.physicalName,
            // Must be stringified policy
            destinationPolicy: this.lazyStringifiedPolicyDocument(),
            roleArn: props.role.roleArn,
            targetArn: props.targetArn,
        });
        this.destinationArn = this.getResourceArnAttribute(this.resource.attrArn, {
            service: 'logs',
            resource: 'destination',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.destinationName = this.getResourceNameAttribute(this.resource.ref);
    }
    addToPolicy(statement) {
        this.policyDocument.addStatements(statement);
    }
    bind(_scope, _sourceLogGroup) {
        return { arn: this.destinationArn };
    }
    /**
     * Generate a unique Destination name in case the user didn't supply one
     */
    generateUniqueName() {
        // Combination of stack name and LogicalID, which are guaranteed to be unique.
        return cdk.Stack.of(this).stackName + '-' + this.resource.logicalId;
    }
    /**
     * Return a stringified JSON version of the PolicyDocument
     */
    lazyStringifiedPolicyDocument() {
        return cdk.Lazy.stringValue({ produce: () => this.policyDocument.isEmpty ? '' : cdk.Stack.of(this).toJsonString(this.policyDocument),
        });
    }
}
exports.CrossAccountDestination = CrossAccountDestination;
//# sourceMappingURL=data:application/json;base64,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