"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SingletonFunction = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const function_1 = require("./function");
const function_base_1 = require("./function-base");
/**
 * A Lambda that will only ever be added to a stack once.
 *
 * This construct is a way to guarantee that the lambda function will be guaranteed to be part of the stack,
 * once and only once, irrespective of how many times the construct is declared to be part of the stack.
 * This is guaranteed as long as the `uuid` property and the optional `lambdaPurpose` property stay the same
 * whenever they're declared into the stack.
 *
 * @resource AWS::Lambda::Function
 */
class SingletonFunction extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaFunction = this.ensureLambda(props);
        this.permissionsNode = this.lambdaFunction.node;
        this.functionArn = this.lambdaFunction.functionArn;
        this.functionName = this.lambdaFunction.functionName;
        this.role = this.lambdaFunction.role;
        this.grantPrincipal = this.lambdaFunction.grantPrincipal;
        this.canCreatePermissions = true; // Doesn't matter, addPermission is overriden anyway
    }
    addPermission(name, permission) {
        return this.lambdaFunction.addPermission(name, permission);
    }
    /**
     * Using node.addDependency() does not work on this method as the underlying lambda function is modeled
     * as a singleton across the stack. Use this method instead to declare dependencies.
     */
    addDependency(...up) {
        this.lambdaFunction.node.addDependency(...up);
    }
    /**
     * The SingletonFunction construct cannot be added as a dependency of another construct using
     * node.addDependency(). Use this method instead to declare this as a dependency of another construct.
     */
    dependOn(down) {
        down.node.addDependency(this.lambdaFunction);
    }
    /**
     * Returns the construct tree node that corresponds to the lambda function.
     * @internal
     */
    _functionNode() {
        return this.lambdaFunction.node;
    }
    ensureLambda(props) {
        const constructName = (props.lambdaPurpose || 'SingletonLambda') + slugify(props.uuid);
        const existing = cdk.Stack.of(this).node.tryFindChild(constructName);
        if (existing) {
            // Just assume this is true
            return existing;
        }
        return new function_1.Function(cdk.Stack.of(this), constructName, props);
    }
}
exports.SingletonFunction = SingletonFunction;
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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