"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LogRetention = void 0;
const path = require("path");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const code_1 = require("./code");
const runtime_1 = require("./runtime");
const singleton_lambda_1 = require("./singleton-lambda");
/**
 * Creates a custom resource to control the retention policy of a CloudWatch Logs
 * log group. The log group is created if it doesn't already exist. The policy
 * is removed when `retentionDays` is `undefined` or equal to `Infinity`.
 */
class LogRetention extends cdk.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        // Custom resource provider
        const provider = new singleton_lambda_1.SingletonFunction(this, 'Provider', {
            code: code_1.Code.fromAsset(path.join(__dirname, 'log-retention-provider')),
            runtime: runtime_1.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            uuid: 'aae0aa3c-5b4d-4f87-b02d-85b201efdd8a',
            lambdaPurpose: 'LogRetention',
            role: props.role,
        });
        // Duplicate statements will be deduplicated by `PolicyDocument`
        provider.addToRolePolicy(new iam.PolicyStatement({
            actions: ['logs:PutRetentionPolicy', 'logs:DeleteRetentionPolicy'],
            // We need '*' here because we will also put a retention policy on
            // the log group of the provider function. Referencing it's name
            // creates a CF circular dependency.
            resources: ['*'],
        }));
        // Need to use a CfnResource here to prevent lerna dependency cycles
        // @aws-cdk/aws-cloudformation -> @aws-cdk/aws-lambda -> @aws-cdk/aws-cloudformation
        const retryOptions = props.logRetentionRetryOptions;
        const resource = new cdk.CfnResource(this, 'Resource', {
            type: 'Custom::LogRetention',
            properties: {
                ServiceToken: provider.functionArn,
                LogGroupName: props.logGroupName,
                SdkRetry: retryOptions ? {
                    maxRetries: retryOptions.maxRetries,
                    base: (_a = retryOptions.base) === null || _a === void 0 ? void 0 : _a.toMilliseconds(),
                } : undefined,
                RetentionInDays: props.retention === logs.RetentionDays.INFINITE ? undefined : props.retention,
            },
        });
        const logGroupName = resource.getAtt('LogGroupName').toString();
        // Append ':*' at the end of the ARN to match with how CloudFormation does this for LogGroup ARNs
        // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-logs-loggroup.html#aws-resource-logs-loggroup-return-values
        this.logGroupArn = cdk.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            resourceName: `${logGroupName}:*`,
            sep: ':',
        });
    }
}
exports.LogRetention = LogRetention;
//# sourceMappingURL=data:application/json;base64,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