"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.verifyCodeConfig = exports.Function = exports.Tracing = void 0;
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const aws_codeguruprofiler_1 = require("../../aws-codeguruprofiler"); // Automatically re-written from '@aws-cdk/aws-codeguruprofiler'
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const sqs = require("../../aws-sqs"); // Automatically re-written from '@aws-cdk/aws-sqs'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const function_base_1 = require("./function-base");
const function_hash_1 = require("./function-hash");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
const log_retention_1 = require("./log-retention");
/**
 * X-Ray Tracing Modes (https://docs.aws.amazon.com/lambda/latest/dg/API_TracingConfig.html)
 */
var Tracing;
(function (Tracing) {
    /**
     * Lambda will respect any tracing header it receives from an upstream service.
     * If no tracing header is received, Lambda will call X-Ray for a tracing decision.
     */
    Tracing["ACTIVE"] = "Active";
    /**
     * Lambda will only trace the request from an upstream service
     * if it contains a tracing header with "sampled=1"
     */
    Tracing["PASS_THROUGH"] = "PassThrough";
    /**
     * Lambda will not trace any request.
     */
    Tracing["DISABLED"] = "Disabled";
})(Tracing = exports.Tracing || (exports.Tracing = {}));
/**
 * Deploys a file from from inside the construct library as a function.
 *
 * The supplied file is subject to the 4096 bytes limit of being embedded in a
 * CloudFormation template.
 *
 * The construct includes an associated role with the lambda.
 *
 * This construct does not yet reproduce all features from the underlying resource
 * library.
 */
class Function extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.functionName,
        });
        this.permissionsNode = this.node;
        this.canCreatePermissions = true;
        this.layers = [];
        const managedPolicies = new Array();
        // the arn is in the form of - arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
        managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        if (props.vpc) {
            // Policy that will have ENI creation permissions
            managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'));
        }
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies,
        });
        this.grantPrincipal = this.role;
        // add additonal managed policies when necessary
        if (props.filesystem) {
            const config = props.filesystem.config;
            if (config.policies) {
                config.policies.forEach(p => {
                    var _a;
                    (_a = this.role) === null || _a === void 0 ? void 0 : _a.addToPolicy(p);
                });
            }
        }
        for (const statement of (props.initialPolicy || [])) {
            this.role.addToPolicy(statement);
        }
        const code = props.code.bind(this);
        verifyCodeConfig(code, props.runtime);
        let profilingGroupEnvironmentVariables = {};
        if (props.profilingGroup && props.profiling !== false) {
            this.validateProfilingEnvironmentVariables(props);
            props.profilingGroup.grantPublish(this.role);
            profilingGroupEnvironmentVariables = {
                AWS_CODEGURU_PROFILER_GROUP_ARN: core_1.Stack.of(scope).formatArn({
                    service: 'codeguru-profiler',
                    resource: 'profilingGroup',
                    resourceName: props.profilingGroup.profilingGroupName,
                }),
                AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
            };
        }
        else if (props.profiling) {
            this.validateProfilingEnvironmentVariables(props);
            const profilingGroup = new aws_codeguruprofiler_1.ProfilingGroup(this, 'ProfilingGroup');
            profilingGroup.grantPublish(this.role);
            profilingGroupEnvironmentVariables = {
                AWS_CODEGURU_PROFILER_GROUP_ARN: profilingGroup.profilingGroupArn,
                AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
            };
        }
        this.environment = { ...profilingGroupEnvironmentVariables, ...(props.environment || {}) };
        this.deadLetterQueue = this.buildDeadLetterQueue(props);
        const resource = new lambda_generated_1.CfnFunction(this, 'Resource', {
            functionName: this.physicalName,
            description: props.description,
            code: {
                s3Bucket: code.s3Location && code.s3Location.bucketName,
                s3Key: code.s3Location && code.s3Location.objectKey,
                s3ObjectVersion: code.s3Location && code.s3Location.objectVersion,
                zipFile: code.inlineCode,
            },
            layers: core_1.Lazy.listValue({ produce: () => this.layers.map(layer => layer.layerVersionArn) }, { omitEmpty: true }),
            handler: props.handler,
            timeout: props.timeout && props.timeout.toSeconds(),
            runtime: props.runtime.name,
            role: this.role.roleArn,
            environment: core_1.Lazy.anyValue({ produce: () => this.renderEnvironment() }),
            memorySize: props.memorySize,
            vpcConfig: this.configureVpc(props),
            deadLetterConfig: this.buildDeadLetterConfig(this.deadLetterQueue),
            tracingConfig: this.buildTracingConfig(props),
            reservedConcurrentExecutions: props.reservedConcurrentExecutions,
        });
        resource.node.addDependency(this.role);
        this.functionName = this.getResourceNameAttribute(resource.ref);
        this.functionArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'lambda',
            resource: 'function',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.runtime = props.runtime;
        if (props.layers) {
            this.addLayers(...props.layers);
        }
        for (const event of props.events || []) {
            this.addEventSource(event);
        }
        // Log retention
        if (props.logRetention) {
            const logretention = new log_retention_1.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: props.logRetention,
                role: props.logRetentionRole,
                logRetentionRetryOptions: props.logRetentionRetryOptions,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, 'LogGroup', logretention.logGroupArn);
        }
        props.code.bindToResource(resource);
        // Event Invoke Config
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
        this.currentVersionOptions = props.currentVersionOptions;
        if (props.filesystem) {
            const config = props.filesystem.config;
            if (config.dependency) {
                this.node.addDependency(...config.dependency);
            }
            resource.addPropertyOverride('FileSystemConfigs', [
                {
                    LocalMountPath: config.localMountPath,
                    Arn: config.arn,
                },
            ]);
        }
    }
    /**
     * Returns a `lambda.Version` which represents the current version of this
     * Lambda function. A new version will be created every time the function's
     * configuration changes.
     *
     * You can specify options for this version using the `currentVersionOptions`
     * prop when initializing the `lambda.Function`.
     */
    get currentVersion() {
        if (this._currentVersion) {
            return this._currentVersion;
        }
        this._currentVersion = new lambda_version_1.Version(this, 'CurrentVersion', {
            lambda: this,
            ...this.currentVersionOptions,
        });
        return this._currentVersion;
    }
    static fromFunctionArn(scope, id, functionArn) {
        return Function.fromFunctionAttributes(scope, id, { functionArn });
    }
    /**
     * Creates a Lambda function object which represents a function not defined
     * within this stack.
     *
     * @param scope The parent construct
     * @param id The name of the lambda construct
     * @param attrs the attributes of the function to import
     */
    static fromFunctionAttributes(scope, id, attrs) {
        const functionArn = attrs.functionArn;
        const functionName = extractNameFromArn(attrs.functionArn);
        const role = attrs.role;
        class Import extends function_base_1.FunctionBase {
            constructor(s, i) {
                super(s, i);
                this.functionName = functionName;
                this.functionArn = functionArn;
                this.role = role;
                this.permissionsNode = this.node;
                this.canCreatePermissions = false;
                this.grantPrincipal = role || new iam.UnknownPrincipal({ resource: this });
                if (attrs.securityGroup) {
                    this._connections = new ec2.Connections({
                        securityGroups: [attrs.securityGroup],
                    });
                }
                else if (attrs.securityGroupId) {
                    this._connections = new ec2.Connections({
                        securityGroups: [ec2.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)],
                    });
                }
            }
        }
        return new Import(scope, id);
    }
    /**
     * Return the given named metric for this Lambda
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Lambda',
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the number of Errors executing all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllErrors(props) {
        return this.metricAll('Errors', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the Duration executing all Lambdas
     *
     * @default average over 5 minutes
     */
    static metricAllDuration(props) {
        return this.metricAll('Duration', props);
    }
    /**
     * Metric for the number of invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllInvocations(props) {
        return this.metricAll('Invocations', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of throttled invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllThrottles(props) {
        return this.metricAll('Throttles', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllConcurrentExecutions(props) {
        // Mini-FAQ: why max? This metric is a gauge that is emitted every
        // minute, so either max or avg or a percentile make sense (but sum
        // doesn't). Max is more sensitive to spiky load changes which is
        // probably what you're interested in if you're looking at this metric
        // (Load spikes may lead to concurrent execution errors that would
        // otherwise not be visible in the avg)
        return this.metricAll('ConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Metric for the number of unreserved concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllUnreservedConcurrentExecutions(props) {
        return this.metricAll('UnreservedConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Adds an environment variable to this Lambda function.
     * If this is a ref to a Lambda function, this operation results in a no-op.
     * @param key The environment variable key.
     * @param value The environment variable's value.
     */
    addEnvironment(key, value) {
        this.environment[key] = value;
        return this;
    }
    /**
     * Adds one or more Lambda Layers to this Lambda function.
     *
     * @param layers the layers to be added.
     *
     * @throws if there are already 5 layers on this function, or the layer is incompatible with this function's runtime.
     */
    addLayers(...layers) {
        for (const layer of layers) {
            if (this.layers.length === 5) {
                throw new Error('Unable to add layer: this lambda function already uses 5 layers.');
            }
            if (layer.compatibleRuntimes && !layer.compatibleRuntimes.find(runtime => runtime.runtimeEquals(this.runtime))) {
                const runtimes = layer.compatibleRuntimes.map(runtime => runtime.name).join(', ');
                throw new Error(`This lambda function uses a runtime that is incompatible with this layer (${this.runtime.name} is not in [${runtimes}])`);
            }
            this.layers.push(layer);
        }
    }
    /**
     * Add a new version for this Lambda
     *
     * If you want to deploy through CloudFormation and use aliases, you need to
     * add a new version (with a new name) to your Lambda every time you want to
     * deploy an update. An alias can then refer to the newly created Version.
     *
     * All versions should have distinct names, and you should not delete versions
     * as long as your Alias needs to refer to them.
     *
     * @param name A unique name for this version.
     * @param codeSha256 The SHA-256 hash of the most recently deployed Lambda
     *  source code, or omit to skip validation.
     * @param description A description for this version.
     * @param provisionedExecutions A provisioned concurrency configuration for a
     * function's version.
     * @param asyncInvokeConfig configuration for this version when it is invoked
     * asynchronously.
     * @returns A new Version object.
     *
     * @deprecated This method will create an AWS::Lambda::Version resource which
     * snapshots the AWS Lambda function *at the time of its creation* and it
     * won't get updated when the function changes. Instead, use
     * `this.currentVersion` to obtain a reference to a version resource that gets
     * automatically recreated when the function configuration (or code) changes.
     */
    addVersion(name, codeSha256, description, provisionedExecutions, asyncInvokeConfig = {}) {
        return new lambda_version_1.Version(this, 'Version' + name, {
            lambda: this,
            codeSha256,
            description,
            provisionedConcurrentExecutions: provisionedExecutions,
            ...asyncInvokeConfig,
        });
    }
    /**
     * The LogGroup where the Lambda function's logs are made available.
     *
     * If either `logRetention` is set or this property is called, a CloudFormation custom resource is added to the stack that
     * pre-creates the log group as part of the stack deployment, if it already doesn't exist, and sets the correct log retention
     * period (never expire, by default).
     *
     * Further, if the log group already exists and the `logRetention` is not set, the custom resource will reset the log retention
     * to never expire even if it was configured with a different value.
     */
    get logGroup() {
        if (!this._logGroup) {
            const logretention = new log_retention_1.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: logs.RetentionDays.INFINITE,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, `${this.node.id}-LogGroup`, logretention.logGroupArn);
        }
        return this._logGroup;
    }
    prepare() {
        super.prepare();
        // if we have a current version resource, override it's logical id
        // so that it includes the hash of the function code and it's configuration.
        if (this._currentVersion) {
            const stack = core_1.Stack.of(this);
            const cfn = this._currentVersion.node.defaultChild;
            const originalLogicalId = stack.resolve(cfn.logicalId);
            const hash = function_hash_1.calculateFunctionHash(this);
            const logicalId = function_hash_1.trimFromStart(originalLogicalId, 255 - 32);
            cfn.overrideLogicalId(`${logicalId}${hash}`);
        }
    }
    renderEnvironment() {
        if (!this.environment || Object.keys(this.environment).length === 0) {
            return undefined;
        }
        // for backwards compatibility we do not sort environment variables in case
        // _currentVersion is not defined. otherwise, this would have invalidated
        // the template, and for example, may cause unneeded updates for nested
        // stacks.
        if (!this._currentVersion) {
            return {
                variables: this.environment,
            };
        }
        // sort environment so the hash of the function used to create
        // `currentVersion` is not affected by key order (this is how lambda does
        // it).
        const variables = {};
        for (const key of Object.keys(this.environment).sort()) {
            variables[key] = this.environment[key];
        }
        return { variables };
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * Lambda creation properties.
     */
    configureVpc(props) {
        if ((props.securityGroup || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroup\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if (props.securityGroup && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroup.');
        }
        let securityGroups;
        if (props.securityGroup && props.securityGroups) {
            throw new Error('Only one of the function props, securityGroup or securityGroups, is allowed');
        }
        if (props.securityGroups) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic security group for Lambda Function ' + this.node.uniqueId,
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        if (props.filesystem) {
            if (props.filesystem.config.connections) {
                props.filesystem.config.connections.allowDefaultPortFrom(this);
            }
        }
        // Pick subnets, make sure they're not Public. Routing through an IGW
        // won't work because the ENIs don't get a Public IP.
        // Why are we not simply forcing vpcSubnets? Because you might still be choosing
        // Isolated networks or selecting among 2 sets of Private subnets by name.
        const { subnetIds } = props.vpc.selectSubnets(props.vpcSubnets);
        const publicSubnetIds = new Set(props.vpc.publicSubnets.map(s => s.subnetId));
        for (const subnetId of subnetIds) {
            if (publicSubnetIds.has(subnetId)) {
                throw new Error('Not possible to place Lambda Functions in a Public subnet');
            }
        }
        // List can't be empty here, if we got this far you intended to put your Lambda
        // in subnets. We're going to guarantee that we get the nice error message by
        // making VpcNetwork do the selection again.
        return {
            subnetIds,
            securityGroupIds: securityGroups.map(sg => sg.securityGroupId),
        };
    }
    buildDeadLetterQueue(props) {
        if (props.deadLetterQueue && props.deadLetterQueueEnabled === false) {
            throw Error('deadLetterQueue defined but deadLetterQueueEnabled explicitly set to false');
        }
        if (!props.deadLetterQueue && !props.deadLetterQueueEnabled) {
            return undefined;
        }
        const deadLetterQueue = props.deadLetterQueue || new sqs.Queue(this, 'DeadLetterQueue', {
            retentionPeriod: core_1.Duration.days(14),
        });
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['sqs:SendMessage'],
            resources: [deadLetterQueue.queueArn],
        }));
        return deadLetterQueue;
    }
    buildDeadLetterConfig(deadLetterQueue) {
        if (deadLetterQueue) {
            return {
                targetArn: deadLetterQueue.queueArn,
            };
        }
        else {
            return undefined;
        }
    }
    buildTracingConfig(props) {
        if (props.tracing === undefined || props.tracing === Tracing.DISABLED) {
            return undefined;
        }
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['xray:PutTraceSegments', 'xray:PutTelemetryRecords'],
            resources: ['*'],
        }));
        return {
            mode: props.tracing,
        };
    }
    validateProfilingEnvironmentVariables(props) {
        if (props.environment && (props.environment.AWS_CODEGURU_PROFILER_GROUP_ARN || props.environment.AWS_CODEGURU_PROFILER_ENABLED)) {
            throw new Error('AWS_CODEGURU_PROFILER_GROUP_ARN and AWS_CODEGURU_PROFILER_ENABLED must not be set when profiling options enabled');
        }
    }
}
exports.Function = Function;
/**
 * Given an opaque (token) ARN, returns a CloudFormation expression that extracts the function
 * name from the ARN.
 *
 * Function ARNs look like this:
 *
 *   arn:aws:lambda:region:account-id:function:function-name
 *
 * ..which means that in order to extract the `function-name` component from the ARN, we can
 * split the ARN using ":" and select the component in index 6.
 *
 * @returns `FnSelect(6, FnSplit(':', arn))`
 */
function extractNameFromArn(arn) {
    return core_1.Fn.select(6, core_1.Fn.split(':', arn));
}
function verifyCodeConfig(code, runtime) {
    // mutually exclusive
    if ((!code.inlineCode && !code.s3Location) || (code.inlineCode && code.s3Location)) {
        throw new Error('lambda.Code must specify one of "inlineCode" or "s3Location" but not both');
    }
    // if this is inline code, check that the runtime supports
    if (code.inlineCode && !runtime.supportsInlineCode) {
        throw new Error(`Inline source not allowed for ${runtime.name}`);
    }
}
exports.verifyCodeConfig = verifyCodeConfig;
//# sourceMappingURL=data:application/json;base64,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