"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findUp = exports.nodeMajorVersion = exports.parseStackTrace = void 0;
const fs = require("fs");
const path = require("path");
// From https://github.com/errwischt/stacktrace-parser/blob/master/src/stack-trace-parser.js
const STACK_RE = /^\s*at (?:((?:\[object object\])?[^\\/]+(?: \[as \S+\])?) )?\(?(.*?):(\d+)(?::(\d+))?\)?\s*$/i;
/**
 * Parses the stack trace of an error
 */
function parseStackTrace(error) {
    const err = error || new Error();
    if (!err.stack) {
        return [];
    }
    const lines = err.stack.split('\n');
    const stackTrace = [];
    for (const line of lines) {
        const results = STACK_RE.exec(line);
        if (results) {
            stackTrace.push({
                file: results[2],
                methodName: results[1],
                lineNumber: parseInt(results[3], 10),
                column: parseInt(results[4], 10),
            });
        }
    }
    return stackTrace;
}
exports.parseStackTrace = parseStackTrace;
/**
 * Returns the major version of node installation
 */
function nodeMajorVersion() {
    return parseInt(process.versions.node.split('.')[0], 10);
}
exports.nodeMajorVersion = nodeMajorVersion;
/**
 * Find a file by walking up parent directories
 */
function findUp(name, directory = process.cwd()) {
    const absoluteDirectory = path.resolve(directory);
    if (fs.existsSync(path.join(directory, name))) {
        return directory;
    }
    const { root } = path.parse(absoluteDirectory);
    if (absoluteDirectory === root) {
        return undefined;
    }
    return findUp(name, path.dirname(absoluteDirectory));
}
exports.findUp = findUp;
//# sourceMappingURL=data:application/json;base64,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