"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Policy = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const iam_generated_1 = require("./iam.generated");
const policy_document_1 = require("./policy-document");
const util_1 = require("./util");
/**
 * The AWS::IAM::Policy resource associates an IAM policy with IAM users, roles,
 * or groups. For more information about IAM policies, see [Overview of IAM
 * Policies](http://docs.aws.amazon.com/IAM/latest/UserGuide/policies_overview.html)
 * in the IAM User Guide guide.
 */
class Policy extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.policyName ||
                // generatePolicyName will take the last 128 characters of the logical id since
                // policy names are limited to 128. the last 8 chars are a stack-unique hash, so
                // that shouod be sufficient to ensure uniqueness within a principal.
                core_1.Lazy.stringValue({ produce: () => util_1.generatePolicyName(scope, resource.logicalId) }),
        });
        /**
         * The policy document.
         */
        this.document = new policy_document_1.PolicyDocument();
        this.roles = new Array();
        this.users = new Array();
        this.groups = new Array();
        this.referenceTaken = false;
        const resource = new iam_generated_1.CfnPolicy(this, 'Resource', {
            policyDocument: this.document,
            policyName: this.physicalName,
            roles: util_1.undefinedIfEmpty(() => this.roles.map(r => r.roleName)),
            users: util_1.undefinedIfEmpty(() => this.users.map(u => u.userName)),
            groups: util_1.undefinedIfEmpty(() => this.groups.map(g => g.groupName)),
        });
        this._policyName = this.physicalName;
        this.force = props.force !== undefined ? props.force : false;
        if (props.users) {
            props.users.forEach(u => this.attachToUser(u));
        }
        if (props.groups) {
            props.groups.forEach(g => this.attachToGroup(g));
        }
        if (props.roles) {
            props.roles.forEach(r => this.attachToRole(r));
        }
        if (props.statements) {
            props.statements.forEach(p => this.addStatements(p));
        }
    }
    /**
     * Import a policy in this app based on its name
     */
    static fromPolicyName(scope, id, policyName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.policyName = policyName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a statement to the policy document.
     */
    addStatements(...statement) {
        this.document.addStatements(...statement);
    }
    /**
     * Attaches this policy to a user.
     */
    attachToUser(user) {
        if (this.users.find(u => u === user)) {
            return;
        }
        this.users.push(user);
        user.attachInlinePolicy(this);
    }
    /**
     * Attaches this policy to a role.
     */
    attachToRole(role) {
        if (this.roles.find(r => r === role)) {
            return;
        }
        this.roles.push(role);
        role.attachInlinePolicy(this);
    }
    /**
     * Attaches this policy to a group.
     */
    attachToGroup(group) {
        if (this.groups.find(g => g === group)) {
            return;
        }
        this.groups.push(group);
        group.attachInlinePolicy(this);
    }
    /**
     * The name of this policy.
     *
     * @attribute
     */
    get policyName() {
        this.referenceTaken = true;
        return this._policyName;
    }
    validate() {
        const result = new Array();
        // validate that the policy document is not empty
        if (this.document.isEmpty) {
            if (this.force) {
                result.push('Policy created with force=true is empty. You must add statements to the policy');
            }
            if (!this.force && this.referenceTaken) {
                result.push('This Policy has been referenced by a resource, so it must contain at least one statement.');
            }
        }
        // validate that the policy is attached to at least one principal (role, user or group).
        if (!this.isAttached) {
            if (this.force) {
                result.push('Policy created with force=true must be attached to at least one principal: user, group or role');
            }
            if (!this.force && this.referenceTaken) {
                result.push('This Policy has been referenced by a resource, so it must be attached to at least one user, group or role.');
            }
        }
        return result;
    }
    prepare() {
        // Remove the resource if it shouldn't exist. This will prevent it from being rendered to the template.
        const shouldExist = this.force || this.referenceTaken || (!this.document.isEmpty && this.isAttached);
        if (!shouldExist) {
            this.node.tryRemoveChild('Resource');
        }
    }
    /**
     * Whether the policy resource has been attached to any identity
     */
    get isAttached() {
        return this.groups.length + this.users.length + this.roles.length > 0;
    }
}
exports.Policy = Policy;
//# sourceMappingURL=data:application/json;base64,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