"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const diff_1 = require("./diff");
const external_1 = require("./external");
async function handler(event) {
    if (event.RequestType === 'Create') {
        return onCreate(event);
    }
    if (event.RequestType === 'Update') {
        return onUpdate(event);
    }
    if (event.RequestType === 'Delete') {
        return onDelete(event);
    }
    throw new Error('invalid request type');
}
exports.handler = handler;
async function onCreate(event) {
    var _a, _b;
    const issuerUrl = event.ResourceProperties.Url;
    const thumbprints = ((_a = event.ResourceProperties.ThumbprintList) !== null && _a !== void 0 ? _a : []).sort(); // keep sorted for UPDATE
    const clients = ((_b = event.ResourceProperties.ClientIDList) !== null && _b !== void 0 ? _b : []).sort();
    if (thumbprints.length === 0) {
        thumbprints.push(await external_1.external.downloadThumbprint(issuerUrl));
    }
    const resp = await external_1.external.createOpenIDConnectProvider({
        Url: issuerUrl,
        ClientIDList: clients,
        ThumbprintList: thumbprints,
    });
    return {
        PhysicalResourceId: resp.OpenIDConnectProviderArn,
    };
}
async function onUpdate(event) {
    var _a, _b;
    const issuerUrl = event.ResourceProperties.Url;
    const thumbprints = ((_a = event.ResourceProperties.ThumbprintList) !== null && _a !== void 0 ? _a : []).sort(); // keep sorted for UPDATE
    const clients = ((_b = event.ResourceProperties.ClientIDList) !== null && _b !== void 0 ? _b : []).sort();
    // determine which update we are talking about.
    const oldIssuerUrl = event.OldResourceProperties.Url;
    // if this is a URL update, then we basically create a new resource and cfn will delete the old one
    // since the physical resource ID will change.
    if (oldIssuerUrl !== issuerUrl) {
        return onCreate({ ...event, RequestType: 'Create' });
    }
    const providerArn = event.PhysicalResourceId;
    // if thumbprints changed, we can update in-place, but bear in mind that if the new thumbprint list
    // is empty, we will grab it from the server like we do in CREATE
    const oldThumbprints = (event.OldResourceProperties.ThumbprintList || []).sort();
    if (JSON.stringify(oldThumbprints) !== JSON.stringify(thumbprints)) {
        const thumbprintList = thumbprints.length > 0 ? thumbprints : [await external_1.external.downloadThumbprint(issuerUrl)];
        external_1.external.log('updating thumbprint list from', oldThumbprints, 'to', thumbprints);
        await external_1.external.updateOpenIDConnectProviderThumbprint({
            OpenIDConnectProviderArn: providerArn,
            ThumbprintList: thumbprintList,
        });
        // don't return, we might have more updates...
    }
    // if client ID list has changed, determine "diff" because the API is add/remove
    const oldClients = (event.OldResourceProperties.ClientIDList || []).sort();
    const diff = diff_1.arrayDiff(oldClients, clients);
    external_1.external.log(`client ID diff: ${JSON.stringify(diff)}`);
    for (const addClient of diff.adds) {
        external_1.external.log(`adding client id "${addClient}" to provider ${providerArn}`);
        await external_1.external.addClientIDToOpenIDConnectProvider({
            OpenIDConnectProviderArn: providerArn,
            ClientID: addClient,
        });
    }
    for (const deleteClient of diff.deletes) {
        external_1.external.log(`removing client id "${deleteClient}" from provider ${providerArn}`);
        await external_1.external.removeClientIDFromOpenIDConnectProvider({
            OpenIDConnectProviderArn: providerArn,
            ClientID: deleteClient,
        });
    }
    return;
}
async function onDelete(deleteEvent) {
    await external_1.external.deleteOpenIDConnectProvider({
        OpenIDConnectProviderArn: deleteEvent.PhysicalResourceId,
    });
}
//# sourceMappingURL=data:application/json;base64,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