"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ManagedPolicy = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const iam_generated_1 = require("./iam.generated");
const policy_document_1 = require("./policy-document");
const util_1 = require("./util");
/**
 * Managed policy
 *
 */
class ManagedPolicy extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.managedPolicyName,
        });
        /**
         * The policy document.
         */
        this.document = new policy_document_1.PolicyDocument();
        this.roles = new Array();
        this.users = new Array();
        this.groups = new Array();
        this.description = props.description || '';
        this.path = props.path || '/';
        if (props.document) {
            this.document = props.document;
        }
        const resource = new iam_generated_1.CfnManagedPolicy(this, 'Resource', {
            policyDocument: this.document,
            managedPolicyName: this.physicalName,
            description: this.description,
            path: this.path,
            roles: util_1.undefinedIfEmpty(() => this.roles.map(r => r.roleName)),
            users: util_1.undefinedIfEmpty(() => this.users.map(u => u.userName)),
            groups: util_1.undefinedIfEmpty(() => this.groups.map(g => g.groupName)),
        });
        if (props.users) {
            props.users.forEach(u => this.attachToUser(u));
        }
        if (props.groups) {
            props.groups.forEach(g => this.attachToGroup(g));
        }
        if (props.roles) {
            props.roles.forEach(r => this.attachToRole(r));
        }
        if (props.statements) {
            props.statements.forEach(p => this.addStatements(p));
        }
        // arn:aws:iam::123456789012:policy/teststack-CreateTestDBPolicy-16M23YE3CS700
        this.managedPolicyName = this.getResourceNameAttribute(core_1.Stack.of(this).parseArn(resource.ref, '/').resourceName);
        this.managedPolicyArn = this.getResourceArnAttribute(resource.ref, {
            region: '',
            service: 'iam',
            resource: 'policy',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import a customer managed policy from the managedPolicyName.
     *
     * For this managed policy, you only need to know the name to be able to use it.
     *
     */
    static fromManagedPolicyName(scope, id, managedPolicyName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.managedPolicyArn = core_1.Stack.of(scope).formatArn({
                    service: 'iam',
                    region: '',
                    account: core_1.Stack.of(scope).account,
                    resource: 'policy',
                    resourceName: managedPolicyName,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an external managed policy by ARN.
     *
     * For this managed policy, you only need to know the ARN to be able to use it.
     * This can be useful if you got the ARN from a CloudFormation Export.
     *
     * If the imported Managed Policy ARN is a Token (such as a
     * `CfnParameter.valueAsString` or a `Fn.importValue()`) *and* the referenced
     * managed policy has a `path` (like `arn:...:policy/AdminPolicy/AdminAllow`), the
     * `managedPolicyName` property will not resolve to the correct value. Instead it
     * will resolve to the first path component. We unfortunately cannot express
     * the correct calculation of the full path name as a CloudFormation
     * expression. In this scenario the Managed Policy ARN should be supplied without the
     * `path` in order to resolve the correct managed policy resource.
     *
     * @param scope construct scope
     * @param id construct id
     * @param managedPolicyArn the ARN of the managed policy to import
     */
    static fromManagedPolicyArn(scope, id, managedPolicyArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.managedPolicyArn = managedPolicyArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import a managed policy from one of the policies that AWS manages.
     *
     * For this managed policy, you only need to know the name to be able to use it.
     *
     * Some managed policy names start with "service-role/", some start with
     * "job-function/", and some don't start with anything. Do include the
     * prefix when constructing this object.
     */
    static fromAwsManagedPolicyName(managedPolicyName) {
        class AwsManagedPolicy {
            constructor() {
                this.managedPolicyArn = core_1.Lazy.stringValue({
                    produce(ctx) {
                        return core_1.Stack.of(ctx.scope).formatArn({
                            service: 'iam',
                            region: '',
                            account: 'aws',
                            resource: 'policy',
                            resourceName: managedPolicyName,
                        });
                    },
                });
            }
        }
        return new AwsManagedPolicy();
    }
    /**
     * Adds a statement to the policy document.
     */
    addStatements(...statement) {
        this.document.addStatements(...statement);
    }
    /**
     * Attaches this policy to a user.
     */
    attachToUser(user) {
        if (this.users.find(u => u === user)) {
            return;
        }
        this.users.push(user);
    }
    /**
     * Attaches this policy to a role.
     */
    attachToRole(role) {
        if (this.roles.find(r => r === role)) {
            return;
        }
        this.roles.push(role);
    }
    /**
     * Attaches this policy to a group.
     */
    attachToGroup(group) {
        if (this.groups.find(g => g === group)) {
            return;
        }
        this.groups.push(group);
    }
    validate() {
        const result = new Array();
        // validate that the policy document is not empty
        if (this.document.isEmpty) {
            result.push('Managed Policy is empty. You must add statements to the policy');
        }
        return result;
    }
}
exports.ManagedPolicy = ManagedPolicy;
//# sourceMappingURL=data:application/json;base64,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