"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationListener = void 0;
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const base_listener_1 = require("../shared/base-listener");
const enums_1 = require("../shared/enums");
const listener_certificate_1 = require("../shared/listener-certificate");
const util_1 = require("../shared/util");
const application_listener_action_1 = require("./application-listener-action");
const application_listener_certificate_1 = require("./application-listener-certificate");
const application_listener_rule_1 = require("./application-listener-rule");
const application_target_group_1 = require("./application-target-group");
/**
 * Define an ApplicationListener
 *
 * @resource AWS::ElasticLoadBalancingV2::Listener
 */
class ApplicationListener extends base_listener_1.BaseListener {
    constructor(scope, id, props) {
        const [protocol, port] = util_1.determineProtocolAndPort(props.protocol, props.port);
        if (protocol === undefined || port === undefined) {
            throw new Error('At least one of \'port\' or \'protocol\' is required');
        }
        super(scope, id, {
            loadBalancerArn: props.loadBalancer.loadBalancerArn,
            certificates: core_1.Lazy.anyValue({ produce: () => this.certificateArns.map(certificateArn => ({ certificateArn })) }, { omitEmptyArray: true }),
            protocol,
            port,
            sslPolicy: props.sslPolicy,
        });
        this.loadBalancer = props.loadBalancer;
        this.protocol = protocol;
        this.certificateArns = [];
        // Attach certificates
        if (props.certificateArns && props.certificateArns.length > 0) {
            this.addCertificateArns('ListenerCertificate', props.certificateArns);
        }
        if (props.certificates && props.certificates.length > 0) {
            this.addCertificates('DefaultCertificates', props.certificates);
        }
        // This listener edits the securitygroup of the load balancer,
        // but adds its own default port.
        this.connections = new ec2.Connections({
            securityGroups: props.loadBalancer.connections.securityGroups,
            defaultPort: ec2.Port.tcp(port),
        });
        if (props.defaultAction && props.defaultTargetGroups) {
            throw new Error('Specify at most one of \'defaultAction\' and \'defaultTargetGroups\'');
        }
        if (props.defaultAction) {
            this.setDefaultAction(props.defaultAction);
        }
        if (props.defaultTargetGroups) {
            this.setDefaultAction(application_listener_action_1.ListenerAction.forward(props.defaultTargetGroups));
        }
        if (props.open !== false) {
            this.connections.allowDefaultPortFrom(ec2.Peer.anyIpv4(), `Allow from anyone on port ${port}`);
            if (this.loadBalancer.ipAddressType === enums_1.IpAddressType.DUAL_STACK) {
                this.connections.allowDefaultPortFrom(ec2.Peer.anyIpv6(), `Allow from anyone on port ${port}`);
            }
        }
    }
    /**
     * Import an existing listener
     */
    static fromApplicationListenerAttributes(scope, id, attrs) {
        return new ImportedApplicationListener(scope, id, attrs);
    }
    /**
     * Add one or more certificates to this listener.
     *
     * After the first certificate, this creates ApplicationListenerCertificates
     * resources since cloudformation requires the certificates array on the
     * listener resource to have a length of 1.
     *
     * @deprecated Use `addCertificates` instead.
     */
    addCertificateArns(id, arns) {
        this.addCertificates(id, arns.map(listener_certificate_1.ListenerCertificate.fromArn));
    }
    /**
     * Add one or more certificates to this listener.
     *
     * After the first certificate, this creates ApplicationListenerCertificates
     * resources since cloudformation requires the certificates array on the
     * listener resource to have a length of 1.
     */
    addCertificates(id, certificates) {
        const additionalCerts = [...certificates];
        if (this.certificateArns.length === 0 && additionalCerts.length > 0) {
            const first = additionalCerts.splice(0, 1)[0];
            this.certificateArns.push(first.certificateArn);
        }
        if (additionalCerts.length > 0) {
            new application_listener_certificate_1.ApplicationListenerCertificate(this, id, {
                listener: this,
                certificates: additionalCerts,
            });
        }
    }
    /**
     * Perform the given default action on incoming requests
     *
     * This allows full control of the default action of the load balancer,
     * including Action chaining, fixed responses and redirect responses. See
     * the `ListenerAction` class for all options.
     *
     * It's possible to add routing conditions to the Action added in this way.
     * At least one Action must be added without conditions (which becomes the
     * default Action).
     */
    addAction(id, props) {
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            //
            // TargetGroup.registerListener is called inside ApplicationListenerRule.
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                conditions: props.conditions,
                hostHeader: props.hostHeader,
                pathPattern: props.pathPattern,
                pathPatterns: props.pathPatterns,
                priority: props.priority,
                action: props.action,
            });
        }
        else {
            // New default target with these targetgroups
            this.setDefaultAction(props.action);
        }
    }
    /**
     * Load balance incoming requests to the given target groups.
     *
     * All target groups will be load balanced to with equal weight and without
     * stickiness. For a more complex configuration than that, use `addAction()`.
     *
     * It's possible to add routing conditions to the TargetGroups added in this
     * way. At least one TargetGroup must be added without conditions (which will
     * become the default Action for this listener).
     */
    addTargetGroups(id, props) {
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            //
            // TargetGroup.registerListener is called inside ApplicationListenerRule.
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                conditions: props.conditions,
                hostHeader: props.hostHeader,
                pathPattern: props.pathPattern,
                pathPatterns: props.pathPatterns,
                priority: props.priority,
                targetGroups: props.targetGroups,
            });
        }
        else {
            // New default target with these targetgroups
            this.setDefaultAction(application_listener_action_1.ListenerAction.forward(props.targetGroups));
        }
    }
    /**
     * Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates an ApplicationTargetGroup for the targets
     * involved, and a 'forward' action to route traffic to the given TargetGroup.
     *
     * If you want more control over the precise setup, create the TargetGroup
     * and use `addAction` yourself.
     *
     * It's possible to add conditions to the targets added in this way. At least
     * one set of targets must be added without conditions.
     *
     * @returns The newly created target group
     */
    addTargets(id, props) {
        if (!this.loadBalancer.vpc) {
            // eslint-disable-next-line max-len
            throw new Error('Can only call addTargets() when using a constructed Load Balancer or an imported Load Balancer with specified vpc; construct a new TargetGroup and use addTargetGroup');
        }
        const group = new application_target_group_1.ApplicationTargetGroup(this, id + 'Group', {
            deregistrationDelay: props.deregistrationDelay,
            healthCheck: props.healthCheck,
            port: props.port,
            protocol: props.protocol,
            slowStart: props.slowStart,
            stickinessCookieDuration: props.stickinessCookieDuration,
            targetGroupName: props.targetGroupName,
            targets: props.targets,
            vpc: this.loadBalancer.vpc,
        });
        this.addTargetGroups(id, {
            conditions: props.conditions,
            hostHeader: props.hostHeader,
            pathPattern: props.pathPattern,
            pathPatterns: props.pathPatterns,
            priority: props.priority,
            targetGroups: [group],
        });
        return group;
    }
    /**
     * Add a fixed response
     *
     * @deprecated Use `addAction()` instead
     */
    addFixedResponse(id, props) {
        checkAddRuleProps(props);
        const fixedResponse = {
            statusCode: props.statusCode,
            contentType: props.contentType,
            messageBody: props.messageBody,
        };
        application_listener_rule_1.validateFixedResponse(fixedResponse);
        if (props.priority) {
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                priority: props.priority,
                fixedResponse,
                ...props,
            });
        }
        else {
            this.setDefaultAction(application_listener_action_1.ListenerAction.fixedResponse(core_1.Token.asNumber(props.statusCode), {
                contentType: props.contentType,
                messageBody: props.messageBody,
            }));
        }
    }
    /**
     * Add a redirect response
     *
     * @deprecated Use `addAction()` instead
     */
    addRedirectResponse(id, props) {
        checkAddRuleProps(props);
        const redirectResponse = {
            host: props.host,
            path: props.path,
            port: props.port,
            protocol: props.protocol,
            query: props.query,
            statusCode: props.statusCode,
        };
        application_listener_rule_1.validateRedirectResponse(redirectResponse);
        if (props.priority) {
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                priority: props.priority,
                redirectResponse,
                ...props,
            });
        }
        else {
            this.setDefaultAction(application_listener_action_1.ListenerAction.redirect({
                host: props.host,
                path: props.path,
                port: props.port,
                protocol: props.protocol,
                query: props.query,
                permanent: props.statusCode === 'HTTP_301',
            }));
        }
    }
    /**
     * Register that a connectable that has been added to this load balancer.
     *
     * Don't call this directly. It is called by ApplicationTargetGroup.
     */
    registerConnectable(connectable, portRange) {
        connectable.connections.allowFrom(this.loadBalancer, portRange, 'Load balancer to target');
    }
    /**
     * Validate this listener.
     */
    validate() {
        const errors = super.validate();
        if (this.protocol === enums_1.ApplicationProtocol.HTTPS && this.certificateArns.length === 0) {
            errors.push('HTTPS Listener needs at least one certificate (call addCertificates)');
        }
        return errors;
    }
    /**
     * Wrapper for _setDefaultAction which does a type-safe bind
     */
    setDefaultAction(action) {
        action.bind(this, this);
        this._setDefaultAction(action);
    }
}
exports.ApplicationListener = ApplicationListener;
class ImportedApplicationListener extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.listenerArn = props.listenerArn;
        const defaultPort = props.defaultPort !== undefined ? ec2.Port.tcp(props.defaultPort) : undefined;
        let securityGroup;
        if (props.securityGroup) {
            securityGroup = props.securityGroup;
        }
        else if (props.securityGroupId) {
            securityGroup = ec2.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', props.securityGroupId, {
                allowAllOutbound: props.securityGroupAllowsAllOutbound,
            });
        }
        else {
            throw new Error('Either `securityGroup` or `securityGroupId` must be specified to import an application listener.');
        }
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort,
        });
    }
    /**
     * Add one or more certificates to this listener.
     */
    addCertificateArns(id, arns) {
        new application_listener_certificate_1.ApplicationListenerCertificate(this, id, {
            listener: this,
            certificateArns: arns,
        });
    }
    /**
     * Load balance incoming requests to the given target groups.
     *
     * It's possible to add conditions to the TargetGroups added in this way.
     * At least one TargetGroup must be added without conditions.
     */
    addTargetGroups(id, props) {
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            new application_listener_rule_1.ApplicationListenerRule(this, id, {
                listener: this,
                hostHeader: props.hostHeader,
                pathPattern: props.pathPattern,
                pathPatterns: props.pathPatterns,
                priority: props.priority,
                targetGroups: props.targetGroups,
            });
        }
        else {
            throw new Error('Cannot add default Target Groups to imported ApplicationListener');
        }
    }
    /**
     * Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates an ApplicationTargetGroup for the targets
     * involved.
     *
     * It's possible to add conditions to the targets added in this way. At least
     * one set of targets must be added without conditions.
     *
     * @returns The newly created target group
     */
    addTargets(_id, _props) {
        // eslint-disable-next-line max-len
        throw new Error('Can only call addTargets() when using a constructed ApplicationListener; construct a new TargetGroup and use addTargetGroup.');
    }
    /**
     * Register that a connectable that has been added to this load balancer.
     *
     * Don't call this directly. It is called by ApplicationTargetGroup.
     */
    registerConnectable(connectable, portRange) {
        this.connections.allowTo(connectable, portRange, 'Load balancer to target');
    }
}
function checkAddRuleProps(props) {
    var _a;
    const conditionsCount = ((_a = props.conditions) === null || _a === void 0 ? void 0 : _a.length) || 0;
    const hasAnyConditions = conditionsCount !== 0 ||
        props.hostHeader !== undefined || props.pathPattern !== undefined || props.pathPatterns !== undefined;
    const hasPriority = props.priority !== undefined;
    if (hasAnyConditions !== hasPriority) {
        throw new Error('Setting \'conditions\', \'pathPattern\' or \'hostHeader\' also requires \'priority\', and vice versa');
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwbGljYXRpb24tbGlzdGVuZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhcHBsaWNhdGlvbi1saXN0ZW5lci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx3Q0FBd0MsQ0FBQyxtREFBbUQ7QUFDNUYsd0NBQXNGLENBQUMsZ0RBQWdEO0FBQ3ZJLDJEQUF1RDtBQUV2RCwyQ0FBZ0Y7QUFDaEYseUVBQTJGO0FBQzNGLHlDQUEwRDtBQUMxRCwrRUFBK0Q7QUFDL0QseUZBQW9GO0FBQ3BGLDJFQUF3SjtBQUV4Six5RUFBNkg7QUFzRjdIOzs7O0dBSUc7QUFDSCxNQUFhLG1CQUFvQixTQUFRLDRCQUFZO0lBdUJqRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQStCO1FBQ3JFLE1BQU0sQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLEdBQUcsK0JBQXdCLENBQUMsS0FBSyxDQUFDLFFBQVEsRUFBRSxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDOUUsSUFBSSxRQUFRLEtBQUssU0FBUyxJQUFJLElBQUksS0FBSyxTQUFTLEVBQUU7WUFDOUMsTUFBTSxJQUFJLEtBQUssQ0FBQyxzREFBc0QsQ0FBQyxDQUFDO1NBQzNFO1FBQ0QsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDYixlQUFlLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxlQUFlO1lBQ25ELFlBQVksRUFBRSxXQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsR0FBRyxDQUFDLGNBQWMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxFQUFFLGNBQWMsRUFBRSxDQUFDLENBQUMsRUFBRSxFQUFFLEVBQUUsY0FBYyxFQUFFLElBQUksRUFBRSxDQUFDO1lBQzFJLFFBQVE7WUFDUixJQUFJO1lBQ0osU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTO1NBQzdCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztRQUN2QyxJQUFJLENBQUMsUUFBUSxHQUFHLFFBQVEsQ0FBQztRQUN6QixJQUFJLENBQUMsZUFBZSxHQUFHLEVBQUUsQ0FBQztRQUMxQixzQkFBc0I7UUFDdEIsSUFBSSxLQUFLLENBQUMsZUFBZSxJQUFJLEtBQUssQ0FBQyxlQUFlLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUMzRCxJQUFJLENBQUMsa0JBQWtCLENBQUMscUJBQXFCLEVBQUUsS0FBSyxDQUFDLGVBQWUsQ0FBQyxDQUFDO1NBQ3pFO1FBQ0QsSUFBSSxLQUFLLENBQUMsWUFBWSxJQUFJLEtBQUssQ0FBQyxZQUFZLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNyRCxJQUFJLENBQUMsZUFBZSxDQUFDLHFCQUFxQixFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQztTQUNuRTtRQUNELDhEQUE4RDtRQUM5RCxpQ0FBaUM7UUFDakMsSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxXQUFXLENBQUM7WUFDbkMsY0FBYyxFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsV0FBVyxDQUFDLGNBQWM7WUFDN0QsV0FBVyxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQztTQUNsQyxDQUFDLENBQUM7UUFDSCxJQUFJLEtBQUssQ0FBQyxhQUFhLElBQUksS0FBSyxDQUFDLG1CQUFtQixFQUFFO1lBQ2xELE1BQU0sSUFBSSxLQUFLLENBQUMsc0VBQXNFLENBQUMsQ0FBQztTQUMzRjtRQUNELElBQUksS0FBSyxDQUFDLGFBQWEsRUFBRTtZQUNyQixJQUFJLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1NBQzlDO1FBQ0QsSUFBSSxLQUFLLENBQUMsbUJBQW1CLEVBQUU7WUFDM0IsSUFBSSxDQUFDLGdCQUFnQixDQUFDLDRDQUFjLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLENBQUM7U0FDNUU7UUFDRCxJQUFJLEtBQUssQ0FBQyxJQUFJLEtBQUssS0FBSyxFQUFFO1lBQ3RCLElBQUksQ0FBQyxXQUFXLENBQUMsb0JBQW9CLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsRUFBRSw2QkFBNkIsSUFBSSxFQUFFLENBQUMsQ0FBQztZQUMvRixJQUFJLElBQUksQ0FBQyxZQUFZLENBQUMsYUFBYSxLQUFLLHFCQUFhLENBQUMsVUFBVSxFQUFFO2dCQUM5RCxJQUFJLENBQUMsV0FBVyxDQUFDLG9CQUFvQixDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLEVBQUUsNkJBQTZCLElBQUksRUFBRSxDQUFDLENBQUM7YUFDbEc7U0FDSjtJQUNMLENBQUM7SUFqRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsaUNBQWlDLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBb0M7UUFDOUcsT0FBTyxJQUFJLDJCQUEyQixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDN0QsQ0FBQztJQTZERDs7Ozs7Ozs7T0FRRztJQUNJLGtCQUFrQixDQUFDLEVBQVUsRUFBRSxJQUFjO1FBQ2hELElBQUksQ0FBQyxlQUFlLENBQUMsRUFBRSxFQUFFLElBQUksQ0FBQyxHQUFHLENBQUMsMENBQW1CLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztJQUNwRSxDQUFDO0lBQ0Q7Ozs7OztPQU1HO0lBQ0ksZUFBZSxDQUFDLEVBQVUsRUFBRSxZQUFvQztRQUNuRSxNQUFNLGVBQWUsR0FBRyxDQUFDLEdBQUcsWUFBWSxDQUFDLENBQUM7UUFDMUMsSUFBSSxJQUFJLENBQUMsZUFBZSxDQUFDLE1BQU0sS0FBSyxDQUFDLElBQUksZUFBZSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDakUsTUFBTSxLQUFLLEdBQUcsZUFBZSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDOUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1NBQ25EO1FBQ0QsSUFBSSxlQUFlLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUM1QixJQUFJLGlFQUE4QixDQUFDLElBQUksRUFBRSxFQUFFLEVBQUU7Z0JBQ3pDLFFBQVEsRUFBRSxJQUFJO2dCQUNkLFlBQVksRUFBRSxlQUFlO2FBQ2hDLENBQUMsQ0FBQztTQUNOO0lBQ0wsQ0FBQztJQUNEOzs7Ozs7Ozs7O09BVUc7SUFDSSxTQUFTLENBQUMsRUFBVSxFQUFFLEtBQWdDO1FBQ3pELGlCQUFpQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ3pCLElBQUksS0FBSyxDQUFDLFFBQVEsS0FBSyxTQUFTLEVBQUU7WUFDOUIsV0FBVztZQUNYLEVBQUU7WUFDRix5RUFBeUU7WUFDekUsSUFBSSxtREFBdUIsQ0FBQyxJQUFJLEVBQUUsRUFBRSxHQUFHLE1BQU0sRUFBRTtnQkFDM0MsUUFBUSxFQUFFLElBQUk7Z0JBQ2QsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVO2dCQUM1QixVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7Z0JBQzVCLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztnQkFDOUIsWUFBWSxFQUFFLEtBQUssQ0FBQyxZQUFZO2dCQUNoQyxRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7Z0JBQ3hCLE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTTthQUN2QixDQUFDLENBQUM7U0FDTjthQUNJO1lBQ0QsNkNBQTZDO1lBQzdDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDdkM7SUFDTCxDQUFDO0lBQ0Q7Ozs7Ozs7OztPQVNHO0lBQ0ksZUFBZSxDQUFDLEVBQVUsRUFBRSxLQUFzQztRQUNyRSxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUN6QixJQUFJLEtBQUssQ0FBQyxRQUFRLEtBQUssU0FBUyxFQUFFO1lBQzlCLFdBQVc7WUFDWCxFQUFFO1lBQ0YseUVBQXlFO1lBQ3pFLElBQUksbURBQXVCLENBQUMsSUFBSSxFQUFFLEVBQUUsR0FBRyxNQUFNLEVBQUU7Z0JBQzNDLFFBQVEsRUFBRSxJQUFJO2dCQUNkLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtnQkFDNUIsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVO2dCQUM1QixXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7Z0JBQzlCLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWTtnQkFDaEMsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRO2dCQUN4QixZQUFZLEVBQUUsS0FBSyxDQUFDLFlBQVk7YUFDbkMsQ0FBQyxDQUFDO1NBQ047YUFDSTtZQUNELDZDQUE2QztZQUM3QyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsNENBQWMsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUM7U0FDckU7SUFDTCxDQUFDO0lBQ0Q7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLFVBQVUsQ0FBQyxFQUFVLEVBQUUsS0FBaUM7UUFDM0QsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsR0FBRyxFQUFFO1lBQ3hCLG1DQUFtQztZQUNuQyxNQUFNLElBQUksS0FBSyxDQUFDLHVLQUF1SyxDQUFDLENBQUM7U0FDNUw7UUFDRCxNQUFNLEtBQUssR0FBRyxJQUFJLGlEQUFzQixDQUFDLElBQUksRUFBRSxFQUFFLEdBQUcsT0FBTyxFQUFFO1lBQ3pELG1CQUFtQixFQUFFLEtBQUssQ0FBQyxtQkFBbUI7WUFDOUMsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO1lBQzlCLElBQUksRUFBRSxLQUFLLENBQUMsSUFBSTtZQUNoQixRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7WUFDeEIsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTO1lBQzFCLHdCQUF3QixFQUFFLEtBQUssQ0FBQyx3QkFBd0I7WUFDeEQsZUFBZSxFQUFFLEtBQUssQ0FBQyxlQUFlO1lBQ3RDLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTztZQUN0QixHQUFHLEVBQUUsSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHO1NBQzdCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxlQUFlLENBQUMsRUFBRSxFQUFFO1lBQ3JCLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtZQUM1QixVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7WUFDNUIsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO1lBQzlCLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWTtZQUNoQyxRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7WUFDeEIsWUFBWSxFQUFFLENBQUMsS0FBSyxDQUFDO1NBQ3hCLENBQUMsQ0FBQztRQUNILE9BQU8sS0FBSyxDQUFDO0lBQ2pCLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksZ0JBQWdCLENBQUMsRUFBVSxFQUFFLEtBQTRCO1FBQzVELGlCQUFpQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ3pCLE1BQU0sYUFBYSxHQUFrQjtZQUNqQyxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7WUFDNUIsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO1lBQzlCLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztTQUNqQyxDQUFDO1FBQ0YsaURBQXFCLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDckMsSUFBSSxLQUFLLENBQUMsUUFBUSxFQUFFO1lBQ2hCLElBQUksbURBQXVCLENBQUMsSUFBSSxFQUFFLEVBQUUsR0FBRyxNQUFNLEVBQUU7Z0JBQzNDLFFBQVEsRUFBRSxJQUFJO2dCQUNkLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTtnQkFDeEIsYUFBYTtnQkFDYixHQUFHLEtBQUs7YUFDWCxDQUFDLENBQUM7U0FDTjthQUNJO1lBQ0QsSUFBSSxDQUFDLGdCQUFnQixDQUFDLDRDQUFjLENBQUMsYUFBYSxDQUFDLFlBQUssQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxFQUFFO2dCQUNqRixXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7Z0JBQzlCLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVzthQUNqQyxDQUFDLENBQUMsQ0FBQztTQUNQO0lBQ0wsQ0FBQztJQUNEOzs7O09BSUc7SUFDSSxtQkFBbUIsQ0FBQyxFQUFVLEVBQUUsS0FBK0I7UUFDbEUsaUJBQWlCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDekIsTUFBTSxnQkFBZ0IsR0FBRztZQUNyQixJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUk7WUFDaEIsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJO1lBQ2hCLElBQUksRUFBRSxLQUFLLENBQUMsSUFBSTtZQUNoQixRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7WUFDeEIsS0FBSyxFQUFFLEtBQUssQ0FBQyxLQUFLO1lBQ2xCLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtTQUMvQixDQUFDO1FBQ0Ysb0RBQXdCLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUMzQyxJQUFJLEtBQUssQ0FBQyxRQUFRLEVBQUU7WUFDaEIsSUFBSSxtREFBdUIsQ0FBQyxJQUFJLEVBQUUsRUFBRSxHQUFHLE1BQU0sRUFBRTtnQkFDM0MsUUFBUSxFQUFFLElBQUk7Z0JBQ2QsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRO2dCQUN4QixnQkFBZ0I7Z0JBQ2hCLEdBQUcsS0FBSzthQUNYLENBQUMsQ0FBQztTQUNOO2FBQ0k7WUFDRCxJQUFJLENBQUMsZ0JBQWdCLENBQUMsNENBQWMsQ0FBQyxRQUFRLENBQUM7Z0JBQzFDLElBQUksRUFBRSxLQUFLLENBQUMsSUFBSTtnQkFDaEIsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJO2dCQUNoQixJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUk7Z0JBQ2hCLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTtnQkFDeEIsS0FBSyxFQUFFLEtBQUssQ0FBQyxLQUFLO2dCQUNsQixTQUFTLEVBQUUsS0FBSyxDQUFDLFVBQVUsS0FBSyxVQUFVO2FBQzdDLENBQUMsQ0FBQyxDQUFDO1NBQ1A7SUFDTCxDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNJLG1CQUFtQixDQUFDLFdBQTZCLEVBQUUsU0FBbUI7UUFDekUsV0FBVyxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLFlBQVksRUFBRSxTQUFTLEVBQUUseUJBQXlCLENBQUMsQ0FBQztJQUMvRixDQUFDO0lBQ0Q7O09BRUc7SUFDTyxRQUFRO1FBQ2QsTUFBTSxNQUFNLEdBQUcsS0FBSyxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBQ2hDLElBQUksSUFBSSxDQUFDLFFBQVEsS0FBSywyQkFBbUIsQ0FBQyxLQUFLLElBQUksSUFBSSxDQUFDLGVBQWUsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ2xGLE1BQU0sQ0FBQyxJQUFJLENBQUMsc0VBQXNFLENBQUMsQ0FBQztTQUN2RjtRQUNELE9BQU8sTUFBTSxDQUFDO0lBQ2xCLENBQUM7SUFDRDs7T0FFRztJQUNLLGdCQUFnQixDQUFDLE1BQXNCO1FBQzNDLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQ3hCLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUNuQyxDQUFDO0NBQ0o7QUFsU0Qsa0RBa1NDO0FBMEVELE1BQU0sMkJBQTRCLFNBQVEsZUFBUTtJQU05QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQW9DO1FBQzFFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDakIsSUFBSSxDQUFDLFdBQVcsR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDO1FBQ3JDLE1BQU0sV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUNsRyxJQUFJLGFBQWlDLENBQUM7UUFDdEMsSUFBSSxLQUFLLENBQUMsYUFBYSxFQUFFO1lBQ3JCLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDO1NBQ3ZDO2FBQ0ksSUFBSSxLQUFLLENBQUMsZUFBZSxFQUFFO1lBQzVCLGFBQWEsR0FBRyxHQUFHLENBQUMsYUFBYSxDQUFDLG1CQUFtQixDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUUsS0FBSyxDQUFDLGVBQWUsRUFBRTtnQkFDakcsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLDhCQUE4QjthQUN6RCxDQUFDLENBQUM7U0FDTjthQUNJO1lBQ0QsTUFBTSxJQUFJLEtBQUssQ0FBQyxrR0FBa0csQ0FBQyxDQUFDO1NBQ3ZIO1FBQ0QsSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxXQUFXLENBQUM7WUFDbkMsY0FBYyxFQUFFLENBQUMsYUFBYSxDQUFDO1lBQy9CLFdBQVc7U0FDZCxDQUFDLENBQUM7SUFDUCxDQUFDO0lBQ0Q7O09BRUc7SUFDSSxrQkFBa0IsQ0FBQyxFQUFVLEVBQUUsSUFBYztRQUNoRCxJQUFJLGlFQUE4QixDQUFDLElBQUksRUFBRSxFQUFFLEVBQUU7WUFDekMsUUFBUSxFQUFFLElBQUk7WUFDZCxlQUFlLEVBQUUsSUFBSTtTQUN4QixDQUFDLENBQUM7SUFDUCxDQUFDO0lBQ0Q7Ozs7O09BS0c7SUFDSSxlQUFlLENBQUMsRUFBVSxFQUFFLEtBQXNDO1FBQ3JFLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ3pCLElBQUksS0FBSyxDQUFDLFFBQVEsS0FBSyxTQUFTLEVBQUU7WUFDOUIsV0FBVztZQUNYLElBQUksbURBQXVCLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtnQkFDbEMsUUFBUSxFQUFFLElBQUk7Z0JBQ2QsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVO2dCQUM1QixXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7Z0JBQzlCLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWTtnQkFDaEMsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRO2dCQUN4QixZQUFZLEVBQUUsS0FBSyxDQUFDLFlBQVk7YUFDbkMsQ0FBQyxDQUFDO1NBQ047YUFDSTtZQUNELE1BQU0sSUFBSSxLQUFLLENBQUMsa0VBQWtFLENBQUMsQ0FBQztTQUN2RjtJQUNMLENBQUM7SUFDRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksVUFBVSxDQUFDLEdBQVcsRUFBRSxNQUFrQztRQUM3RCxtQ0FBbUM7UUFDbkMsTUFBTSxJQUFJLEtBQUssQ0FBQyw4SEFBOEgsQ0FBQyxDQUFDO0lBQ3BKLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksbUJBQW1CLENBQUMsV0FBNkIsRUFBRSxTQUFtQjtRQUN6RSxJQUFJLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsU0FBUyxFQUFFLHlCQUF5QixDQUFDLENBQUM7SUFDaEYsQ0FBQztDQUNKO0FBbUtELFNBQVMsaUJBQWlCLENBQUMsS0FBbUI7O0lBQzFDLE1BQU0sZUFBZSxHQUFHLE9BQUEsS0FBSyxDQUFDLFVBQVUsMENBQUUsTUFBTSxLQUFJLENBQUMsQ0FBQztJQUN0RCxNQUFNLGdCQUFnQixHQUFHLGVBQWUsS0FBSyxDQUFDO1FBQzFDLEtBQUssQ0FBQyxVQUFVLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxXQUFXLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxZQUFZLEtBQUssU0FBUyxDQUFDO0lBQzFHLE1BQU0sV0FBVyxHQUFHLEtBQUssQ0FBQyxRQUFRLEtBQUssU0FBUyxDQUFDO0lBQ2pELElBQUksZ0JBQWdCLEtBQUssV0FBVyxFQUFFO1FBQ2xDLE1BQU0sSUFBSSxLQUFLLENBQUMsc0dBQXNHLENBQUMsQ0FBQztLQUMzSDtBQUNMLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBlYzIgZnJvbSBcIi4uLy4uLy4uL2F3cy1lYzJcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInXG5pbXBvcnQgeyBDb25zdHJ1Y3QsIER1cmF0aW9uLCBJUmVzb3VyY2UsIExhenksIFJlc291cmNlLCBUb2tlbiB9IGZyb20gXCIuLi8uLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0IHsgQmFzZUxpc3RlbmVyIH0gZnJvbSAnLi4vc2hhcmVkL2Jhc2UtbGlzdGVuZXInO1xuaW1wb3J0IHsgSGVhbHRoQ2hlY2sgfSBmcm9tICcuLi9zaGFyZWQvYmFzZS10YXJnZXQtZ3JvdXAnO1xuaW1wb3J0IHsgQXBwbGljYXRpb25Qcm90b2NvbCwgSXBBZGRyZXNzVHlwZSwgU3NsUG9saWN5IH0gZnJvbSAnLi4vc2hhcmVkL2VudW1zJztcbmltcG9ydCB7IElMaXN0ZW5lckNlcnRpZmljYXRlLCBMaXN0ZW5lckNlcnRpZmljYXRlIH0gZnJvbSAnLi4vc2hhcmVkL2xpc3RlbmVyLWNlcnRpZmljYXRlJztcbmltcG9ydCB7IGRldGVybWluZVByb3RvY29sQW5kUG9ydCB9IGZyb20gJy4uL3NoYXJlZC91dGlsJztcbmltcG9ydCB7IExpc3RlbmVyQWN0aW9uIH0gZnJvbSAnLi9hcHBsaWNhdGlvbi1saXN0ZW5lci1hY3Rpb24nO1xuaW1wb3J0IHsgQXBwbGljYXRpb25MaXN0ZW5lckNlcnRpZmljYXRlIH0gZnJvbSAnLi9hcHBsaWNhdGlvbi1saXN0ZW5lci1jZXJ0aWZpY2F0ZSc7XG5pbXBvcnQgeyBBcHBsaWNhdGlvbkxpc3RlbmVyUnVsZSwgRml4ZWRSZXNwb25zZSwgUmVkaXJlY3RSZXNwb25zZSwgdmFsaWRhdGVGaXhlZFJlc3BvbnNlLCB2YWxpZGF0ZVJlZGlyZWN0UmVzcG9uc2UgfSBmcm9tICcuL2FwcGxpY2F0aW9uLWxpc3RlbmVyLXJ1bGUnO1xuaW1wb3J0IHsgSUFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyIH0gZnJvbSAnLi9hcHBsaWNhdGlvbi1sb2FkLWJhbGFuY2VyJztcbmltcG9ydCB7IEFwcGxpY2F0aW9uVGFyZ2V0R3JvdXAsIElBcHBsaWNhdGlvbkxvYWRCYWxhbmNlclRhcmdldCwgSUFwcGxpY2F0aW9uVGFyZ2V0R3JvdXAgfSBmcm9tICcuL2FwcGxpY2F0aW9uLXRhcmdldC1ncm91cCc7XG5pbXBvcnQgeyBMaXN0ZW5lckNvbmRpdGlvbiB9IGZyb20gJy4vY29uZGl0aW9ucyc7XG4vKipcbiAqIEJhc2ljIHByb3BlcnRpZXMgZm9yIGFuIEFwcGxpY2F0aW9uTGlzdGVuZXJcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBCYXNlQXBwbGljYXRpb25MaXN0ZW5lclByb3BzIHtcbiAgICAvKipcbiAgICAgKiBUaGUgcHJvdG9jb2wgdG8gdXNlXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIERldGVybWluZWQgZnJvbSBwb3J0IGlmIGtub3duLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHByb3RvY29sPzogQXBwbGljYXRpb25Qcm90b2NvbDtcbiAgICAvKipcbiAgICAgKiBUaGUgcG9ydCBvbiB3aGljaCB0aGUgbGlzdGVuZXIgbGlzdGVucyBmb3IgcmVxdWVzdHMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIERldGVybWluZWQgZnJvbSBwcm90b2NvbCBpZiBrbm93bi5cbiAgICAgKi9cbiAgICByZWFkb25seSBwb3J0PzogbnVtYmVyO1xuICAgIC8qKlxuICAgICAqIFRoZSBjZXJ0aWZpY2F0ZXMgdG8gdXNlIG9uIHRoaXMgbGlzdGVuZXJcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gY2VydGlmaWNhdGVzLlxuICAgICAqIEBkZXByZWNhdGVkIFVzZSB0aGUgYGNlcnRpZmljYXRlc2AgcHJvcGVydHkgaW5zdGVhZFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNlcnRpZmljYXRlQXJucz86IHN0cmluZ1tdO1xuICAgIC8qKlxuICAgICAqIENlcnRpZmljYXRlIGxpc3Qgb2YgQUNNIGNlcnQgQVJOc1xuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBjZXJ0aWZpY2F0ZXMuXG4gICAgICovXG4gICAgcmVhZG9ubHkgY2VydGlmaWNhdGVzPzogSUxpc3RlbmVyQ2VydGlmaWNhdGVbXTtcbiAgICAvKipcbiAgICAgKiBUaGUgc2VjdXJpdHkgcG9saWN5IHRoYXQgZGVmaW5lcyB3aGljaCBjaXBoZXJzIGFuZCBwcm90b2NvbHMgYXJlIHN1cHBvcnRlZC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gVGhlIGN1cnJlbnQgcHJlZGVmaW5lZCBzZWN1cml0eSBwb2xpY3kuXG4gICAgICovXG4gICAgcmVhZG9ubHkgc3NsUG9saWN5PzogU3NsUG9saWN5O1xuICAgIC8qKlxuICAgICAqIERlZmF1bHQgdGFyZ2V0IGdyb3VwcyB0byBsb2FkIGJhbGFuY2UgdG9cbiAgICAgKlxuICAgICAqIEFsbCB0YXJnZXQgZ3JvdXBzIHdpbGwgYmUgbG9hZCBiYWxhbmNlZCB0byB3aXRoIGVxdWFsIHdlaWdodCBhbmQgd2l0aG91dFxuICAgICAqIHN0aWNraW5lc3MuIEZvciBhIG1vcmUgY29tcGxleCBjb25maWd1cmF0aW9uIHRoYW4gdGhhdCwgdXNlXG4gICAgICogZWl0aGVyIGBkZWZhdWx0QWN0aW9uYCBvciBgYWRkQWN0aW9uKClgLlxuICAgICAqXG4gICAgICogQ2Fubm90IGJlIHNwZWNpZmllZCB0b2dldGhlciB3aXRoIGBkZWZhdWx0QWN0aW9uYC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm9uZS5cbiAgICAgKi9cbiAgICByZWFkb25seSBkZWZhdWx0VGFyZ2V0R3JvdXBzPzogSUFwcGxpY2F0aW9uVGFyZ2V0R3JvdXBbXTtcbiAgICAvKipcbiAgICAgKiBEZWZhdWx0IGFjdGlvbiB0byB0YWtlIGZvciByZXF1ZXN0cyB0byB0aGlzIGxpc3RlbmVyXG4gICAgICpcbiAgICAgKiBUaGlzIGFsbG93cyBmdWxsIGNvbnRyb2wgb2YgdGhlIGRlZmF1bHQgYWN0aW9uIG9mIHRoZSBsb2FkIGJhbGFuY2VyLFxuICAgICAqIGluY2x1ZGluZyBBY3Rpb24gY2hhaW5pbmcsIGZpeGVkIHJlc3BvbnNlcyBhbmQgcmVkaXJlY3QgcmVzcG9uc2VzLlxuICAgICAqXG4gICAgICogU2VlIHRoZSBgTGlzdGVuZXJBY3Rpb25gIGNsYXNzIGZvciBhbGwgb3B0aW9ucy5cbiAgICAgKlxuICAgICAqIENhbm5vdCBiZSBzcGVjaWZpZWQgdG9nZXRoZXIgd2l0aCBgZGVmYXVsdFRhcmdldEdyb3Vwc2AuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmUuXG4gICAgICovXG4gICAgcmVhZG9ubHkgZGVmYXVsdEFjdGlvbj86IExpc3RlbmVyQWN0aW9uO1xuICAgIC8qKlxuICAgICAqIEFsbG93IGFueW9uZSB0byBjb25uZWN0IHRvIHRoaXMgbGlzdGVuZXJcbiAgICAgKlxuICAgICAqIElmIHRoaXMgaXMgc3BlY2lmaWVkLCB0aGUgbGlzdGVuZXIgd2lsbCBiZSBvcGVuZWQgdXAgdG8gYW55b25lIHdobyBjYW4gcmVhY2ggaXQuXG4gICAgICogRm9yIGludGVybmFsIGxvYWQgYmFsYW5jZXJzIHRoaXMgaXMgYW55b25lIGluIHRoZSBzYW1lIFZQQy4gRm9yIHB1YmxpYyBsb2FkXG4gICAgICogYmFsYW5jZXJzLCB0aGlzIGlzIGFueW9uZSBvbiB0aGUgaW50ZXJuZXQuXG4gICAgICpcbiAgICAgKiBJZiB5b3Ugd2FudCB0byBiZSBtb3JlIHNlbGVjdGl2ZSBhYm91dCB3aG8gY2FuIGFjY2VzcyB0aGlzIGxvYWRcbiAgICAgKiBiYWxhbmNlciwgc2V0IHRoaXMgdG8gYGZhbHNlYCBhbmQgdXNlIHRoZSBsaXN0ZW5lcidzIGBjb25uZWN0aW9uc2BcbiAgICAgKiBvYmplY3QgdG8gc2VsZWN0aXZlbHkgZ3JhbnQgYWNjZXNzIHRvIHRoZSBsaXN0ZW5lci5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IHRydWVcbiAgICAgKi9cbiAgICByZWFkb25seSBvcGVuPzogYm9vbGVhbjtcbn1cbi8qKlxuICogUHJvcGVydGllcyBmb3IgZGVmaW5pbmcgYSBzdGFuZGFsb25lIEFwcGxpY2F0aW9uTGlzdGVuZXJcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBcHBsaWNhdGlvbkxpc3RlbmVyUHJvcHMgZXh0ZW5kcyBCYXNlQXBwbGljYXRpb25MaXN0ZW5lclByb3BzIHtcbiAgICAvKipcbiAgICAgKiBUaGUgbG9hZCBiYWxhbmNlciB0byBhdHRhY2ggdGhpcyBsaXN0ZW5lciB0b1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGxvYWRCYWxhbmNlcjogSUFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyO1xufVxuLyoqXG4gKiBEZWZpbmUgYW4gQXBwbGljYXRpb25MaXN0ZW5lclxuICpcbiAqIEByZXNvdXJjZSBBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6Okxpc3RlbmVyXG4gKi9cbmV4cG9ydCBjbGFzcyBBcHBsaWNhdGlvbkxpc3RlbmVyIGV4dGVuZHMgQmFzZUxpc3RlbmVyIGltcGxlbWVudHMgSUFwcGxpY2F0aW9uTGlzdGVuZXIge1xuICAgIC8qKlxuICAgICAqIEltcG9ydCBhbiBleGlzdGluZyBsaXN0ZW5lclxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZnJvbUFwcGxpY2F0aW9uTGlzdGVuZXJBdHRyaWJ1dGVzKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGF0dHJzOiBBcHBsaWNhdGlvbkxpc3RlbmVyQXR0cmlidXRlcyk6IElBcHBsaWNhdGlvbkxpc3RlbmVyIHtcbiAgICAgICAgcmV0dXJuIG5ldyBJbXBvcnRlZEFwcGxpY2F0aW9uTGlzdGVuZXIoc2NvcGUsIGlkLCBhdHRycyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIE1hbmFnZSBjb25uZWN0aW9ucyB0byB0aGlzIEFwcGxpY2F0aW9uTGlzdGVuZXJcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgY29ubmVjdGlvbnM6IGVjMi5Db25uZWN0aW9ucztcbiAgICAvKipcbiAgICAgKiBMb2FkIGJhbGFuY2VyIHRoaXMgbGlzdGVuZXIgaXMgYXNzb2NpYXRlZCB3aXRoXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGxvYWRCYWxhbmNlcjogSUFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyO1xuICAgIC8qKlxuICAgICAqIEFSTnMgb2YgY2VydGlmaWNhdGVzIGFkZGVkIHRvIHRoaXMgbGlzdGVuZXJcbiAgICAgKi9cbiAgICBwcml2YXRlIHJlYWRvbmx5IGNlcnRpZmljYXRlQXJuczogc3RyaW5nW107XG4gICAgLyoqXG4gICAgICogTGlzdGVuZXIgcHJvdG9jb2wgZm9yIHRoaXMgbGlzdGVuZXIuXG4gICAgICovXG4gICAgcHJpdmF0ZSByZWFkb25seSBwcm90b2NvbDogQXBwbGljYXRpb25Qcm90b2NvbDtcbiAgICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQXBwbGljYXRpb25MaXN0ZW5lclByb3BzKSB7XG4gICAgICAgIGNvbnN0IFtwcm90b2NvbCwgcG9ydF0gPSBkZXRlcm1pbmVQcm90b2NvbEFuZFBvcnQocHJvcHMucHJvdG9jb2wsIHByb3BzLnBvcnQpO1xuICAgICAgICBpZiAocHJvdG9jb2wgPT09IHVuZGVmaW5lZCB8fCBwb3J0ID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQXQgbGVhc3Qgb25lIG9mIFxcJ3BvcnRcXCcgb3IgXFwncHJvdG9jb2xcXCcgaXMgcmVxdWlyZWQnKTtcbiAgICAgICAgfVxuICAgICAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgICAgICAgIGxvYWRCYWxhbmNlckFybjogcHJvcHMubG9hZEJhbGFuY2VyLmxvYWRCYWxhbmNlckFybixcbiAgICAgICAgICAgIGNlcnRpZmljYXRlczogTGF6eS5hbnlWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMuY2VydGlmaWNhdGVBcm5zLm1hcChjZXJ0aWZpY2F0ZUFybiA9PiAoeyBjZXJ0aWZpY2F0ZUFybiB9KSkgfSwgeyBvbWl0RW1wdHlBcnJheTogdHJ1ZSB9KSxcbiAgICAgICAgICAgIHByb3RvY29sLFxuICAgICAgICAgICAgcG9ydCxcbiAgICAgICAgICAgIHNzbFBvbGljeTogcHJvcHMuc3NsUG9saWN5LFxuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5sb2FkQmFsYW5jZXIgPSBwcm9wcy5sb2FkQmFsYW5jZXI7XG4gICAgICAgIHRoaXMucHJvdG9jb2wgPSBwcm90b2NvbDtcbiAgICAgICAgdGhpcy5jZXJ0aWZpY2F0ZUFybnMgPSBbXTtcbiAgICAgICAgLy8gQXR0YWNoIGNlcnRpZmljYXRlc1xuICAgICAgICBpZiAocHJvcHMuY2VydGlmaWNhdGVBcm5zICYmIHByb3BzLmNlcnRpZmljYXRlQXJucy5sZW5ndGggPiAwKSB7XG4gICAgICAgICAgICB0aGlzLmFkZENlcnRpZmljYXRlQXJucygnTGlzdGVuZXJDZXJ0aWZpY2F0ZScsIHByb3BzLmNlcnRpZmljYXRlQXJucyk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHByb3BzLmNlcnRpZmljYXRlcyAmJiBwcm9wcy5jZXJ0aWZpY2F0ZXMubGVuZ3RoID4gMCkge1xuICAgICAgICAgICAgdGhpcy5hZGRDZXJ0aWZpY2F0ZXMoJ0RlZmF1bHRDZXJ0aWZpY2F0ZXMnLCBwcm9wcy5jZXJ0aWZpY2F0ZXMpO1xuICAgICAgICB9XG4gICAgICAgIC8vIFRoaXMgbGlzdGVuZXIgZWRpdHMgdGhlIHNlY3VyaXR5Z3JvdXAgb2YgdGhlIGxvYWQgYmFsYW5jZXIsXG4gICAgICAgIC8vIGJ1dCBhZGRzIGl0cyBvd24gZGVmYXVsdCBwb3J0LlxuICAgICAgICB0aGlzLmNvbm5lY3Rpb25zID0gbmV3IGVjMi5Db25uZWN0aW9ucyh7XG4gICAgICAgICAgICBzZWN1cml0eUdyb3VwczogcHJvcHMubG9hZEJhbGFuY2VyLmNvbm5lY3Rpb25zLnNlY3VyaXR5R3JvdXBzLFxuICAgICAgICAgICAgZGVmYXVsdFBvcnQ6IGVjMi5Qb3J0LnRjcChwb3J0KSxcbiAgICAgICAgfSk7XG4gICAgICAgIGlmIChwcm9wcy5kZWZhdWx0QWN0aW9uICYmIHByb3BzLmRlZmF1bHRUYXJnZXRHcm91cHMpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignU3BlY2lmeSBhdCBtb3N0IG9uZSBvZiBcXCdkZWZhdWx0QWN0aW9uXFwnIGFuZCBcXCdkZWZhdWx0VGFyZ2V0R3JvdXBzXFwnJyk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHByb3BzLmRlZmF1bHRBY3Rpb24pIHtcbiAgICAgICAgICAgIHRoaXMuc2V0RGVmYXVsdEFjdGlvbihwcm9wcy5kZWZhdWx0QWN0aW9uKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAocHJvcHMuZGVmYXVsdFRhcmdldEdyb3Vwcykge1xuICAgICAgICAgICAgdGhpcy5zZXREZWZhdWx0QWN0aW9uKExpc3RlbmVyQWN0aW9uLmZvcndhcmQocHJvcHMuZGVmYXVsdFRhcmdldEdyb3VwcykpO1xuICAgICAgICB9XG4gICAgICAgIGlmIChwcm9wcy5vcGVuICE9PSBmYWxzZSkge1xuICAgICAgICAgICAgdGhpcy5jb25uZWN0aW9ucy5hbGxvd0RlZmF1bHRQb3J0RnJvbShlYzIuUGVlci5hbnlJcHY0KCksIGBBbGxvdyBmcm9tIGFueW9uZSBvbiBwb3J0ICR7cG9ydH1gKTtcbiAgICAgICAgICAgIGlmICh0aGlzLmxvYWRCYWxhbmNlci5pcEFkZHJlc3NUeXBlID09PSBJcEFkZHJlc3NUeXBlLkRVQUxfU1RBQ0spIHtcbiAgICAgICAgICAgICAgICB0aGlzLmNvbm5lY3Rpb25zLmFsbG93RGVmYXVsdFBvcnRGcm9tKGVjMi5QZWVyLmFueUlwdjYoKSwgYEFsbG93IGZyb20gYW55b25lIG9uIHBvcnQgJHtwb3J0fWApO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZCBvbmUgb3IgbW9yZSBjZXJ0aWZpY2F0ZXMgdG8gdGhpcyBsaXN0ZW5lci5cbiAgICAgKlxuICAgICAqIEFmdGVyIHRoZSBmaXJzdCBjZXJ0aWZpY2F0ZSwgdGhpcyBjcmVhdGVzIEFwcGxpY2F0aW9uTGlzdGVuZXJDZXJ0aWZpY2F0ZXNcbiAgICAgKiByZXNvdXJjZXMgc2luY2UgY2xvdWRmb3JtYXRpb24gcmVxdWlyZXMgdGhlIGNlcnRpZmljYXRlcyBhcnJheSBvbiB0aGVcbiAgICAgKiBsaXN0ZW5lciByZXNvdXJjZSB0byBoYXZlIGEgbGVuZ3RoIG9mIDEuXG4gICAgICpcbiAgICAgKiBAZGVwcmVjYXRlZCBVc2UgYGFkZENlcnRpZmljYXRlc2AgaW5zdGVhZC5cbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkQ2VydGlmaWNhdGVBcm5zKGlkOiBzdHJpbmcsIGFybnM6IHN0cmluZ1tdKTogdm9pZCB7XG4gICAgICAgIHRoaXMuYWRkQ2VydGlmaWNhdGVzKGlkLCBhcm5zLm1hcChMaXN0ZW5lckNlcnRpZmljYXRlLmZyb21Bcm4pKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQWRkIG9uZSBvciBtb3JlIGNlcnRpZmljYXRlcyB0byB0aGlzIGxpc3RlbmVyLlxuICAgICAqXG4gICAgICogQWZ0ZXIgdGhlIGZpcnN0IGNlcnRpZmljYXRlLCB0aGlzIGNyZWF0ZXMgQXBwbGljYXRpb25MaXN0ZW5lckNlcnRpZmljYXRlc1xuICAgICAqIHJlc291cmNlcyBzaW5jZSBjbG91ZGZvcm1hdGlvbiByZXF1aXJlcyB0aGUgY2VydGlmaWNhdGVzIGFycmF5IG9uIHRoZVxuICAgICAqIGxpc3RlbmVyIHJlc291cmNlIHRvIGhhdmUgYSBsZW5ndGggb2YgMS5cbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkQ2VydGlmaWNhdGVzKGlkOiBzdHJpbmcsIGNlcnRpZmljYXRlczogSUxpc3RlbmVyQ2VydGlmaWNhdGVbXSk6IHZvaWQge1xuICAgICAgICBjb25zdCBhZGRpdGlvbmFsQ2VydHMgPSBbLi4uY2VydGlmaWNhdGVzXTtcbiAgICAgICAgaWYgKHRoaXMuY2VydGlmaWNhdGVBcm5zLmxlbmd0aCA9PT0gMCAmJiBhZGRpdGlvbmFsQ2VydHMubGVuZ3RoID4gMCkge1xuICAgICAgICAgICAgY29uc3QgZmlyc3QgPSBhZGRpdGlvbmFsQ2VydHMuc3BsaWNlKDAsIDEpWzBdO1xuICAgICAgICAgICAgdGhpcy5jZXJ0aWZpY2F0ZUFybnMucHVzaChmaXJzdC5jZXJ0aWZpY2F0ZUFybik7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKGFkZGl0aW9uYWxDZXJ0cy5sZW5ndGggPiAwKSB7XG4gICAgICAgICAgICBuZXcgQXBwbGljYXRpb25MaXN0ZW5lckNlcnRpZmljYXRlKHRoaXMsIGlkLCB7XG4gICAgICAgICAgICAgICAgbGlzdGVuZXI6IHRoaXMsXG4gICAgICAgICAgICAgICAgY2VydGlmaWNhdGVzOiBhZGRpdGlvbmFsQ2VydHMsXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgIH1cbiAgICAvKipcbiAgICAgKiBQZXJmb3JtIHRoZSBnaXZlbiBkZWZhdWx0IGFjdGlvbiBvbiBpbmNvbWluZyByZXF1ZXN0c1xuICAgICAqXG4gICAgICogVGhpcyBhbGxvd3MgZnVsbCBjb250cm9sIG9mIHRoZSBkZWZhdWx0IGFjdGlvbiBvZiB0aGUgbG9hZCBiYWxhbmNlcixcbiAgICAgKiBpbmNsdWRpbmcgQWN0aW9uIGNoYWluaW5nLCBmaXhlZCByZXNwb25zZXMgYW5kIHJlZGlyZWN0IHJlc3BvbnNlcy4gU2VlXG4gICAgICogdGhlIGBMaXN0ZW5lckFjdGlvbmAgY2xhc3MgZm9yIGFsbCBvcHRpb25zLlxuICAgICAqXG4gICAgICogSXQncyBwb3NzaWJsZSB0byBhZGQgcm91dGluZyBjb25kaXRpb25zIHRvIHRoZSBBY3Rpb24gYWRkZWQgaW4gdGhpcyB3YXkuXG4gICAgICogQXQgbGVhc3Qgb25lIEFjdGlvbiBtdXN0IGJlIGFkZGVkIHdpdGhvdXQgY29uZGl0aW9ucyAod2hpY2ggYmVjb21lcyB0aGVcbiAgICAgKiBkZWZhdWx0IEFjdGlvbikuXG4gICAgICovXG4gICAgcHVibGljIGFkZEFjdGlvbihpZDogc3RyaW5nLCBwcm9wczogQWRkQXBwbGljYXRpb25BY3Rpb25Qcm9wcyk6IHZvaWQge1xuICAgICAgICBjaGVja0FkZFJ1bGVQcm9wcyhwcm9wcyk7XG4gICAgICAgIGlmIChwcm9wcy5wcmlvcml0eSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgICAvLyBOZXcgcnVsZVxuICAgICAgICAgICAgLy9cbiAgICAgICAgICAgIC8vIFRhcmdldEdyb3VwLnJlZ2lzdGVyTGlzdGVuZXIgaXMgY2FsbGVkIGluc2lkZSBBcHBsaWNhdGlvbkxpc3RlbmVyUnVsZS5cbiAgICAgICAgICAgIG5ldyBBcHBsaWNhdGlvbkxpc3RlbmVyUnVsZSh0aGlzLCBpZCArICdSdWxlJywge1xuICAgICAgICAgICAgICAgIGxpc3RlbmVyOiB0aGlzLFxuICAgICAgICAgICAgICAgIGNvbmRpdGlvbnM6IHByb3BzLmNvbmRpdGlvbnMsXG4gICAgICAgICAgICAgICAgaG9zdEhlYWRlcjogcHJvcHMuaG9zdEhlYWRlcixcbiAgICAgICAgICAgICAgICBwYXRoUGF0dGVybjogcHJvcHMucGF0aFBhdHRlcm4sXG4gICAgICAgICAgICAgICAgcGF0aFBhdHRlcm5zOiBwcm9wcy5wYXRoUGF0dGVybnMsXG4gICAgICAgICAgICAgICAgcHJpb3JpdHk6IHByb3BzLnByaW9yaXR5LFxuICAgICAgICAgICAgICAgIGFjdGlvbjogcHJvcHMuYWN0aW9uLFxuICAgICAgICAgICAgfSk7XG4gICAgICAgIH1cbiAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICAvLyBOZXcgZGVmYXVsdCB0YXJnZXQgd2l0aCB0aGVzZSB0YXJnZXRncm91cHNcbiAgICAgICAgICAgIHRoaXMuc2V0RGVmYXVsdEFjdGlvbihwcm9wcy5hY3Rpb24pO1xuICAgICAgICB9XG4gICAgfVxuICAgIC8qKlxuICAgICAqIExvYWQgYmFsYW5jZSBpbmNvbWluZyByZXF1ZXN0cyB0byB0aGUgZ2l2ZW4gdGFyZ2V0IGdyb3Vwcy5cbiAgICAgKlxuICAgICAqIEFsbCB0YXJnZXQgZ3JvdXBzIHdpbGwgYmUgbG9hZCBiYWxhbmNlZCB0byB3aXRoIGVxdWFsIHdlaWdodCBhbmQgd2l0aG91dFxuICAgICAqIHN0aWNraW5lc3MuIEZvciBhIG1vcmUgY29tcGxleCBjb25maWd1cmF0aW9uIHRoYW4gdGhhdCwgdXNlIGBhZGRBY3Rpb24oKWAuXG4gICAgICpcbiAgICAgKiBJdCdzIHBvc3NpYmxlIHRvIGFkZCByb3V0aW5nIGNvbmRpdGlvbnMgdG8gdGhlIFRhcmdldEdyb3VwcyBhZGRlZCBpbiB0aGlzXG4gICAgICogd2F5LiBBdCBsZWFzdCBvbmUgVGFyZ2V0R3JvdXAgbXVzdCBiZSBhZGRlZCB3aXRob3V0IGNvbmRpdGlvbnMgKHdoaWNoIHdpbGxcbiAgICAgKiBiZWNvbWUgdGhlIGRlZmF1bHQgQWN0aW9uIGZvciB0aGlzIGxpc3RlbmVyKS5cbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkVGFyZ2V0R3JvdXBzKGlkOiBzdHJpbmcsIHByb3BzOiBBZGRBcHBsaWNhdGlvblRhcmdldEdyb3Vwc1Byb3BzKTogdm9pZCB7XG4gICAgICAgIGNoZWNrQWRkUnVsZVByb3BzKHByb3BzKTtcbiAgICAgICAgaWYgKHByb3BzLnByaW9yaXR5ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAgIC8vIE5ldyBydWxlXG4gICAgICAgICAgICAvL1xuICAgICAgICAgICAgLy8gVGFyZ2V0R3JvdXAucmVnaXN0ZXJMaXN0ZW5lciBpcyBjYWxsZWQgaW5zaWRlIEFwcGxpY2F0aW9uTGlzdGVuZXJSdWxlLlxuICAgICAgICAgICAgbmV3IEFwcGxpY2F0aW9uTGlzdGVuZXJSdWxlKHRoaXMsIGlkICsgJ1J1bGUnLCB7XG4gICAgICAgICAgICAgICAgbGlzdGVuZXI6IHRoaXMsXG4gICAgICAgICAgICAgICAgY29uZGl0aW9uczogcHJvcHMuY29uZGl0aW9ucyxcbiAgICAgICAgICAgICAgICBob3N0SGVhZGVyOiBwcm9wcy5ob3N0SGVhZGVyLFxuICAgICAgICAgICAgICAgIHBhdGhQYXR0ZXJuOiBwcm9wcy5wYXRoUGF0dGVybixcbiAgICAgICAgICAgICAgICBwYXRoUGF0dGVybnM6IHByb3BzLnBhdGhQYXR0ZXJucyxcbiAgICAgICAgICAgICAgICBwcmlvcml0eTogcHJvcHMucHJpb3JpdHksXG4gICAgICAgICAgICAgICAgdGFyZ2V0R3JvdXBzOiBwcm9wcy50YXJnZXRHcm91cHMsXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgIC8vIE5ldyBkZWZhdWx0IHRhcmdldCB3aXRoIHRoZXNlIHRhcmdldGdyb3Vwc1xuICAgICAgICAgICAgdGhpcy5zZXREZWZhdWx0QWN0aW9uKExpc3RlbmVyQWN0aW9uLmZvcndhcmQocHJvcHMudGFyZ2V0R3JvdXBzKSk7XG4gICAgICAgIH1cbiAgICB9XG4gICAgLyoqXG4gICAgICogTG9hZCBiYWxhbmNlIGluY29taW5nIHJlcXVlc3RzIHRvIHRoZSBnaXZlbiBsb2FkIGJhbGFuY2luZyB0YXJnZXRzLlxuICAgICAqXG4gICAgICogVGhpcyBtZXRob2QgaW1wbGljaXRseSBjcmVhdGVzIGFuIEFwcGxpY2F0aW9uVGFyZ2V0R3JvdXAgZm9yIHRoZSB0YXJnZXRzXG4gICAgICogaW52b2x2ZWQsIGFuZCBhICdmb3J3YXJkJyBhY3Rpb24gdG8gcm91dGUgdHJhZmZpYyB0byB0aGUgZ2l2ZW4gVGFyZ2V0R3JvdXAuXG4gICAgICpcbiAgICAgKiBJZiB5b3Ugd2FudCBtb3JlIGNvbnRyb2wgb3ZlciB0aGUgcHJlY2lzZSBzZXR1cCwgY3JlYXRlIHRoZSBUYXJnZXRHcm91cFxuICAgICAqIGFuZCB1c2UgYGFkZEFjdGlvbmAgeW91cnNlbGYuXG4gICAgICpcbiAgICAgKiBJdCdzIHBvc3NpYmxlIHRvIGFkZCBjb25kaXRpb25zIHRvIHRoZSB0YXJnZXRzIGFkZGVkIGluIHRoaXMgd2F5LiBBdCBsZWFzdFxuICAgICAqIG9uZSBzZXQgb2YgdGFyZ2V0cyBtdXN0IGJlIGFkZGVkIHdpdGhvdXQgY29uZGl0aW9ucy5cbiAgICAgKlxuICAgICAqIEByZXR1cm5zIFRoZSBuZXdseSBjcmVhdGVkIHRhcmdldCBncm91cFxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRUYXJnZXRzKGlkOiBzdHJpbmcsIHByb3BzOiBBZGRBcHBsaWNhdGlvblRhcmdldHNQcm9wcyk6IEFwcGxpY2F0aW9uVGFyZ2V0R3JvdXAge1xuICAgICAgICBpZiAoIXRoaXMubG9hZEJhbGFuY2VyLnZwYykge1xuICAgICAgICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG1heC1sZW5cbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQ2FuIG9ubHkgY2FsbCBhZGRUYXJnZXRzKCkgd2hlbiB1c2luZyBhIGNvbnN0cnVjdGVkIExvYWQgQmFsYW5jZXIgb3IgYW4gaW1wb3J0ZWQgTG9hZCBCYWxhbmNlciB3aXRoIHNwZWNpZmllZCB2cGM7IGNvbnN0cnVjdCBhIG5ldyBUYXJnZXRHcm91cCBhbmQgdXNlIGFkZFRhcmdldEdyb3VwJyk7XG4gICAgICAgIH1cbiAgICAgICAgY29uc3QgZ3JvdXAgPSBuZXcgQXBwbGljYXRpb25UYXJnZXRHcm91cCh0aGlzLCBpZCArICdHcm91cCcsIHtcbiAgICAgICAgICAgIGRlcmVnaXN0cmF0aW9uRGVsYXk6IHByb3BzLmRlcmVnaXN0cmF0aW9uRGVsYXksXG4gICAgICAgICAgICBoZWFsdGhDaGVjazogcHJvcHMuaGVhbHRoQ2hlY2ssXG4gICAgICAgICAgICBwb3J0OiBwcm9wcy5wb3J0LFxuICAgICAgICAgICAgcHJvdG9jb2w6IHByb3BzLnByb3RvY29sLFxuICAgICAgICAgICAgc2xvd1N0YXJ0OiBwcm9wcy5zbG93U3RhcnQsXG4gICAgICAgICAgICBzdGlja2luZXNzQ29va2llRHVyYXRpb246IHByb3BzLnN0aWNraW5lc3NDb29raWVEdXJhdGlvbixcbiAgICAgICAgICAgIHRhcmdldEdyb3VwTmFtZTogcHJvcHMudGFyZ2V0R3JvdXBOYW1lLFxuICAgICAgICAgICAgdGFyZ2V0czogcHJvcHMudGFyZ2V0cyxcbiAgICAgICAgICAgIHZwYzogdGhpcy5sb2FkQmFsYW5jZXIudnBjLFxuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5hZGRUYXJnZXRHcm91cHMoaWQsIHtcbiAgICAgICAgICAgIGNvbmRpdGlvbnM6IHByb3BzLmNvbmRpdGlvbnMsXG4gICAgICAgICAgICBob3N0SGVhZGVyOiBwcm9wcy5ob3N0SGVhZGVyLFxuICAgICAgICAgICAgcGF0aFBhdHRlcm46IHByb3BzLnBhdGhQYXR0ZXJuLFxuICAgICAgICAgICAgcGF0aFBhdHRlcm5zOiBwcm9wcy5wYXRoUGF0dGVybnMsXG4gICAgICAgICAgICBwcmlvcml0eTogcHJvcHMucHJpb3JpdHksXG4gICAgICAgICAgICB0YXJnZXRHcm91cHM6IFtncm91cF0sXG4gICAgICAgIH0pO1xuICAgICAgICByZXR1cm4gZ3JvdXA7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZCBhIGZpeGVkIHJlc3BvbnNlXG4gICAgICpcbiAgICAgKiBAZGVwcmVjYXRlZCBVc2UgYGFkZEFjdGlvbigpYCBpbnN0ZWFkXG4gICAgICovXG4gICAgcHVibGljIGFkZEZpeGVkUmVzcG9uc2UoaWQ6IHN0cmluZywgcHJvcHM6IEFkZEZpeGVkUmVzcG9uc2VQcm9wcykge1xuICAgICAgICBjaGVja0FkZFJ1bGVQcm9wcyhwcm9wcyk7XG4gICAgICAgIGNvbnN0IGZpeGVkUmVzcG9uc2U6IEZpeGVkUmVzcG9uc2UgPSB7XG4gICAgICAgICAgICBzdGF0dXNDb2RlOiBwcm9wcy5zdGF0dXNDb2RlLFxuICAgICAgICAgICAgY29udGVudFR5cGU6IHByb3BzLmNvbnRlbnRUeXBlLFxuICAgICAgICAgICAgbWVzc2FnZUJvZHk6IHByb3BzLm1lc3NhZ2VCb2R5LFxuICAgICAgICB9O1xuICAgICAgICB2YWxpZGF0ZUZpeGVkUmVzcG9uc2UoZml4ZWRSZXNwb25zZSk7XG4gICAgICAgIGlmIChwcm9wcy5wcmlvcml0eSkge1xuICAgICAgICAgICAgbmV3IEFwcGxpY2F0aW9uTGlzdGVuZXJSdWxlKHRoaXMsIGlkICsgJ1J1bGUnLCB7XG4gICAgICAgICAgICAgICAgbGlzdGVuZXI6IHRoaXMsXG4gICAgICAgICAgICAgICAgcHJpb3JpdHk6IHByb3BzLnByaW9yaXR5LFxuICAgICAgICAgICAgICAgIGZpeGVkUmVzcG9uc2UsXG4gICAgICAgICAgICAgICAgLi4ucHJvcHMsXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgIHRoaXMuc2V0RGVmYXVsdEFjdGlvbihMaXN0ZW5lckFjdGlvbi5maXhlZFJlc3BvbnNlKFRva2VuLmFzTnVtYmVyKHByb3BzLnN0YXR1c0NvZGUpLCB7XG4gICAgICAgICAgICAgICAgY29udGVudFR5cGU6IHByb3BzLmNvbnRlbnRUeXBlLFxuICAgICAgICAgICAgICAgIG1lc3NhZ2VCb2R5OiBwcm9wcy5tZXNzYWdlQm9keSxcbiAgICAgICAgICAgIH0pKTtcbiAgICAgICAgfVxuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGQgYSByZWRpcmVjdCByZXNwb25zZVxuICAgICAqXG4gICAgICogQGRlcHJlY2F0ZWQgVXNlIGBhZGRBY3Rpb24oKWAgaW5zdGVhZFxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRSZWRpcmVjdFJlc3BvbnNlKGlkOiBzdHJpbmcsIHByb3BzOiBBZGRSZWRpcmVjdFJlc3BvbnNlUHJvcHMpIHtcbiAgICAgICAgY2hlY2tBZGRSdWxlUHJvcHMocHJvcHMpO1xuICAgICAgICBjb25zdCByZWRpcmVjdFJlc3BvbnNlID0ge1xuICAgICAgICAgICAgaG9zdDogcHJvcHMuaG9zdCxcbiAgICAgICAgICAgIHBhdGg6IHByb3BzLnBhdGgsXG4gICAgICAgICAgICBwb3J0OiBwcm9wcy5wb3J0LFxuICAgICAgICAgICAgcHJvdG9jb2w6IHByb3BzLnByb3RvY29sLFxuICAgICAgICAgICAgcXVlcnk6IHByb3BzLnF1ZXJ5LFxuICAgICAgICAgICAgc3RhdHVzQ29kZTogcHJvcHMuc3RhdHVzQ29kZSxcbiAgICAgICAgfTtcbiAgICAgICAgdmFsaWRhdGVSZWRpcmVjdFJlc3BvbnNlKHJlZGlyZWN0UmVzcG9uc2UpO1xuICAgICAgICBpZiAocHJvcHMucHJpb3JpdHkpIHtcbiAgICAgICAgICAgIG5ldyBBcHBsaWNhdGlvbkxpc3RlbmVyUnVsZSh0aGlzLCBpZCArICdSdWxlJywge1xuICAgICAgICAgICAgICAgIGxpc3RlbmVyOiB0aGlzLFxuICAgICAgICAgICAgICAgIHByaW9yaXR5OiBwcm9wcy5wcmlvcml0eSxcbiAgICAgICAgICAgICAgICByZWRpcmVjdFJlc3BvbnNlLFxuICAgICAgICAgICAgICAgIC4uLnByb3BzLFxuICAgICAgICAgICAgfSk7XG4gICAgICAgIH1cbiAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICB0aGlzLnNldERlZmF1bHRBY3Rpb24oTGlzdGVuZXJBY3Rpb24ucmVkaXJlY3Qoe1xuICAgICAgICAgICAgICAgIGhvc3Q6IHByb3BzLmhvc3QsXG4gICAgICAgICAgICAgICAgcGF0aDogcHJvcHMucGF0aCxcbiAgICAgICAgICAgICAgICBwb3J0OiBwcm9wcy5wb3J0LFxuICAgICAgICAgICAgICAgIHByb3RvY29sOiBwcm9wcy5wcm90b2NvbCxcbiAgICAgICAgICAgICAgICBxdWVyeTogcHJvcHMucXVlcnksXG4gICAgICAgICAgICAgICAgcGVybWFuZW50OiBwcm9wcy5zdGF0dXNDb2RlID09PSAnSFRUUF8zMDEnLFxuICAgICAgICAgICAgfSkpO1xuICAgICAgICB9XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJlZ2lzdGVyIHRoYXQgYSBjb25uZWN0YWJsZSB0aGF0IGhhcyBiZWVuIGFkZGVkIHRvIHRoaXMgbG9hZCBiYWxhbmNlci5cbiAgICAgKlxuICAgICAqIERvbid0IGNhbGwgdGhpcyBkaXJlY3RseS4gSXQgaXMgY2FsbGVkIGJ5IEFwcGxpY2F0aW9uVGFyZ2V0R3JvdXAuXG4gICAgICovXG4gICAgcHVibGljIHJlZ2lzdGVyQ29ubmVjdGFibGUoY29ubmVjdGFibGU6IGVjMi5JQ29ubmVjdGFibGUsIHBvcnRSYW5nZTogZWMyLlBvcnQpOiB2b2lkIHtcbiAgICAgICAgY29ubmVjdGFibGUuY29ubmVjdGlvbnMuYWxsb3dGcm9tKHRoaXMubG9hZEJhbGFuY2VyLCBwb3J0UmFuZ2UsICdMb2FkIGJhbGFuY2VyIHRvIHRhcmdldCcpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBWYWxpZGF0ZSB0aGlzIGxpc3RlbmVyLlxuICAgICAqL1xuICAgIHByb3RlY3RlZCB2YWxpZGF0ZSgpOiBzdHJpbmdbXSB7XG4gICAgICAgIGNvbnN0IGVycm9ycyA9IHN1cGVyLnZhbGlkYXRlKCk7XG4gICAgICAgIGlmICh0aGlzLnByb3RvY29sID09PSBBcHBsaWNhdGlvblByb3RvY29sLkhUVFBTICYmIHRoaXMuY2VydGlmaWNhdGVBcm5zLmxlbmd0aCA9PT0gMCkge1xuICAgICAgICAgICAgZXJyb3JzLnB1c2goJ0hUVFBTIExpc3RlbmVyIG5lZWRzIGF0IGxlYXN0IG9uZSBjZXJ0aWZpY2F0ZSAoY2FsbCBhZGRDZXJ0aWZpY2F0ZXMpJyk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIGVycm9ycztcbiAgICB9XG4gICAgLyoqXG4gICAgICogV3JhcHBlciBmb3IgX3NldERlZmF1bHRBY3Rpb24gd2hpY2ggZG9lcyBhIHR5cGUtc2FmZSBiaW5kXG4gICAgICovXG4gICAgcHJpdmF0ZSBzZXREZWZhdWx0QWN0aW9uKGFjdGlvbjogTGlzdGVuZXJBY3Rpb24pIHtcbiAgICAgICAgYWN0aW9uLmJpbmQodGhpcywgdGhpcyk7XG4gICAgICAgIHRoaXMuX3NldERlZmF1bHRBY3Rpb24oYWN0aW9uKTtcbiAgICB9XG59XG4vKipcbiAqIFByb3BlcnRpZXMgdG8gcmVmZXJlbmNlIGFuIGV4aXN0aW5nIGxpc3RlbmVyXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSUFwcGxpY2F0aW9uTGlzdGVuZXIgZXh0ZW5kcyBJUmVzb3VyY2UsIGVjMi5JQ29ubmVjdGFibGUge1xuICAgIC8qKlxuICAgICAqIEFSTiBvZiB0aGUgbGlzdGVuZXJcbiAgICAgKiBAYXR0cmlidXRlXG4gICAgICovXG4gICAgcmVhZG9ubHkgbGlzdGVuZXJBcm46IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBBZGQgb25lIG9yIG1vcmUgY2VydGlmaWNhdGVzIHRvIHRoaXMgbGlzdGVuZXIuXG4gICAgICovXG4gICAgYWRkQ2VydGlmaWNhdGVBcm5zKGlkOiBzdHJpbmcsIGFybnM6IHN0cmluZ1tdKTogdm9pZDtcbiAgICAvKipcbiAgICAgKiBMb2FkIGJhbGFuY2UgaW5jb21pbmcgcmVxdWVzdHMgdG8gdGhlIGdpdmVuIHRhcmdldCBncm91cHMuXG4gICAgICpcbiAgICAgKiBJdCdzIHBvc3NpYmxlIHRvIGFkZCBjb25kaXRpb25zIHRvIHRoZSBUYXJnZXRHcm91cHMgYWRkZWQgaW4gdGhpcyB3YXkuXG4gICAgICogQXQgbGVhc3Qgb25lIFRhcmdldEdyb3VwIG11c3QgYmUgYWRkZWQgd2l0aG91dCBjb25kaXRpb25zLlxuICAgICAqL1xuICAgIGFkZFRhcmdldEdyb3VwcyhpZDogc3RyaW5nLCBwcm9wczogQWRkQXBwbGljYXRpb25UYXJnZXRHcm91cHNQcm9wcyk6IHZvaWQ7XG4gICAgLyoqXG4gICAgICogTG9hZCBiYWxhbmNlIGluY29taW5nIHJlcXVlc3RzIHRvIHRoZSBnaXZlbiBsb2FkIGJhbGFuY2luZyB0YXJnZXRzLlxuICAgICAqXG4gICAgICogVGhpcyBtZXRob2QgaW1wbGljaXRseSBjcmVhdGVzIGFuIEFwcGxpY2F0aW9uVGFyZ2V0R3JvdXAgZm9yIHRoZSB0YXJnZXRzXG4gICAgICogaW52b2x2ZWQuXG4gICAgICpcbiAgICAgKiBJdCdzIHBvc3NpYmxlIHRvIGFkZCBjb25kaXRpb25zIHRvIHRoZSB0YXJnZXRzIGFkZGVkIGluIHRoaXMgd2F5LiBBdCBsZWFzdFxuICAgICAqIG9uZSBzZXQgb2YgdGFyZ2V0cyBtdXN0IGJlIGFkZGVkIHdpdGhvdXQgY29uZGl0aW9ucy5cbiAgICAgKlxuICAgICAqIEByZXR1cm5zIFRoZSBuZXdseSBjcmVhdGVkIHRhcmdldCBncm91cFxuICAgICAqL1xuICAgIGFkZFRhcmdldHMoaWQ6IHN0cmluZywgcHJvcHM6IEFkZEFwcGxpY2F0aW9uVGFyZ2V0c1Byb3BzKTogQXBwbGljYXRpb25UYXJnZXRHcm91cDtcbiAgICAvKipcbiAgICAgKiBSZWdpc3RlciB0aGF0IGEgY29ubmVjdGFibGUgdGhhdCBoYXMgYmVlbiBhZGRlZCB0byB0aGlzIGxvYWQgYmFsYW5jZXIuXG4gICAgICpcbiAgICAgKiBEb24ndCBjYWxsIHRoaXMgZGlyZWN0bHkuIEl0IGlzIGNhbGxlZCBieSBBcHBsaWNhdGlvblRhcmdldEdyb3VwLlxuICAgICAqL1xuICAgIHJlZ2lzdGVyQ29ubmVjdGFibGUoY29ubmVjdGFibGU6IGVjMi5JQ29ubmVjdGFibGUsIHBvcnRSYW5nZTogZWMyLlBvcnQpOiB2b2lkO1xufVxuLyoqXG4gKiBQcm9wZXJ0aWVzIHRvIHJlZmVyZW5jZSBhbiBleGlzdGluZyBsaXN0ZW5lclxuICovXG5leHBvcnQgaW50ZXJmYWNlIEFwcGxpY2F0aW9uTGlzdGVuZXJBdHRyaWJ1dGVzIHtcbiAgICAvKipcbiAgICAgKiBBUk4gb2YgdGhlIGxpc3RlbmVyXG4gICAgICovXG4gICAgcmVhZG9ubHkgbGlzdGVuZXJBcm46IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBTZWN1cml0eSBncm91cCBJRCBvZiB0aGUgbG9hZCBiYWxhbmNlciB0aGlzIGxpc3RlbmVyIGlzIGFzc29jaWF0ZWQgd2l0aFxuICAgICAqXG4gICAgICogQGRlcHJlY2F0ZWQgdXNlIGBzZWN1cml0eUdyb3VwYCBpbnN0ZWFkXG4gICAgICovXG4gICAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cElkPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFNlY3VyaXR5IGdyb3VwIG9mIHRoZSBsb2FkIGJhbGFuY2VyIHRoaXMgbGlzdGVuZXIgaXMgYXNzb2NpYXRlZCB3aXRoXG4gICAgICovXG4gICAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cD86IGVjMi5JU2VjdXJpdHlHcm91cDtcbiAgICAvKipcbiAgICAgKiBUaGUgZGVmYXVsdCBwb3J0IG9uIHdoaWNoIHRoaXMgbGlzdGVuZXIgaXMgbGlzdGVuaW5nXG4gICAgICovXG4gICAgcmVhZG9ubHkgZGVmYXVsdFBvcnQ/OiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogV2hldGhlciB0aGUgaW1wb3J0ZWQgc2VjdXJpdHkgZ3JvdXAgYWxsb3dzIGFsbCBvdXRib3VuZCB0cmFmZmljIG9yIG5vdCB3aGVuXG4gICAgICogaW1wb3J0ZWQgdXNpbmcgYHNlY3VyaXR5R3JvdXBJZGBcbiAgICAgKlxuICAgICAqIFVubGVzcyBzZXQgdG8gYGZhbHNlYCwgbm8gZWdyZXNzIHJ1bGVzIHdpbGwgYmUgYWRkZWQgdG8gdGhlIHNlY3VyaXR5IGdyb3VwLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgdHJ1ZVxuICAgICAqXG4gICAgICogQGRlcHJlY2F0ZWQgdXNlIGBzZWN1cml0eUdyb3VwYCBpbnN0ZWFkXG4gICAgICovXG4gICAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cEFsbG93c0FsbE91dGJvdW5kPzogYm9vbGVhbjtcbn1cbmNsYXNzIEltcG9ydGVkQXBwbGljYXRpb25MaXN0ZW5lciBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSUFwcGxpY2F0aW9uTGlzdGVuZXIge1xuICAgIHB1YmxpYyByZWFkb25seSBjb25uZWN0aW9uczogZWMyLkNvbm5lY3Rpb25zO1xuICAgIC8qKlxuICAgICAqIEFSTiBvZiB0aGUgbGlzdGVuZXJcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgbGlzdGVuZXJBcm46IHN0cmluZztcbiAgICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQXBwbGljYXRpb25MaXN0ZW5lckF0dHJpYnV0ZXMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgdGhpcy5saXN0ZW5lckFybiA9IHByb3BzLmxpc3RlbmVyQXJuO1xuICAgICAgICBjb25zdCBkZWZhdWx0UG9ydCA9IHByb3BzLmRlZmF1bHRQb3J0ICE9PSB1bmRlZmluZWQgPyBlYzIuUG9ydC50Y3AocHJvcHMuZGVmYXVsdFBvcnQpIDogdW5kZWZpbmVkO1xuICAgICAgICBsZXQgc2VjdXJpdHlHcm91cDogZWMyLklTZWN1cml0eUdyb3VwO1xuICAgICAgICBpZiAocHJvcHMuc2VjdXJpdHlHcm91cCkge1xuICAgICAgICAgICAgc2VjdXJpdHlHcm91cCA9IHByb3BzLnNlY3VyaXR5R3JvdXA7XG4gICAgICAgIH1cbiAgICAgICAgZWxzZSBpZiAocHJvcHMuc2VjdXJpdHlHcm91cElkKSB7XG4gICAgICAgICAgICBzZWN1cml0eUdyb3VwID0gZWMyLlNlY3VyaXR5R3JvdXAuZnJvbVNlY3VyaXR5R3JvdXBJZChzY29wZSwgJ1NlY3VyaXR5R3JvdXAnLCBwcm9wcy5zZWN1cml0eUdyb3VwSWQsIHtcbiAgICAgICAgICAgICAgICBhbGxvd0FsbE91dGJvdW5kOiBwcm9wcy5zZWN1cml0eUdyb3VwQWxsb3dzQWxsT3V0Ym91bmQsXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignRWl0aGVyIGBzZWN1cml0eUdyb3VwYCBvciBgc2VjdXJpdHlHcm91cElkYCBtdXN0IGJlIHNwZWNpZmllZCB0byBpbXBvcnQgYW4gYXBwbGljYXRpb24gbGlzdGVuZXIuJyk7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5jb25uZWN0aW9ucyA9IG5ldyBlYzIuQ29ubmVjdGlvbnMoe1xuICAgICAgICAgICAgc2VjdXJpdHlHcm91cHM6IFtzZWN1cml0eUdyb3VwXSxcbiAgICAgICAgICAgIGRlZmF1bHRQb3J0LFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQWRkIG9uZSBvciBtb3JlIGNlcnRpZmljYXRlcyB0byB0aGlzIGxpc3RlbmVyLlxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRDZXJ0aWZpY2F0ZUFybnMoaWQ6IHN0cmluZywgYXJuczogc3RyaW5nW10pOiB2b2lkIHtcbiAgICAgICAgbmV3IEFwcGxpY2F0aW9uTGlzdGVuZXJDZXJ0aWZpY2F0ZSh0aGlzLCBpZCwge1xuICAgICAgICAgICAgbGlzdGVuZXI6IHRoaXMsXG4gICAgICAgICAgICBjZXJ0aWZpY2F0ZUFybnM6IGFybnMsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBMb2FkIGJhbGFuY2UgaW5jb21pbmcgcmVxdWVzdHMgdG8gdGhlIGdpdmVuIHRhcmdldCBncm91cHMuXG4gICAgICpcbiAgICAgKiBJdCdzIHBvc3NpYmxlIHRvIGFkZCBjb25kaXRpb25zIHRvIHRoZSBUYXJnZXRHcm91cHMgYWRkZWQgaW4gdGhpcyB3YXkuXG4gICAgICogQXQgbGVhc3Qgb25lIFRhcmdldEdyb3VwIG11c3QgYmUgYWRkZWQgd2l0aG91dCBjb25kaXRpb25zLlxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRUYXJnZXRHcm91cHMoaWQ6IHN0cmluZywgcHJvcHM6IEFkZEFwcGxpY2F0aW9uVGFyZ2V0R3JvdXBzUHJvcHMpOiB2b2lkIHtcbiAgICAgICAgY2hlY2tBZGRSdWxlUHJvcHMocHJvcHMpO1xuICAgICAgICBpZiAocHJvcHMucHJpb3JpdHkgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgLy8gTmV3IHJ1bGVcbiAgICAgICAgICAgIG5ldyBBcHBsaWNhdGlvbkxpc3RlbmVyUnVsZSh0aGlzLCBpZCwge1xuICAgICAgICAgICAgICAgIGxpc3RlbmVyOiB0aGlzLFxuICAgICAgICAgICAgICAgIGhvc3RIZWFkZXI6IHByb3BzLmhvc3RIZWFkZXIsXG4gICAgICAgICAgICAgICAgcGF0aFBhdHRlcm46IHByb3BzLnBhdGhQYXR0ZXJuLFxuICAgICAgICAgICAgICAgIHBhdGhQYXR0ZXJuczogcHJvcHMucGF0aFBhdHRlcm5zLFxuICAgICAgICAgICAgICAgIHByaW9yaXR5OiBwcm9wcy5wcmlvcml0eSxcbiAgICAgICAgICAgICAgICB0YXJnZXRHcm91cHM6IHByb3BzLnRhcmdldEdyb3VwcyxcbiAgICAgICAgICAgIH0pO1xuICAgICAgICB9XG4gICAgICAgIGVsc2Uge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgYWRkIGRlZmF1bHQgVGFyZ2V0IEdyb3VwcyB0byBpbXBvcnRlZCBBcHBsaWNhdGlvbkxpc3RlbmVyJyk7XG4gICAgICAgIH1cbiAgICB9XG4gICAgLyoqXG4gICAgICogTG9hZCBiYWxhbmNlIGluY29taW5nIHJlcXVlc3RzIHRvIHRoZSBnaXZlbiBsb2FkIGJhbGFuY2luZyB0YXJnZXRzLlxuICAgICAqXG4gICAgICogVGhpcyBtZXRob2QgaW1wbGljaXRseSBjcmVhdGVzIGFuIEFwcGxpY2F0aW9uVGFyZ2V0R3JvdXAgZm9yIHRoZSB0YXJnZXRzXG4gICAgICogaW52b2x2ZWQuXG4gICAgICpcbiAgICAgKiBJdCdzIHBvc3NpYmxlIHRvIGFkZCBjb25kaXRpb25zIHRvIHRoZSB0YXJnZXRzIGFkZGVkIGluIHRoaXMgd2F5LiBBdCBsZWFzdFxuICAgICAqIG9uZSBzZXQgb2YgdGFyZ2V0cyBtdXN0IGJlIGFkZGVkIHdpdGhvdXQgY29uZGl0aW9ucy5cbiAgICAgKlxuICAgICAqIEByZXR1cm5zIFRoZSBuZXdseSBjcmVhdGVkIHRhcmdldCBncm91cFxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRUYXJnZXRzKF9pZDogc3RyaW5nLCBfcHJvcHM6IEFkZEFwcGxpY2F0aW9uVGFyZ2V0c1Byb3BzKTogQXBwbGljYXRpb25UYXJnZXRHcm91cCB7XG4gICAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBtYXgtbGVuXG4gICAgICAgIHRocm93IG5ldyBFcnJvcignQ2FuIG9ubHkgY2FsbCBhZGRUYXJnZXRzKCkgd2hlbiB1c2luZyBhIGNvbnN0cnVjdGVkIEFwcGxpY2F0aW9uTGlzdGVuZXI7IGNvbnN0cnVjdCBhIG5ldyBUYXJnZXRHcm91cCBhbmQgdXNlIGFkZFRhcmdldEdyb3VwLicpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZWdpc3RlciB0aGF0IGEgY29ubmVjdGFibGUgdGhhdCBoYXMgYmVlbiBhZGRlZCB0byB0aGlzIGxvYWQgYmFsYW5jZXIuXG4gICAgICpcbiAgICAgKiBEb24ndCBjYWxsIHRoaXMgZGlyZWN0bHkuIEl0IGlzIGNhbGxlZCBieSBBcHBsaWNhdGlvblRhcmdldEdyb3VwLlxuICAgICAqL1xuICAgIHB1YmxpYyByZWdpc3RlckNvbm5lY3RhYmxlKGNvbm5lY3RhYmxlOiBlYzIuSUNvbm5lY3RhYmxlLCBwb3J0UmFuZ2U6IGVjMi5Qb3J0KTogdm9pZCB7XG4gICAgICAgIHRoaXMuY29ubmVjdGlvbnMuYWxsb3dUbyhjb25uZWN0YWJsZSwgcG9ydFJhbmdlLCAnTG9hZCBiYWxhbmNlciB0byB0YXJnZXQnKTtcbiAgICB9XG59XG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGFkZGluZyBhIGNvbmRpdGlvbmFsIGxvYWQgYmFsYW5jaW5nIHJ1bGVcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBZGRSdWxlUHJvcHMge1xuICAgIC8qKlxuICAgICAqIFByaW9yaXR5IG9mIHRoaXMgdGFyZ2V0IGdyb3VwXG4gICAgICpcbiAgICAgKiBUaGUgcnVsZSB3aXRoIHRoZSBsb3dlc3QgcHJpb3JpdHkgd2lsbCBiZSB1c2VkIGZvciBldmVyeSByZXF1ZXN0LlxuICAgICAqIElmIHByaW9yaXR5IGlzIG5vdCBnaXZlbiwgdGhlc2UgdGFyZ2V0IGdyb3VwcyB3aWxsIGJlIGFkZGVkIGFzXG4gICAgICogZGVmYXVsdHMsIGFuZCBtdXN0IG5vdCBoYXZlIGNvbmRpdGlvbnMuXG4gICAgICpcbiAgICAgKiBQcmlvcml0aWVzIG11c3QgYmUgdW5pcXVlLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgVGFyZ2V0IGdyb3VwcyBhcmUgdXNlZCBhcyBkZWZhdWx0c1xuICAgICAqL1xuICAgIHJlYWRvbmx5IHByaW9yaXR5PzogbnVtYmVyO1xuICAgIC8qKlxuICAgICAqIFJ1bGUgYXBwbGllcyBpZiBtYXRjaGVzIHRoZSBjb25kaXRpb25zLlxuICAgICAqXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2xvYWRiYWxhbmNpbmcvbGF0ZXN0L2FwcGxpY2F0aW9uL2xvYWQtYmFsYW5jZXItbGlzdGVuZXJzLmh0bWxcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gY29uZGl0aW9ucy5cbiAgICAgKi9cbiAgICByZWFkb25seSBjb25kaXRpb25zPzogTGlzdGVuZXJDb25kaXRpb25bXTtcbiAgICAvKipcbiAgICAgKiBSdWxlIGFwcGxpZXMgaWYgdGhlIHJlcXVlc3RlZCBob3N0IG1hdGNoZXMgdGhlIGluZGljYXRlZCBob3N0XG4gICAgICpcbiAgICAgKiBNYXkgY29udGFpbiB1cCB0byB0aHJlZSAnKicgd2lsZGNhcmRzLlxuICAgICAqXG4gICAgICogUmVxdWlyZXMgdGhhdCBwcmlvcml0eSBpcyBzZXQuXG4gICAgICpcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljbG9hZGJhbGFuY2luZy9sYXRlc3QvYXBwbGljYXRpb24vbG9hZC1iYWxhbmNlci1saXN0ZW5lcnMuaHRtbCNob3N0LWNvbmRpdGlvbnNcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGhvc3QgY29uZGl0aW9uXG4gICAgICogQGRlcHJlY2F0ZWQgVXNlIGBjb25kaXRpb25zYCBpbnN0ZWFkLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGhvc3RIZWFkZXI/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogUnVsZSBhcHBsaWVzIGlmIHRoZSByZXF1ZXN0ZWQgcGF0aCBtYXRjaGVzIHRoZSBnaXZlbiBwYXRoIHBhdHRlcm5cbiAgICAgKlxuICAgICAqIE1heSBjb250YWluIHVwIHRvIHRocmVlICcqJyB3aWxkY2FyZHMuXG4gICAgICpcbiAgICAgKiBSZXF1aXJlcyB0aGF0IHByaW9yaXR5IGlzIHNldC5cbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNsb2FkYmFsYW5jaW5nL2xhdGVzdC9hcHBsaWNhdGlvbi9sb2FkLWJhbGFuY2VyLWxpc3RlbmVycy5odG1sI3BhdGgtY29uZGl0aW9uc1xuICAgICAqIEBkZWZhdWx0IE5vIHBhdGggY29uZGl0aW9uXG4gICAgICogQGRlcHJlY2F0ZWQgVXNlIGBjb25kaXRpb25zYCBpbnN0ZWFkLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHBhdGhQYXR0ZXJuPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFJ1bGUgYXBwbGllcyBpZiB0aGUgcmVxdWVzdGVkIHBhdGggbWF0Y2hlcyBhbnkgb2YgdGhlIGdpdmVuIHBhdHRlcm5zLlxuICAgICAqXG4gICAgICogTWF5IGNvbnRhaW4gdXAgdG8gdGhyZWUgJyonIHdpbGRjYXJkcy5cbiAgICAgKlxuICAgICAqIFJlcXVpcmVzIHRoYXQgcHJpb3JpdHkgaXMgc2V0LlxuICAgICAqXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2xvYWRiYWxhbmNpbmcvbGF0ZXN0L2FwcGxpY2F0aW9uL2xvYWQtYmFsYW5jZXItbGlzdGVuZXJzLmh0bWwjcGF0aC1jb25kaXRpb25zXG4gICAgICogQGRlZmF1bHQgLSBObyBwYXRoIGNvbmRpdGlvbi5cbiAgICAgKiBAZGVwcmVjYXRlZCBVc2UgYGNvbmRpdGlvbnNgIGluc3RlYWQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgcGF0aFBhdHRlcm5zPzogc3RyaW5nW107XG59XG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGFkZGluZyBhIG5ldyB0YXJnZXQgZ3JvdXAgdG8gYSBsaXN0ZW5lclxuICovXG5leHBvcnQgaW50ZXJmYWNlIEFkZEFwcGxpY2F0aW9uVGFyZ2V0R3JvdXBzUHJvcHMgZXh0ZW5kcyBBZGRSdWxlUHJvcHMge1xuICAgIC8qKlxuICAgICAqIFRhcmdldCBncm91cHMgdG8gZm9yd2FyZCByZXF1ZXN0cyB0b1xuICAgICAqL1xuICAgIHJlYWRvbmx5IHRhcmdldEdyb3VwczogSUFwcGxpY2F0aW9uVGFyZ2V0R3JvdXBbXTtcbn1cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYWRkaW5nIGEgbmV3IGFjdGlvbiB0byBhIGxpc3RlbmVyXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQWRkQXBwbGljYXRpb25BY3Rpb25Qcm9wcyBleHRlbmRzIEFkZFJ1bGVQcm9wcyB7XG4gICAgLyoqXG4gICAgICogQWN0aW9uIHRvIHBlcmZvcm1cbiAgICAgKi9cbiAgICByZWFkb25seSBhY3Rpb246IExpc3RlbmVyQWN0aW9uO1xufVxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBhZGRpbmcgbmV3IHRhcmdldHMgdG8gYSBsaXN0ZW5lclxuICovXG5leHBvcnQgaW50ZXJmYWNlIEFkZEFwcGxpY2F0aW9uVGFyZ2V0c1Byb3BzIGV4dGVuZHMgQWRkUnVsZVByb3BzIHtcbiAgICAvKipcbiAgICAgKiBUaGUgcHJvdG9jb2wgdG8gdXNlXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBEZXRlcm1pbmVkIGZyb20gcG9ydCBpZiBrbm93blxuICAgICAqL1xuICAgIHJlYWRvbmx5IHByb3RvY29sPzogQXBwbGljYXRpb25Qcm90b2NvbDtcbiAgICAvKipcbiAgICAgKiBUaGUgcG9ydCBvbiB3aGljaCB0aGUgbGlzdGVuZXIgbGlzdGVucyBmb3IgcmVxdWVzdHMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBEZXRlcm1pbmVkIGZyb20gcHJvdG9jb2wgaWYga25vd25cbiAgICAgKi9cbiAgICByZWFkb25seSBwb3J0PzogbnVtYmVyO1xuICAgIC8qKlxuICAgICAqIFRoZSB0aW1lIHBlcmlvZCBkdXJpbmcgd2hpY2ggdGhlIGxvYWQgYmFsYW5jZXIgc2VuZHMgYSBuZXdseSByZWdpc3RlcmVkXG4gICAgICogdGFyZ2V0IGEgbGluZWFybHkgaW5jcmVhc2luZyBzaGFyZSBvZiB0aGUgdHJhZmZpYyB0byB0aGUgdGFyZ2V0IGdyb3VwLlxuICAgICAqXG4gICAgICogVGhlIHJhbmdlIGlzIDMwLTkwMCBzZWNvbmRzICgxNSBtaW51dGVzKS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IDBcbiAgICAgKi9cbiAgICByZWFkb25seSBzbG93U3RhcnQ/OiBEdXJhdGlvbjtcbiAgICAvKipcbiAgICAgKiBUaGUgc3RpY2tpbmVzcyBjb29raWUgZXhwaXJhdGlvbiBwZXJpb2QuXG4gICAgICpcbiAgICAgKiBTZXR0aW5nIHRoaXMgdmFsdWUgZW5hYmxlcyBsb2FkIGJhbGFuY2VyIHN0aWNraW5lc3MuXG4gICAgICpcbiAgICAgKiBBZnRlciB0aGlzIHBlcmlvZCwgdGhlIGNvb2tpZSBpcyBjb25zaWRlcmVkIHN0YWxlLiBUaGUgbWluaW11bSB2YWx1ZSBpc1xuICAgICAqIDEgc2Vjb25kIGFuZCB0aGUgbWF4aW11bSB2YWx1ZSBpcyA3IGRheXMgKDYwNDgwMCBzZWNvbmRzKS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IFN0aWNraW5lc3MgZGlzYWJsZWRcbiAgICAgKi9cbiAgICByZWFkb25seSBzdGlja2luZXNzQ29va2llRHVyYXRpb24/OiBEdXJhdGlvbjtcbiAgICAvKipcbiAgICAgKiBUaGUgdGFyZ2V0cyB0byBhZGQgdG8gdGhpcyB0YXJnZXQgZ3JvdXAuXG4gICAgICpcbiAgICAgKiBDYW4gYmUgYEluc3RhbmNlYCwgYElQQWRkcmVzc2AsIG9yIGFueSBzZWxmLXJlZ2lzdGVyaW5nIGxvYWQgYmFsYW5jaW5nXG4gICAgICogdGFyZ2V0LiBBbGwgdGFyZ2V0IG11c3QgYmUgb2YgdGhlIHNhbWUgdHlwZS5cbiAgICAgKi9cbiAgICByZWFkb25seSB0YXJnZXRzPzogSUFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyVGFyZ2V0W107XG4gICAgLyoqXG4gICAgICogVGhlIG5hbWUgb2YgdGhlIHRhcmdldCBncm91cC5cbiAgICAgKlxuICAgICAqIFRoaXMgbmFtZSBtdXN0IGJlIHVuaXF1ZSBwZXIgcmVnaW9uIHBlciBhY2NvdW50LCBjYW4gaGF2ZSBhIG1heGltdW0gb2ZcbiAgICAgKiAzMiBjaGFyYWN0ZXJzLCBtdXN0IGNvbnRhaW4gb25seSBhbHBoYW51bWVyaWMgY2hhcmFjdGVycyBvciBoeXBoZW5zLCBhbmRcbiAgICAgKiBtdXN0IG5vdCBiZWdpbiBvciBlbmQgd2l0aCBhIGh5cGhlbi5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IEF1dG9tYXRpY2FsbHkgZ2VuZXJhdGVkXG4gICAgICovXG4gICAgcmVhZG9ubHkgdGFyZ2V0R3JvdXBOYW1lPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBhbW91bnQgb2YgdGltZSBmb3IgRWxhc3RpYyBMb2FkIEJhbGFuY2luZyB0byB3YWl0IGJlZm9yZSBkZXJlZ2lzdGVyaW5nIGEgdGFyZ2V0LlxuICAgICAqXG4gICAgICogVGhlIHJhbmdlIGlzIDAtMzYwMCBzZWNvbmRzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgRHVyYXRpb24ubWludXRlcyg1KVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGRlcmVnaXN0cmF0aW9uRGVsYXk/OiBEdXJhdGlvbjtcbiAgICAvKipcbiAgICAgKiBIZWFsdGggY2hlY2sgY29uZmlndXJhdGlvblxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gaGVhbHRoIGNoZWNrXG4gICAgICovXG4gICAgcmVhZG9ubHkgaGVhbHRoQ2hlY2s/OiBIZWFsdGhDaGVjaztcbn1cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYWRkaW5nIGEgZml4ZWQgcmVzcG9uc2UgdG8gYSBsaXN0ZW5lclxuICpcbiAqIEBkZXByZWNhdGVkIFVzZSBgQXBwbGljYXRpb25MaXN0ZW5lci5hZGRBY3Rpb25gIGluc3RlYWQuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQWRkRml4ZWRSZXNwb25zZVByb3BzIGV4dGVuZHMgQWRkUnVsZVByb3BzLCBGaXhlZFJlc3BvbnNlIHtcbn1cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYWRkaW5nIGEgcmVkaXJlY3QgcmVzcG9uc2UgdG8gYSBsaXN0ZW5lclxuICpcbiAqIEBkZXByZWNhdGVkIFVzZSBgQXBwbGljYXRpb25MaXN0ZW5lci5hZGRBY3Rpb25gIGluc3RlYWQuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQWRkUmVkaXJlY3RSZXNwb25zZVByb3BzIGV4dGVuZHMgQWRkUnVsZVByb3BzLCBSZWRpcmVjdFJlc3BvbnNlIHtcbn1cbmZ1bmN0aW9uIGNoZWNrQWRkUnVsZVByb3BzKHByb3BzOiBBZGRSdWxlUHJvcHMpIHtcbiAgICBjb25zdCBjb25kaXRpb25zQ291bnQgPSBwcm9wcy5jb25kaXRpb25zPy5sZW5ndGggfHwgMDtcbiAgICBjb25zdCBoYXNBbnlDb25kaXRpb25zID0gY29uZGl0aW9uc0NvdW50ICE9PSAwIHx8XG4gICAgICAgIHByb3BzLmhvc3RIZWFkZXIgIT09IHVuZGVmaW5lZCB8fCBwcm9wcy5wYXRoUGF0dGVybiAhPT0gdW5kZWZpbmVkIHx8IHByb3BzLnBhdGhQYXR0ZXJucyAhPT0gdW5kZWZpbmVkO1xuICAgIGNvbnN0IGhhc1ByaW9yaXR5ID0gcHJvcHMucHJpb3JpdHkgIT09IHVuZGVmaW5lZDtcbiAgICBpZiAoaGFzQW55Q29uZGl0aW9ucyAhPT0gaGFzUHJpb3JpdHkpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdTZXR0aW5nIFxcJ2NvbmRpdGlvbnNcXCcsIFxcJ3BhdGhQYXR0ZXJuXFwnIG9yIFxcJ2hvc3RIZWFkZXJcXCcgYWxzbyByZXF1aXJlcyBcXCdwcmlvcml0eVxcJywgYW5kIHZpY2UgdmVyc2EnKTtcbiAgICB9XG59XG4iXX0=