"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UnauthenticatedAction = exports.ListenerAction = void 0;
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * What to do when a client makes a request to a listener
 *
 * Some actions can be combined with other ones (specifically,
 * you can perform authentication before serving the request).
 *
 * Multiple actions form a linked chain; the chain must always terminate in a
 * *(weighted)forward*, *fixedResponse* or *redirect* action.
 *
 * If an action supports chaining, the next action can be indicated
 * by passing it in the `next` property.
 *
 * (Called `ListenerAction` instead of the more strictly correct
 * `ListenerAction` because this is the class most users interact
 * with, and we want to make it not too visually overwhelming).
 */
class ListenerAction {
    /**
     * Create an instance of ListenerAction
     *
     * The default class should be good enough for most cases and
     * should be created by using one of the static factory functions,
     * but allow overriding to make sure we allow flexibility for the future.
     */
    constructor(actionJson, next) {
        this.actionJson = actionJson;
        this.next = next;
    }
    /**
     * Authenticate using an identity provider (IdP) that is compliant with OpenID Connect (OIDC)
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/listener-authenticate-users.html#oidc-requirements
     */
    static authenticateOidc(options) {
        var _a;
        return new ListenerAction({
            type: 'authenticate-oidc',
            authenticateOidcConfig: {
                authorizationEndpoint: options.authorizationEndpoint,
                clientId: options.clientId,
                clientSecret: options.clientSecret.toString(),
                issuer: options.issuer,
                tokenEndpoint: options.tokenEndpoint,
                userInfoEndpoint: options.userInfoEndpoint,
                authenticationRequestExtraParams: options.authenticationRequestExtraParams,
                onUnauthenticatedRequest: options.onUnauthenticatedRequest,
                scope: options.scope,
                sessionCookieName: options.sessionCookieName,
                sessionTimeout: (_a = options.sessionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            },
        }, options.next);
    }
    /**
     * Forward to one or more Target Groups
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#forward-actions
     */
    static forward(targetGroups, options = {}) {
        if (targetGroups.length === 0) {
            throw new Error('Need at least one targetGroup in a ListenerAction.forward()');
        }
        if (targetGroups.length === 1 && options.stickinessDuration === undefined) {
            // Render a "simple" action for backwards compatibility with old templates
            return new TargetGroupListenerAction(targetGroups, {
                type: 'forward',
                targetGroupArn: targetGroups[0].targetGroupArn,
            });
        }
        return new TargetGroupListenerAction(targetGroups, {
            type: 'forward',
            forwardConfig: {
                targetGroups: targetGroups.map(g => ({ targetGroupArn: g.targetGroupArn })),
                targetGroupStickinessConfig: options.stickinessDuration ? {
                    durationSeconds: options.stickinessDuration.toSeconds(),
                    enabled: true,
                } : undefined,
            },
        });
    }
    /**
     * Forward to one or more Target Groups which are weighted differently
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#forward-actions
     */
    static weightedForward(targetGroups, options = {}) {
        if (targetGroups.length === 0) {
            throw new Error('Need at least one targetGroup in a ListenerAction.weightedForward()');
        }
        return new TargetGroupListenerAction(targetGroups.map(g => g.targetGroup), {
            type: 'forward',
            forwardConfig: {
                targetGroups: targetGroups.map(g => ({ targetGroupArn: g.targetGroup.targetGroupArn, weight: g.weight })),
                targetGroupStickinessConfig: options.stickinessDuration ? {
                    durationSeconds: options.stickinessDuration.toSeconds(),
                    enabled: true,
                } : undefined,
            },
        });
    }
    /**
     * Return a fixed response
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#fixed-response-actions
     */
    static fixedResponse(statusCode, options = {}) {
        return new ListenerAction({
            type: 'fixed-response',
            fixedResponseConfig: {
                statusCode: core_1.Tokenization.stringifyNumber(statusCode),
                contentType: options.contentType,
                messageBody: options.messageBody,
            },
        });
    }
    /**
     * Redirect to a different URI
     *
     * A URI consists of the following components:
     * protocol://hostname:port/path?query. You must modify at least one of the
     * following components to avoid a redirect loop: protocol, hostname, port, or
     * path. Any components that you do not modify retain their original values.
     *
     * You can reuse URI components using the following reserved keywords:
     *
     * - `#{protocol}`
     * - `#{host}`
     * - `#{port}`
     * - `#{path}` (the leading "/" is removed)
     * - `#{query}`
     *
     * For example, you can change the path to "/new/#{path}", the hostname to
     * "example.#{host}", or the query to "#{query}&value=xyz".
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#redirect-actions
     */
    static redirect(options) {
        if ([options.host, options.path, options.port, options.protocol, options.query].findIndex(x => x !== undefined) === -1) {
            throw new Error('To prevent redirect loops, set at least one of \'protocol\', \'host\', \'port\', \'path\', or \'query\'.');
        }
        return new ListenerAction({
            type: 'redirect',
            redirectConfig: {
                statusCode: options.permanent ? 'HTTP_301' : 'HTTP_302',
                host: options.host,
                path: options.path,
                port: options.port,
                protocol: options.protocol,
                query: options.query,
            },
        });
    }
    /**
     * Render the actions in this chain
     */
    renderActions() {
        var _a, _b;
        return this.renumber([this.actionJson, ...(_b = (_a = this.next) === null || _a === void 0 ? void 0 : _a.renderActions()) !== null && _b !== void 0 ? _b : []]);
    }
    /**
     * Called when the action is being used in a listener
     */
    bind(scope, listener, associatingConstruct) {
        // Empty on purpose
        Array.isArray(scope);
        Array.isArray(listener);
        Array.isArray(associatingConstruct);
    }
    /**
     * Renumber the "order" fields in the actions array.
     *
     * We don't number for 0 or 1 elements, but otherwise number them 1...#actions
     * so ELB knows about the right order.
     *
     * Do this in `ListenerAction` instead of in `Listener` so that we give
     * users the opportunity to override by subclassing and overriding `renderActions`.
     */
    renumber(actions) {
        if (actions.length < 2) {
            return actions;
        }
        return actions.map((action, i) => ({ ...action, order: i + 1 }));
    }
}
exports.ListenerAction = ListenerAction;
/**
 * What to do with unauthenticated requests
 */
var UnauthenticatedAction;
(function (UnauthenticatedAction) {
    /**
     * Return an HTTP 401 Unauthorized error.
     */
    UnauthenticatedAction["DENY"] = "deny";
    /**
     * Allow the request to be forwarded to the target.
     */
    UnauthenticatedAction["ALLOW"] = "allow";
    /**
     * Redirect the request to the IdP authorization endpoint.
     */
    UnauthenticatedAction["AUTHENTICATE"] = "authenticate";
})(UnauthenticatedAction = exports.UnauthenticatedAction || (exports.UnauthenticatedAction = {}));
/**
 * Listener Action that calls "registerListener" on TargetGroups
 */
class TargetGroupListenerAction extends ListenerAction {
    constructor(targetGroups, actionJson) {
        super(actionJson);
        this.targetGroups = targetGroups;
    }
    bind(_scope, listener, associatingConstruct) {
        for (const tg of this.targetGroups) {
            tg.registerListener(listener, associatingConstruct);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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