"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ListenerPort = exports.LoadBalancer = exports.LoadBalancingProtocol = void 0;
const aws_ec2_1 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const elasticloadbalancing_generated_1 = require("./elasticloadbalancing.generated");
var LoadBalancingProtocol;
(function (LoadBalancingProtocol) {
    LoadBalancingProtocol["TCP"] = "tcp";
    LoadBalancingProtocol["SSL"] = "ssl";
    LoadBalancingProtocol["HTTP"] = "http";
    LoadBalancingProtocol["HTTPS"] = "https";
})(LoadBalancingProtocol = exports.LoadBalancingProtocol || (exports.LoadBalancingProtocol = {}));
/**
 * A load balancer with a single listener
 *
 * Routes to a fleet of of instances in a VPC.
 */
class LoadBalancer extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * An object controlling specifically the connections for each listener added to this load balancer
         */
        this.listenerPorts = [];
        this.listeners = [];
        this.instancePorts = [];
        this.targets = [];
        this.securityGroup = new aws_ec2_1.SecurityGroup(this, 'SecurityGroup', { vpc: props.vpc, allowAllOutbound: false });
        this.connections = new aws_ec2_1.Connections({ securityGroups: [this.securityGroup] });
        // Depending on whether the ELB has public or internal IPs, pick the right backend subnets
        const selectedSubnets = loadBalancerSubnets(props);
        this.elb = new elasticloadbalancing_generated_1.CfnLoadBalancer(this, 'Resource', {
            securityGroups: [this.securityGroup.securityGroupId],
            subnets: selectedSubnets.subnetIds,
            listeners: core_1.Lazy.anyValue({ produce: () => this.listeners }),
            scheme: props.internetFacing ? 'internet-facing' : 'internal',
            healthCheck: props.healthCheck && healthCheckToJSON(props.healthCheck),
            crossZone: (props.crossZone === undefined || props.crossZone) ? true : false,
        });
        if (props.internetFacing) {
            this.elb.node.addDependency(selectedSubnets.internetConnectivityEstablished);
        }
        ifUndefined(props.listeners, []).forEach(b => this.addListener(b));
        ifUndefined(props.targets, []).forEach(t => this.addTarget(t));
    }
    /**
     * Add a backend to the load balancer
     *
     * @returns A ListenerPort object that controls connections to the listener port
     */
    addListener(listener) {
        const protocol = ifUndefinedLazy(listener.externalProtocol, () => wellKnownProtocol(listener.externalPort));
        const instancePort = listener.internalPort || listener.externalPort;
        const instanceProtocol = ifUndefined(listener.internalProtocol, ifUndefined(tryWellKnownProtocol(instancePort), isHttpProtocol(protocol) ? LoadBalancingProtocol.HTTP : LoadBalancingProtocol.TCP));
        this.listeners.push({
            loadBalancerPort: listener.externalPort.toString(),
            protocol,
            instancePort: instancePort.toString(),
            instanceProtocol,
            sslCertificateId: listener.sslCertificateId,
            policyNames: listener.policyNames,
        });
        const port = new ListenerPort(this.securityGroup, aws_ec2_1.Port.tcp(listener.externalPort));
        // Allow connections on the public port for all supplied peers (default: everyone)
        ifUndefined(listener.allowConnectionsFrom, [aws_ec2_1.Peer.anyIpv4()]).forEach(peer => {
            port.connections.allowDefaultPortFrom(peer, `Default rule allow on ${listener.externalPort}`);
        });
        this.newInstancePort(instancePort);
        // Keep track using array so user can get to them even if they were all supplied in the constructor
        this.listenerPorts.push(port);
        return port;
    }
    addTarget(target) {
        target.attachToClassicLB(this);
        this.newTarget(target);
    }
    /**
     * @attribute
     */
    get loadBalancerName() {
        return this.elb.ref;
    }
    /**
     * @attribute
     */
    get loadBalancerCanonicalHostedZoneNameId() {
        return this.elb.attrCanonicalHostedZoneNameId;
    }
    /**
     * @attribute
     */
    get loadBalancerCanonicalHostedZoneName() {
        return this.elb.attrCanonicalHostedZoneName;
    }
    /**
     * @attribute
     */
    get loadBalancerDnsName() {
        return this.elb.attrDnsName;
    }
    /**
     * @attribute
     */
    get loadBalancerSourceSecurityGroupGroupName() {
        return this.elb.attrSourceSecurityGroupGroupName;
    }
    /**
     * @attribute
     */
    get loadBalancerSourceSecurityGroupOwnerAlias() {
        return this.elb.attrSourceSecurityGroupOwnerAlias;
    }
    /**
     * Allow connections to all existing targets on new instance port
     */
    newInstancePort(instancePort) {
        this.targets.forEach(t => this.allowTargetConnection(instancePort, t));
        // Keep track of port for future targets
        this.instancePorts.push(instancePort);
    }
    /**
     * Allow connections to target on all existing instance ports
     */
    newTarget(target) {
        this.instancePorts.forEach(p => this.allowTargetConnection(p, target));
        // Keep track of target for future listeners.
        this.targets.push(target);
    }
    /**
     * Allow connections for a single (port, target) pair
     */
    allowTargetConnection(instancePort, target) {
        this.connections.allowTo(target, aws_ec2_1.Port.tcp(instancePort), `Port ${instancePort} LB to fleet`);
    }
}
exports.LoadBalancer = LoadBalancer;
/**
 * Reference to a listener's port just created.
 *
 * This implements IConnectable with a default port (the port that an ELB
 * listener was just created on) for a given security group so that it can be
 * conveniently used just like any Connectable. E.g:
 *
 *    const listener = elb.addListener(...);
 *
 *    listener.connections.allowDefaultPortFromAnyIPv4();
 *    // or
 *    instance.connections.allowToDefaultPort(listener);
 */
class ListenerPort {
    constructor(securityGroup, defaultPort) {
        this.connections = new aws_ec2_1.Connections({ securityGroups: [securityGroup], defaultPort });
    }
}
exports.ListenerPort = ListenerPort;
function wellKnownProtocol(port) {
    const proto = tryWellKnownProtocol(port);
    if (!proto) {
        throw new Error(`Please supply protocol to go with port ${port}`);
    }
    return proto;
}
function tryWellKnownProtocol(port) {
    if (port === 80) {
        return LoadBalancingProtocol.HTTP;
    }
    if (port === 443) {
        return LoadBalancingProtocol.HTTPS;
    }
    return undefined;
}
function isHttpProtocol(proto) {
    return proto === LoadBalancingProtocol.HTTPS || proto === LoadBalancingProtocol.HTTP;
}
function ifUndefined(x, def) {
    return x != null ? x : def;
}
function ifUndefinedLazy(x, def) {
    return x != null ? x : def();
}
/**
 * Turn health check parameters into a parameter blob for the LB
 */
function healthCheckToJSON(healthCheck) {
    const protocol = ifUndefined(healthCheck.protocol, ifUndefined(tryWellKnownProtocol(healthCheck.port), LoadBalancingProtocol.TCP));
    const path = protocol === LoadBalancingProtocol.HTTP || protocol === LoadBalancingProtocol.HTTPS ? ifUndefined(healthCheck.path, '/') : '';
    const target = `${protocol.toUpperCase()}:${healthCheck.port}${path}`;
    return {
        healthyThreshold: ifUndefined(healthCheck.healthyThreshold, 2).toString(),
        interval: (healthCheck.interval || core_1.Duration.seconds(30)).toSeconds().toString(),
        target,
        timeout: (healthCheck.timeout || core_1.Duration.seconds(5)).toSeconds().toString(),
        unhealthyThreshold: ifUndefined(healthCheck.unhealthyThreshold, 5).toString(),
    };
}
function loadBalancerSubnets(props) {
    if (props.subnetSelection !== undefined) {
        return props.vpc.selectSubnets(props.subnetSelection);
    }
    else if (props.internetFacing) {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        });
    }
    else {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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